<?php
header('Content-Type: application/json; charset=utf-8');

// POST 요청만 허용
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => '잘못된 요청 방법입니다.']);
    exit;
}

// 세션 시작 및 로그인 확인
session_start();
if (!isset($_SESSION["id"]) || empty($_SESSION["id"])) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => '로그인이 필요합니다.']);
    exit;
}

// 데이터베이스 연결
include "inc/db_connect.php";

// MySQLi 연결 확인
if (!isset($con) || !$con) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => '데이터베이스 연결 오류가 발생했습니다.']);
    exit;
}

try {
    // 현재 로그인한 관리자 정보 조회
    $adminUserId = $_SESSION["id"];
    $adminCheckQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $adminCheckStmt = mysqli_prepare($con, $adminCheckQuery);
    mysqli_stmt_bind_param($adminCheckStmt, "s", $adminUserId);
    mysqli_stmt_execute($adminCheckStmt);
    $adminResult = mysqli_stmt_get_result($adminCheckStmt);
    $adminData = mysqli_fetch_assoc($adminResult);
    mysqli_stmt_close($adminCheckStmt);

    if (!$adminData) {
        throw new Exception('관리자 정보를 찾을 수 없습니다.');
    }

    // 관리자 권한 확인 (admin 또는 특정 권한만 가맹점 생성 가능)
    if ($adminData['grade'] !== 'admin') {
        writeLog($con, 'MEMBER_CREATE_DENIED', null, null, 0, 
                "가맹점 생성 권한 없음 - 시도한 관리자: {$adminData['user_name']}({$adminUserId})", 
                $adminData['id']);
        throw new Exception('가맹점 생성 권한이 없습니다.');
    }

    $adminMemberId = $adminData['id'];
    $adminName = $adminData['user_name'];

    // 입력 데이터 검증 및 정제
    $user_id = trim($_POST['user_id'] ?? '');
    $user_name = trim($_POST['user_name'] ?? '');
    $password = trim($_POST['password'] ?? '');
    $code = trim($_POST['code'] ?? '');
    $kakao_account_id = intval($_POST['kakao_account_id'] ?? 0);
    $kakao_link_type = trim($_POST['kakao_link_type'] ?? '');
    $bank_account_id = intval($_POST['bank_account_id'] ?? 0);
    $bank = trim($_POST['bank'] ?? '');
    $account_number = trim($_POST['account_number'] ?? '');
    $account_holder = trim($_POST['account_holder'] ?? '');
    $rate = floatval($_POST['rate'] ?? 0);
    $grade = trim($_POST['grade'] ?? '');

    // 입력 데이터 로그 기록
    $inputData = [
        'user_id' => $user_id,
        'user_name' => $user_name,
        'code' => $code,
        'kakao_account_id' => $kakao_account_id,
        'kakao_link_type' => $kakao_link_type,
        'bank_account_id' => $bank_account_id,
        'bank' => $bank,
        'account_number' => $account_number,
        'account_holder' => $account_holder,
        'rate' => $rate,
        'grade' => $grade,
        'requested_by' => $adminName
    ];

    // 필수 필드 검증
    if (empty($user_id)) {
        writeLog($con, 'MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "가맹점 생성 실패 - 가맹점 아이디 누락", $adminMemberId);
        throw new Exception('가맹점 아이디를 입력해주세요.');
    }
    
    // ... 기존 검증 코드들 ...

    // 중복 확인 시작
    $user_id_escaped = mysqli_real_escape_string($con, $user_id);
    $code_escaped = mysqli_real_escape_string($con, $code);
    
    // 가맹점 아이디 중복 확인
    $duplicateUserCheckSql = "SELECT user_id FROM member WHERE user_id = '$user_id_escaped'";
    $duplicateUserResult = mysqli_query($con, $duplicateUserCheckSql);
    if ($duplicateUserResult && mysqli_num_rows($duplicateUserResult) > 0) {
        writeLog($con, 'MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "가맹점 생성 실패 - 중복된 가맹점 아이디: {$user_id}", $adminMemberId);
        throw new Exception('이미 사용 중인 가맹점 아이디입니다.');
    }

    // 가맹점 코드 중복 확인
    $duplicateCodeCheckSql = "SELECT code FROM member WHERE code = '$code_escaped'";
    $duplicateCodeResult = mysqli_query($con, $duplicateCodeCheckSql);
    if ($duplicateCodeResult && mysqli_num_rows($duplicateCodeResult) > 0) {
        writeLog($con, 'MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "가맹점 생성 실패 - 중복된 가맹점 코드: {$code}", $adminMemberId);
        throw new Exception('이미 사용 중인 가맹점 코드입니다.');
    }

    // 트랜잭션 시작
    mysqli_autocommit($con, false);
    mysqli_begin_transaction($con);

    // 데이터 이스케이프
    $user_name_escaped = mysqli_real_escape_string($con, $user_name);
    $bank_escaped = mysqli_real_escape_string($con, $bank);
    $account_number_escaped = mysqli_real_escape_string($con, $account_number);
    $account_holder_escaped = mysqli_real_escape_string($con, $account_holder);
    $kakao_link_type_escaped = mysqli_real_escape_string($con, $kakao_link_type);
    $grade_escaped = mysqli_real_escape_string($con, $grade);

    $level = 1; // 기본값

    // 가맹점 추가
    $insertSql = "INSERT INTO member (
                    user_id, 
                    user_name, 
                    password, 
                    code, 
                    kakao_account_id, 
                    kakao_link_type,
                    bank_account_id, 
                    bank, 
                    account_number, 
                    account_holder, 
                    rate,
                    grade,
                    level,
                    parents,
                    amount,
                    created_at,
                    updated_at
                  ) VALUES (
                    '$user_id_escaped',
                    '$user_name_escaped',
                    '$password',
                    '$code_escaped',
                    " . ($kakao_account_id > 0 ? $kakao_account_id : "0") . ",
                    " . (!empty($kakao_link_type) ? "'$kakao_link_type_escaped'" : "NULL") . ",
                    " . ($bank_account_id > 0 ? $bank_account_id : "0") . ",
                    '$bank_escaped',
                    '$account_number_escaped',
                    '$account_holder_escaped',
                    $rate,
                    '$grade_escaped',
                    $level,
                    '[]',
                    0,
                    NOW(),
                    NOW()
                  )";

    $result = mysqli_query($con, $insertSql);

    if ($result) {
        $memberId = mysqli_insert_id($con);
        
        // 생성된 가맹점 데이터
        $createdData = [
            'id' => $memberId,
            'user_id' => $user_id,
            'user_name' => $user_name,
            'code' => $code,
            'kakao_account_id' => $kakao_account_id,
            'kakao_link_type' => $kakao_link_type,
            'bank_account_id' => $bank_account_id,
            'bank' => $bank,
            'account_number' => $account_number,
            'account_holder' => $account_holder,
            'rate' => $rate,
            'grade' => $grade,
            'level' => $level,
            'created_by' => $adminName,
            'created_at' => date('Y-m-d H:i:s')
        ];

        // 성공 로그 기록
        $detail = "가맹점 생성 완료 - ID: {$user_id}, 가맹점명: {$user_name}, 코드: {$code}, " .
                 "등급: {$grade}, 수수료율: {$rate}%, 생성자: {$adminName}({$adminUserId})";
        
        writeLog($con, 'MEMBER_CREATE_SUCCESS', $inputData, $createdData, 1, $detail, $adminMemberId);

        // 트랜잭션 커밋
        mysqli_commit($con);
        
        echo json_encode([
            'success' => true,
            'message' => '가맹점이 성공적으로 추가되었습니다.',
            'data' => $createdData
        ]);
        
    } else {
        // 실패 로그 기록
        $errorMsg = '가맹점 추가 중 오류가 발생했습니다: ' . mysqli_error($con);
        writeLog($con, 'MEMBER_CREATE_FAILED', $inputData, null, 0, $errorMsg, $adminMemberId);
        
        mysqli_rollback($con);
        throw new Exception($errorMsg);
    }

} catch (Exception $e) {
    // 트랜잭션 롤백
    if (isset($con)) {
        mysqli_rollback($con);
    }
    
    // 에러 로그 기록
    if (isset($adminMemberId) && isset($inputData)) {
        writeLog($con, 'MEMBER_CREATE_ERROR', $inputData ?? null, null, 0, 
                $e->getMessage(), $adminMemberId ?? 0);
    }
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
    
} finally {
    // autocommit 복원
    if (isset($con)) {
        mysqli_autocommit($con, true);
        mysqli_close($con);
    }
}
?>