<?php
// 에러 표시 (디버깅용 - 운영환경에서는 주석처리)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크 - 로그인된 사용자만
if (!isset($_SESSION['id'])) {
    writeLog($con, 'BANK_ACCOUNT_ADD_DENIED', null, null, 0, 
            '은행 계좌 추가 시도 실패 - 미인증 사용자');
    
    http_response_code(403);
    echo json_encode([
        'success' => false, 
        'message' => '접근 권한이 없습니다.',
        'error_code' => 'AUTH_FAILED'
    ]);
    exit;
}

try {
    // 현재 로그인한 사용자 정보 조회
    $currentUserId = $_SESSION['id'];
    $currentUserQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $currentUserQuery);
    mysqli_stmt_bind_param($stmt, "s", $currentUserId);
    mysqli_stmt_execute($stmt);
    $currentUserResult = mysqli_stmt_get_result($stmt);
    $currentUserData = mysqli_fetch_assoc($currentUserResult);
    mysqli_stmt_close($stmt);

    if (!$currentUserData) {
        throw new Exception('사용자 정보를 찾을 수 없습니다.');
    }

    $currentMemberId = $currentUserData['id'];
    $currentUserName = $currentUserData['user_name'];

    // POST 데이터 받기
    $bank = isset($_POST['bank']) ? trim($_POST['bank']) : '';
    $account_number = isset($_POST['account_number']) ? trim($_POST['account_number']) : '';
    $account_holder = isset($_POST['account_holder']) ? trim($_POST['account_holder']) : '';

    // 입력 데이터 로그용
    $inputData = [
        'bank' => $bank,
        'account_number' => $account_number,
        'account_holder' => $account_holder,
        'requested_by' => $currentUserName
    ];

    // 유효성 검사
    $errors = [];

    if (empty($bank)) {
        $errors[] = '은행을 선택해주세요.';
    }
    if (empty($account_number)) {
        $errors[] = '계좌번호를 입력해주세요.';
    }
    if (empty($account_holder)) {
        $errors[] = '예금주를 입력해주세요.';
    }

    // 계좌번호 숫자만 입력 검증
    if (!empty($account_number) && !preg_match('/^\d+$/', $account_number)) {
        $errors[] = '계좌번호는 숫자만 입력 가능합니다.';
    }

    // 입력값 길이 체크
    if (strlen($bank) > 100) {
        $errors[] = '은행명은 100자 이내로 입력해주세요.';
    }
    if (strlen($account_number) > 100) {
        $errors[] = '계좌번호는 100자 이내로 입력해주세요.';
    }
    if (strlen($account_holder) > 100) {
        $errors[] = '예금주명은 100자 이내로 입력해주세요.';
    }

    // 에러가 있으면 반환
    if (!empty($errors)) {
        $errorDetail = implode(' ', $errors);
        writeLog($con, 'BANK_ACCOUNT_ADD_FAILED', $inputData, null, 0, 
                "은행 계좌 추가 실패 - 유효성 검사: {$errorDetail}", $currentMemberId);
        
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => $errorDetail,
            'errors' => $errors
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // 트랜잭션 시작
    beginTransaction($con);
    
    // 1. 중복 체크 - 같은 은행의 같은 계좌번호는 중복 불가
    $checkQuery = "SELECT id FROM bank_account WHERE bank = ? AND account_number = ?";
    $checkStmt = mysqli_prepare($con, $checkQuery);
    
    if (!$checkStmt) {
        throw new Exception("중복 체크 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($checkStmt, "ss", $bank, $account_number);
    mysqli_stmt_execute($checkStmt);
    mysqli_stmt_store_result($checkStmt);
    
    if (mysqli_stmt_num_rows($checkStmt) > 0) {
        mysqli_stmt_close($checkStmt);
        writeLog($con, 'BANK_ACCOUNT_ADD_FAILED', $inputData, null, 0, 
                "은행 계좌 추가 실패 - 중복된 계좌: {$bank} {$account_number}", $currentMemberId);
        throw new Exception("이미 등록된 계좌번호입니다.");
    }
    
    mysqli_stmt_close($checkStmt);
    
    // 2. 은행 계좌 추가
    $insertQuery = "
        INSERT INTO bank_account (
            bank, 
            account_number, 
            account_holder, 
            created_at,
            updated_at
        ) VALUES (?, ?, ?, NOW(), NOW())
    ";
    
    $insertStmt = mysqli_prepare($con, $insertQuery);
    
    if (!$insertStmt) {
        throw new Exception("INSERT 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($insertStmt, "sss", 
        $bank, 
        $account_number, 
        $account_holder
    );
    
    if (!mysqli_stmt_execute($insertStmt)) {
        throw new Exception("데이터 추가 실패: " . mysqli_stmt_error($insertStmt));
    }
    
    // 추가된 ID 가져오기
    $insertedId = mysqli_insert_id($con);
    
    mysqli_stmt_close($insertStmt);
    
    // 트랜잭션 커밋
    commitTransaction($con);
    
    // 로그용 데이터 준비
    $newData = [
        'id' => $insertedId,
        'bank' => $bank,
        'account_number' => $account_number,
        'account_holder' => $account_holder,
        'created_by' => $currentUserName,
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    // 성공 로그 기록 - currentMemberId 전달
    $detail = "은행 계좌 추가 성공 - 은행: {$bank}, 계좌: {$account_number}, 예금주: {$account_holder}, 추가자: {$currentUserName}({$currentUserId})";
    writeLog($con, 'BANK_ACCOUNT_ADD_SUCCESS', $inputData, $newData, 1, $detail, $currentMemberId);
    
    // 성공 응답
    echo json_encode([
        'success' => true,
        'message' => '은행 계좌가 성공적으로 추가되었습니다.',
        'id' => $insertedId,
        'data' => $newData
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    rollbackTransaction($con);
    
    $errorMessage = $e->getMessage();
    
    // 실패 로그 기록 - currentMemberId 전달
    if (isset($currentMemberId) && isset($inputData)) {
        writeLog($con, 'BANK_ACCOUNT_ADD_FAILED', $inputData, null, 0, 
                "은행 계좌 추가 실패 - " . $errorMessage, $currentMemberId);
    }
    
    // 에러 응답
    http_response_code(500);
    
    // 중복 에러인 경우 409 Conflict
    if (strpos($errorMessage, '이미 등록된') !== false) {
        http_response_code(409);
    }
    
    echo json_encode([
        'success' => false,
        'message' => $errorMessage,
        'error_code' => 'ADD_FAILED'
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>