<?php
include '../inc/db_connect.php';
session_start();

// JSON 응답 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

try {
    // 관리자 권한 확인
    if (!isset($_SESSION['grade']) || $_SESSION['grade'] < 9) {
        writeLog($con, 'IP_BLACKLIST_ADD_DENIED', null, null, 0, 
                'IP 블랙리스트 추가 권한 없음 - grade: ' . ($_SESSION['grade'] ?? 'null'));
        echo json_encode(['success' => false, 'message' => '권한이 없습니다.']);
        exit;
    }

    // 현재 로그인한 관리자 정보 조회
    $currentUserId = $_SESSION['id'];
    $currentUserQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $currentUserQuery);
    mysqli_stmt_bind_param($stmt, "s", $currentUserId);
    mysqli_stmt_execute($stmt);
    $currentUserResult = mysqli_stmt_get_result($stmt);
    $currentUserData = mysqli_fetch_assoc($currentUserResult);
    mysqli_stmt_close($stmt);

    if (!$currentUserData) {
        throw new Exception('관리자 정보를 찾을 수 없습니다.');
    }

    $currentMemberId = $currentUserData['id'];
    $currentUserName = $currentUserData['user_name'];

    // POST 데이터 받기 및 검증
    $ip_address = trim($_POST['ip_address'] ?? '');
    $reason = trim($_POST['reason'] ?? '');
    $block_hours = (int)($_POST['block_hours'] ?? 24);

    // 입력 데이터 로그용
    $inputData = [
        'ip_address' => $ip_address,
        'reason' => $reason,
        'block_hours' => $block_hours,
        'admin_id' => $currentUserId,
        'admin_name' => $currentUserName
    ];

    // 유효성 검사
    if (empty($ip_address)) {
        writeLog($con, 'IP_BLACKLIST_ADD_FAILED', $inputData, null, 0, 
                'IP 블랙리스트 추가 실패 - IP 주소 누락', $currentMemberId);
        echo json_encode(['success' => false, 'message' => 'IP 주소를 입력해주세요.']);
        exit;
    }

    // IP 형식 검증
    if (!filter_var($ip_address, FILTER_VALIDATE_IP)) {
        writeLog($con, 'IP_BLACKLIST_ADD_FAILED', $inputData, null, 0, 
                "IP 블랙리스트 추가 실패 - 잘못된 IP 형식: {$ip_address}", $currentMemberId);
        echo json_encode(['success' => false, 'message' => '올바른 IP 주소가 아닙니다.']);
        exit;
    }

    if (empty($reason)) {
        writeLog($con, 'IP_BLACKLIST_ADD_FAILED', $inputData, null, 0, 
                'IP 블랙리스트 추가 실패 - 차단 사유 누락', $currentMemberId);
        echo json_encode(['success' => false, 'message' => '차단 사유를 입력해주세요.']);
        exit;
    }

    if ($block_hours < 0 || $block_hours > 8760) { // 최대 1년
        writeLog($con, 'IP_BLACKLIST_ADD_FAILED', $inputData, null, 0, 
                "IP 블랙리스트 추가 실패 - 잘못된 차단 시간: {$block_hours}", $currentMemberId);
        echo json_encode(['success' => false, 'message' => '차단 시간은 0~8760시간 사이여야 합니다.']);
        exit;
    }

    // 트랜잭션 시작
    beginTransaction($con);

    // 기존 데이터 조회 (업데이트 대상)
    $existingQuery = "SELECT * FROM ip_blacklist WHERE ip_address = ?";
    $existingStmt = mysqli_prepare($con, $existingQuery);
    mysqli_stmt_bind_param($existingStmt, "s", $ip_address);
    mysqli_stmt_execute($existingStmt);
    $existingResult = mysqli_stmt_get_result($existingStmt);
    $existingData = mysqli_fetch_assoc($existingResult);
    mysqli_stmt_close($existingStmt);

    // 데이터 이스케이프
    $ip_escaped = mysqli_real_escape_string($con, $ip_address);
    $reason_escaped = mysqli_real_escape_string($con, $reason);
    $admin_id_escaped = mysqli_real_escape_string($con, $currentUserId);

    $blocked_until = $block_hours > 0 ? "DATE_ADD(NOW(), INTERVAL $block_hours HOUR)" : "NULL";

    // 중복 확인 및 추가/업데이트
    $sql = "INSERT INTO ip_blacklist (ip_address, reason, blocked_until, admin_id, status, created_at, updated_at) 
            VALUES ('$ip_escaped', '$reason_escaped', $blocked_until, '$admin_id_escaped', 'active', NOW(), NOW())
            ON DUPLICATE KEY UPDATE 
                status = 'active',
                reason = '$reason_escaped',
                blocked_until = $blocked_until,
                admin_id = '$admin_id_escaped',
                updated_at = NOW()";

    if (!mysqli_query($con, $sql)) {
        writeLog($con, 'IP_BLACKLIST_ADD_FAILED', $inputData, null, 0, 
                'IP 블랙리스트 추가 실패 - DB 오류: ' . mysqli_error($con), $currentMemberId);
        throw new Exception('데이터베이스 오류: ' . mysqli_error($con));
    }

    // 새로운 데이터 구성
    $newData = [
        'ip_address' => $ip_address,
        'reason' => $reason,
        'block_hours' => $block_hours,
        'blocked_until' => $block_hours > 0 ? date('Y-m-d H:i:s', strtotime("+{$block_hours} hours")) : null,
        'admin_id' => $currentUserId,
        'admin_name' => $currentUserName,
        'status' => 'active',
        'action' => $existingData ? 'updated' : 'created',
        'updated_at' => date('Y-m-d H:i:s')
    ];

    // 트랜잭션 커밋
    commitTransaction($con);

    // 성공 로그 기록
    $action = $existingData ? '업데이트' : '추가';
    $detail = "IP 블랙리스트 {$action} - IP: {$ip_address}, 사유: {$reason}, " .
             "차단시간: {$block_hours}시간, 처리자: {$currentUserName}({$currentUserId})";
    
    writeLog($con, 'IP_BLACKLIST_ADD_SUCCESS', $existingData, $newData, 1, $detail, $currentMemberId);

    echo json_encode([
        'success' => true,
        'message' => "IP 주소가 성공적으로 블랙리스트에 {$action}되었습니다.",
        'data' => $newData
    ]);

} catch (Exception $e) {
    // 트랜잭션 롤백
    if (isset($con)) {
        rollbackTransaction($con);
    }

    // 에러 로그 기록
    if (isset($currentMemberId) && isset($inputData)) {
        writeLog($con, 'IP_BLACKLIST_ADD_ERROR', $inputData, null, 0, 
                'IP 블랙리스트 추가 오류: ' . $e->getMessage(), $currentMemberId);
    }

    echo json_encode([
        'success' => false, 
        'message' => $e->getMessage()
    ]);

} finally {
    // 데이터베이스 연결 종료
    if (isset($con)) {
        mysqli_close($con);
    }
}
?>