<?php
// 에러 표시 (디버깅용 - 운영환경에서는 주석처리)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크 - 로그인된 사용자만
if (!isset($_SESSION['id'])) {
    writeLog($con, 'KAKAO_ACCOUNT_ADD_DENIED', null, null, 0, 
            '카카오 계정 추가 시도 실패 - 미인증 사용자');
    
    http_response_code(403);
    echo json_encode([
        'success' => false, 
        'message' => '접근 권한이 없습니다.',
        'error_code' => 'AUTH_FAILED'
    ]);
    exit;
}

try {
    // 현재 로그인한 사용자 정보 조회
    $currentUserId = $_SESSION['id'];
    $currentUserQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $currentUserQuery);
    mysqli_stmt_bind_param($stmt, "s", $currentUserId);
    mysqli_stmt_execute($stmt);
    $currentUserResult = mysqli_stmt_get_result($stmt);
    $currentUserData = mysqli_fetch_assoc($currentUserResult);
    mysqli_stmt_close($stmt);

    if (!$currentUserData) {
        throw new Exception('사용자 정보를 찾을 수 없습니다.');
    }

    $currentMemberId = $currentUserData['id'];
    $currentUserName = $currentUserData['user_name'];

    // POST 데이터 받기
    $kakao_tid = isset($_POST['kakao_tid']) ? trim($_POST['kakao_tid']) : '';
    $name = isset($_POST['name']) ? trim($_POST['name']) : '';
    $qr_link = isset($_POST['qr_link']) ? trim($_POST['qr_link']) : '';
    $friend_link = isset($_POST['friend_link']) ? trim($_POST['friend_link']) : '';
    $default_link = isset($_POST['default_link']) ? trim($_POST['default_link']) : '';

    // 입력 데이터 로그용
    $inputData = [
        'kakao_tid' => $kakao_tid,
        'name' => $name,
        'qr_link' => $qr_link,
        'friend_link' => $friend_link,
        'default_link' => $default_link,
        'requested_by' => $currentUserName
    ];

    // 유효성 검사
    $errors = [];

    if (empty($kakao_tid)) {
        $errors[] = '카카오 TID를 입력해주세요.';
    }
    if (empty($name)) {
        $errors[] = '계정명을 입력해주세요.';
    }

    // 입력값 길이 체크
    if (strlen($kakao_tid) > 100) {
        $errors[] = '카카오 TID는 100자 이내로 입력해주세요.';
    }
    if (strlen($name) > 45) {
        $errors[] = '계정명은 45자 이내로 입력해주세요.';
    }

    // 에러가 있으면 반환
    if (!empty($errors)) {
        $errorDetail = implode(' ', $errors);
        writeLog($con, 'KAKAO_ACCOUNT_ADD_FAILED', $inputData, null, 0, 
                "카카오 계정 추가 실패 - 유효성 검사: {$errorDetail}", $currentMemberId);
        
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => $errorDetail,
            'errors' => $errors
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    if (!empty($default_link) && !empty($qr_link)) {
        try {
            $qr_data = json_decode($qr_link, true);
            if (is_array($qr_data)) {
                $qr_data['default'] = $default_link;
                $qr_link = json_encode($qr_data);
            }
        } catch (Exception $e) {
            // 오류 시 무시하고 계속 진행
        }
    }

    // 트랜잭션 시작
    beginTransaction($con);
    
    // 1. 중복 체크 - kakao_tid가 유니크해야 함
    $checkQuery = "SELECT id FROM kakao_account WHERE kakao_tid = ?";
    $checkStmt = mysqli_prepare($con, $checkQuery);
    
    if (!$checkStmt) {
        throw new Exception("중복 체크 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($checkStmt, "s", $kakao_tid);
    mysqli_stmt_execute($checkStmt);
    mysqli_stmt_store_result($checkStmt);
    
    if (mysqli_stmt_num_rows($checkStmt) > 0) {
        mysqli_stmt_close($checkStmt);
        writeLog($con, 'KAKAO_ACCOUNT_ADD_FAILED', $inputData, null, 0, 
                "카카오 계정 추가 실패 - 중복된 TID: {$kakao_tid}", $currentMemberId);
        throw new Exception("이미 등록된 카카오 TID입니다.");
    }
    
    mysqli_stmt_close($checkStmt);
    
    // 올바른 INSERT 쿼리
    $insertQuery = "
        INSERT INTO kakao_account (
            kakao_tid, 
            name,
            qr_link,
            friend_link, 
            created_at
        ) VALUES (?, ?, ?, ?, NOW())
    ";

    $insertStmt = mysqli_prepare($con, $insertQuery);

    if (!$insertStmt) {
        throw new Exception("INSERT 쿼리 준비 실패: " . mysqli_error($con));
    }

    // 4개 파라미터로 수정
    mysqli_stmt_bind_param($insertStmt, "ssss", 
        $kakao_tid, 
        $name,
        $qr_link,
        $friend_link
    );
    
    if (!mysqli_stmt_execute($insertStmt)) {
        throw new Exception("데이터 추가 실패: " . mysqli_stmt_error($insertStmt));
    }
    
    // 추가된 ID 가져오기
    $insertedId = mysqli_insert_id($con);
    
    mysqli_stmt_close($insertStmt);
    
    // 트랜잭션 커밋
    commitTransaction($con);
    
    // 로그용 데이터 준비
    $newData = [
        'id' => $insertedId,
        'kakao_tid' => $kakao_tid,
        'name' => $name,
        'qr_link' => json_decode($qr_link, true),
        'friend_link' => json_decode($friend_link, true),
        'created_by' => $currentUserName,
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    // 성공 로그 기록 - currentMemberId 전달
    $detail = "카카오 계정 추가 성공 - 계정명: {$name}, TID: {$kakao_tid}, 추가자: {$currentUserName}({$currentUserId})";
    writeLog($con, 'KAKAO_ACCOUNT_ADD_SUCCESS', $inputData, $newData, 1, $detail, $currentMemberId);
    
    // 성공 응답
    echo json_encode([
        'success' => true,
        'message' => '카카오 계정이 성공적으로 추가되었습니다.',
        'id' => $insertedId,
        'data' => $newData
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    rollbackTransaction($con);
    
    $errorMessage = $e->getMessage();
    
    // 실패 로그 기록 - currentMemberId 전달
    if (isset($currentMemberId) && isset($inputData)) {
        writeLog($con, 'KAKAO_ACCOUNT_ADD_FAILED', $inputData, null, 0, 
                "카카오 계정 추가 실패 - " . $errorMessage, $currentMemberId);
    }
    
    // 에러 응답
    http_response_code(500);
    
    // 중복 에러인 경우 409 Conflict
    if (strpos($errorMessage, '이미 등록된') !== false) {
        http_response_code(409);
    }
    
    echo json_encode([
        'success' => false,
        'message' => $errorMessage,
        'error_code' => 'ADD_FAILED'
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>