<?php
session_start();
header('Content-Type: application/json');

// 로그인 체크
if (!isset($_SESSION['id'])) {
    echo json_encode(['success' => false, 'message' => '로그인이 필요합니다.']);
    exit;
}

// 데이터베이스 연결
include '../inc/session.php';

try {
    // 현재 로그인한 사용자 정보 조회
    $currentUserId = $_SESSION['id'];
    $currentUserQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $currentUserQuery);
    mysqli_stmt_bind_param($stmt, "s", $currentUserId);
    mysqli_stmt_execute($stmt);
    $currentUserResult = mysqli_stmt_get_result($stmt);
    $currentUserData = mysqli_fetch_assoc($currentUserResult);
    mysqli_stmt_close($stmt);

    if (!$currentUserData) {
        throw new Exception('사용자 정보를 찾을 수 없습니다.');
    }

    $currentMemberId = $currentUserData['id'];
    $currentUserName = $currentUserData['user_name'];

    // POST 데이터 받기
    $parent_user_id = $_POST['parent_user_id'] ?? '';
    $user_id = $_POST['user_id'] ?? '';
    $user_name = $_POST['user_name'] ?? '';
    $password = $_POST['password'] ?? '';
    $code = $_POST['code'] ?? '';
    $kakao_account_id = !empty($_POST['kakao_account_id']) ? $_POST['kakao_account_id'] : null;
    $kakao_link_type = trim($_POST['kakao_link_type'] ?? '');
    $bank_account_id = !empty($_POST['bank_account_id']) ? $_POST['bank_account_id'] : null;
    $bank = $_POST['bank'] ?? '';
    $account_number = $_POST['account_number'] ?? '';
    $account_holder = $_POST['account_holder'] ?? '';
    $rate = $_POST['rate'] ?? 0;
    $initial_balance = $_POST['initial_balance'] ?? 0;

    // 입력 데이터 로그용
    $inputData = [
        'parent_user_id' => $parent_user_id,
        'user_id' => $user_id,
        'user_name' => $user_name,
        'code' => $code,
        'kakao_account_id' => $kakao_account_id,
        'kakao_link_type' => $kakao_link_type,
        'bank_account_id' => $bank_account_id,
        'bank' => $bank,
        'account_number' => $account_number,
        'account_holder' => $account_holder,
        'rate' => $rate,
        'initial_balance' => $initial_balance,
        'requested_by' => $currentUserName
    ];

    // 유효성 검사
    if (empty($parent_user_id)) {
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "하부 가맹점 생성 실패 - 상위 가맹점 정보 누락", $currentMemberId);
        echo json_encode(['success' => false, 'message' => '상위 가맹점 정보가 없습니다.']);
        exit;
    }

    if (empty($user_id) || empty($user_name) || empty($password) || empty($code)) {
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "하부 가맹점 생성 실패 - 필수 정보 누락", $currentMemberId);
        echo json_encode(['success' => false, 'message' => '필수 정보를 모두 입력해주세요.']);
        exit;
    }

    // 카카오 계정과 은행 연결계정 검증
    if ($kakao_account_id && $bank_account_id) {
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "하부 가맹점 생성 실패 - 카카오/은행 계정 중복 선택", $currentMemberId);
        echo json_encode(['success' => false, 'message' => '카카오 연결계정과 은행 연결계정 중 하나만 선택할 수 있습니다.']);
        exit;
    }

    if (!$kakao_account_id && !$bank_account_id) {
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "하부 가맹점 생성 실패 - 연결계정 미선택", $currentMemberId);
        echo json_encode(['success' => false, 'message' => '카카오 연결계정 또는 은행 연결계정 중 하나를 선택해주세요.']);
        exit;
    }

    // 카카오 계정 선택 시 링크 타입 필수 확인
    if ($kakao_account_id && empty($kakao_link_type)) {
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "하부 가맹점 생성 실패 - 카카오 링크 타입 누락", $currentMemberId);
        echo json_encode(['success' => false, 'message' => '카카오 링크 타입을 선택해주세요.']);
        exit;
    }

    // 카카오 링크 타입 유효성 검사
    if (!empty($kakao_link_type) && !in_array($kakao_link_type, ['qr', 'friend'])) {
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "하부 가맹점 생성 실패 - 유효하지 않은 카카오 링크 타입: {$kakao_link_type}", $currentMemberId);
        echo json_encode(['success' => false, 'message' => '유효하지 않은 카카오 링크 타입입니다.']);
        exit;
    }

    // 은행 계정 선택 시 카카오 링크 타입 초기화
    if ($bank_account_id) {
        $kakao_link_type = '';
    }

    // 개별 출금 정보 필수 체크
    if (empty($bank) || empty($account_number) || empty($account_holder)) {
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "하부 가맹점 생성 실패 - 출금 정보 누락", $currentMemberId);
        echo json_encode(['success' => false, 'message' => '개별 출금 정보는 필수 입력사항입니다.']);
        exit;
    }

    // 트랜잭션 시작
    mysqli_begin_transaction($con);
    
    // 상위 가맹점 정보 조회 (ID와 parents 정보 가져오기)
    $parent_query = "SELECT id, parents, level, rate, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $parent_query);
    mysqli_stmt_bind_param($stmt, "s", $parent_user_id);
    mysqli_stmt_execute($stmt);
    $parent_result = mysqli_stmt_get_result($stmt);
    
    if (!$parent_result || mysqli_num_rows($parent_result) == 0) {
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "하부 가맹점 생성 실패 - 상위 가맹점을 찾을 수 없음: {$parent_user_id}", $currentMemberId);
        throw new Exception("상위 가맹점을 찾을 수 없습니다.");
    }
    
    $parent_data = mysqli_fetch_assoc($parent_result);
    $parent_id = $parent_data['id'];
    $parent_parents = $parent_data['parents'];
    $parent_level = $parent_data['level'] ?? 1;
    $parent_rate = $parent_data['rate'] ?? 0;
    $parent_grade = $parent_data['grade'] ?? 'basic';
    
    // 수수료율 유효성 검사
    if ((float)$rate < (float)$parent_rate) {
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "하부 가맹점 생성 실패 - 수수료율 오류: 하부({$rate}%) < 상위({$parent_rate}%)", $currentMemberId);
        throw new Exception("하부 가맹점의 수수료율({$rate}%)은 상위 가맹점의 수수료율({$parent_rate}%)보다 높을 수 없습니다.");
    }
    
    // parents 배열 생성
    $parents_array = [];
    if (!empty($parent_parents)) {
        $parents_array = json_decode($parent_parents, true);
        if (!is_array($parents_array)) {
            $parents_array = [];
        }
    }
    $parents_array[] = (int)$parent_id;
    $parents_json = json_encode($parents_array);
    
    // 하부 가맹점의 레벨 설정 (상위 가맹점 레벨 + 1)
    $sub_level = $parent_level + 1;
    
    // 하부 가맹점의 등급은 무조건 상위 가맹점의 등급을 따름
    $sub_grade = $parent_grade;
    
    // 아이디 중복 체크
    $check_query = "SELECT user_id FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $check_query);
    mysqli_stmt_bind_param($stmt, "s", $user_id);
    mysqli_stmt_execute($stmt);
    $check_result = mysqli_stmt_get_result($stmt);
    
    if (mysqli_num_rows($check_result) > 0) {
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "하부 가맹점 생성 실패 - 중복된 사용자 ID: {$user_id}", $currentMemberId);
        throw new Exception("이미 사용중인 아이디입니다.");
    }

    // 코드 중복 체크
    $check_query = "SELECT user_id FROM member WHERE code = ?";
    $stmt = mysqli_prepare($con, $check_query);
    mysqli_stmt_bind_param($stmt, "s", $code);
    mysqli_stmt_execute($stmt);
    $check_result = mysqli_stmt_get_result($stmt);
    
    if (mysqli_num_rows($check_result) > 0) {
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, 
                "하부 가맹점 생성 실패 - 중복된 가맹점 코드: {$code}", $currentMemberId);
        throw new Exception("이미 사용중인 가맹점 코드입니다.");
    }
    
    $plain_password = $password;
    
    // 하부 가맹점 추가
    $insert_query = "INSERT INTO member (
        user_id, 
        user_name, 
        password, 
        code, 
        kakao_account_id, 
        kakao_link_type,
        bank_account_id, 
        bank, 
        account_number, 
        account_holder, 
        rate, 
        amount, 
        parents, 
        level, 
        grade,
        created_at,
        updated_at
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())";
    
    $stmt = mysqli_prepare($con, $insert_query);
    
    // NULL 값 처리를 위한 타입 문자열 생성
    $types = "ssss"; // user_id, user_name, password, code
    $types .= ($kakao_account_id !== null) ? "i" : "i"; // kakao_account_id
    $types .= "s"; // kakao_link_type
    $types .= ($bank_account_id !== null) ? "i" : "i"; // bank_account_id
    $types .= "sss"; // bank, account_number, account_holder
    $types .= "dd"; // rate, amount
    $types .= "sis"; // parents(json), level, grade
    
    // kakao_link_type이 빈 문자열인 경우 NULL로 변환
    $kakao_link_type_for_db = !empty($kakao_link_type) ? $kakao_link_type : null;
    
    mysqli_stmt_bind_param($stmt, $types,
        $user_id,
        $user_name,
        $plain_password,
        $code,
        $kakao_account_id,
        $kakao_link_type_for_db,
        $bank_account_id,
        $bank,
        $account_number,
        $account_holder,
        $rate,
        $initial_balance,
        $parents_json,
        $sub_level,
        $sub_grade
    );
    
    if (!mysqli_stmt_execute($stmt)) {
        $errorMsg = "하부 가맹점 추가 중 오류가 발생했습니다: " . mysqli_error($con);
        writeLog($con, 'SUB_MEMBER_CREATE_FAILED', $inputData, null, 0, $errorMsg, $currentMemberId);
        throw new Exception($errorMsg);
    }
    
    $new_member_id = mysqli_insert_id($con);
    
    // 생성된 하부 가맹점 데이터
    $createdData = [
        'id' => $new_member_id,
        'user_id' => $user_id,
        'user_name' => $user_name,
        'code' => $code,
        'parent_user_id' => $parent_user_id,
        'parent_id' => $parent_id,
        'parents' => $parents_array,
        'level' => $sub_level,
        'grade' => $sub_grade,
        'rate' => $rate,
        'initial_balance' => $initial_balance,
        'kakao_account_id' => $kakao_account_id,
        'kakao_link_type' => $kakao_link_type,
        'bank_account_id' => $bank_account_id,
        'bank' => $bank,
        'account_number' => $account_number,
        'account_holder' => $account_holder,
        'created_by' => $currentUserName,
        'created_at' => date('Y-m-d H:i:s')
    ];

    // 성공 로그 기록
    $detail = "하부 가맹점 생성 완료 - ID: {$user_id}, 가맹점명: {$user_name}, 코드: {$code}, " .
             "상위: {$parent_user_id}, 등급: {$sub_grade}, 레벨: {$sub_level}, 수수료율: {$rate}%, " .
             "생성자: {$currentUserName}({$currentUserId})";
    
    writeLog($con, 'SUB_MEMBER_CREATE_SUCCESS', $inputData, $createdData, 1, $detail, $currentMemberId);
    
    // 트랜잭션 커밋
    mysqli_commit($con);
    
    // 로그 기록
    error_log("하부 가맹점 추가 성공: {$user_id} (상위: {$parent_user_id}, ID: {$parent_id}, parents: {$parents_json}, grade: {$sub_grade}, kakao_link_type: {$kakao_link_type})");
    
    echo json_encode([
        'success' => true, 
        'message' => '하부 가맹점이 성공적으로 추가되었습니다.',
        'member_id' => $new_member_id,
        'parent_id' => $parent_id,
        'parents' => $parents_array,
        'level' => $sub_level,
        'grade' => $sub_grade,
        'kakao_link_type' => $kakao_link_type
    ]);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    mysqli_rollback($con);
    
    // 에러 로그 기록
    if (isset($currentMemberId) && isset($inputData)) {
        writeLog($con, 'SUB_MEMBER_CREATE_ERROR', $inputData ?? null, null, 0, 
                $e->getMessage(), $currentMemberId ?? 0);
    }
    
    error_log("하부 가맹점 추가 실패: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($stmt)) {
        mysqli_stmt_close($stmt);
    }
    if (isset($con)) {
        mysqli_close($con);
    }
}
?>