<?php
// cancel_auto_approval.php - 자동승인 대기 취소 처리 (내역 삭제)

// 에러 보고 설정
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

session_start();

// JSON 헤더 설정
header('Content-Type: application/json; charset=utf-8');

// 출력 버퍼링 시작
ob_start();

try {
    // DB 연결
    require_once "../inc/db_connect.php";
    
    if (!isset($con)) {
        throw new Exception('DB 연결 객체가 존재하지 않습니다.');
    }

    // DB 연결 상태 확인
    if (mysqli_connect_errno()) {
        throw new Exception('DB 연결 실패: ' . mysqli_connect_error());
    }

    // 세션에서 사용자 정보 확인
    $userGrade = isset($_SESSION['grade']) ? $_SESSION['grade'] : '';
    $sessionId = isset($_SESSION['id']) ? $_SESSION['id'] : '';
    
    // 로그인 체크
    if (empty($userGrade) || empty($sessionId)) {
        throw new Exception('로그인이 필요합니다.');
    }
    
    // 관리자 권한 체크
    if ($userGrade !== 'admin') {
        throw new Exception('관리자 권한이 필요합니다.');
    }

    // POST 데이터 받기
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('잘못된 JSON 형식입니다.');
    }
    
    if (!isset($data['id']) || !is_numeric($data['id'])) {
        throw new Exception('유효하지 않은 ID입니다.');
    }
    
    $transferId = intval($data['id']);

    // 트랜잭션 시작
    mysqli_begin_transaction($con);

    try {
        // 1. 자동승인 큐에서 해당 거래 확인
        $checkQuery = "SELECT aaq.id, aaq.transfer_id, aaq.status, 
                              t.status as transfer_status,
                              t.depositor_name,
                              t.amount
                       FROM auto_approval_queue aaq
                       INNER JOIN transfer t ON aaq.transfer_id = t.id
                       WHERE aaq.transfer_id = ? AND aaq.status = 'waiting'
                       LIMIT 1";
        
        $checkStmt = mysqli_prepare($con, $checkQuery);
        if (!$checkStmt) {
            throw new Exception('자동승인 큐 조회 쿼리 준비 실패: ' . mysqli_error($con));
        }
        
        mysqli_stmt_bind_param($checkStmt, "i", $transferId);
        
        if (!mysqli_stmt_execute($checkStmt)) {
            throw new Exception('자동승인 큐 조회 실행 실패: ' . mysqli_stmt_error($checkStmt));
        }
        
        $checkResult = mysqli_stmt_get_result($checkStmt);
        $queueData = mysqli_fetch_assoc($checkResult);
        mysqli_stmt_close($checkStmt);
        
        if (!$queueData) {
            throw new Exception('자동승인 대기 중인 거래가 아닙니다.');
        }
        
        if ($queueData['transfer_status'] !== 'pending') {
            throw new Exception('대기 상태의 거래만 자동승인을 취소할 수 있습니다.');
        }

        // 2. 로그 기록 (삭제 전에 기록)
        // $logQuery = "INSERT INTO admin_action_log 
        //              (admin_id, action_type, target_type, target_id, description, created_at)
        //              VALUES (?, 'delete_auto_approval', 'transfer', ?, ?, NOW())";
        
        // $logStmt = mysqli_prepare($con, $logQuery);
        // if ($logStmt) {
        //     $description = "자동승인 취소 및 삭제 - Transfer ID: {$transferId}, 입금자: {$queueData['depositor_name']}, 금액: {$queueData['amount']}원";
        //     mysqli_stmt_bind_param($logStmt, "sis", $sessionId, $transferId, $description);
        //     mysqli_stmt_execute($logStmt);
        //     mysqli_stmt_close($logStmt);
        // }

        // 3. auto_approval_queue 테이블에서 삭제
        $deleteQueueQuery = "DELETE FROM auto_approval_queue WHERE id = ?";
        
        $deleteQueueStmt = mysqli_prepare($con, $deleteQueueQuery);
        if (!$deleteQueueStmt) {
            throw new Exception('자동승인 큐 삭제 쿼리 준비 실패: ' . mysqli_error($con));
        }
        
        mysqli_stmt_bind_param($deleteQueueStmt, "i", $queueData['id']);
        
        if (!mysqli_stmt_execute($deleteQueueStmt)) {
            throw new Exception('자동승인 큐 삭제 실행 실패: ' . mysqli_stmt_error($deleteQueueStmt));
        }
        
        $queueAffectedRows = mysqli_stmt_affected_rows($deleteQueueStmt);
        mysqli_stmt_close($deleteQueueStmt);

        // 4. transfer 테이블에서도 삭제
        $deleteTransferQuery = "DELETE FROM transfer WHERE id = ?";
        
        $deleteTransferStmt = mysqli_prepare($con, $deleteTransferQuery);
        if (!$deleteTransferStmt) {
            throw new Exception('Transfer 삭제 쿼리 준비 실패: ' . mysqli_error($con));
        }
        
        mysqli_stmt_bind_param($deleteTransferStmt, "i", $transferId);
        
        if (!mysqli_stmt_execute($deleteTransferStmt)) {
            throw new Exception('Transfer 삭제 실행 실패: ' . mysqli_stmt_error($deleteTransferStmt));
        }
        
        $transferAffectedRows = mysqli_stmt_affected_rows($deleteTransferStmt);
        mysqli_stmt_close($deleteTransferStmt);
        
        if ($queueAffectedRows === 0 && $transferAffectedRows === 0) {
            throw new Exception('삭제할 데이터가 없습니다.');
        }

        // 트랜잭션 커밋
        mysqli_commit($con);

        // 버퍼 내용 정리
        ob_clean();

        // 성공 응답
        echo json_encode([
            'success' => true,
            'message' => '자동승인이 취소되고 내역이 삭제되었습니다.',
            'data' => [
                'transfer_id' => $transferId,
                'queue_id' => $queueData['id'],
                'deleted_at' => date('Y-m-d H:i:s'),
                'depositor_name' => $queueData['depositor_name'],
                'amount' => $queueData['amount']
            ]
        ], JSON_UNESCAPED_UNICODE);

    } catch (Exception $e) {
        // 트랜잭션 롤백
        mysqli_rollback($con);
        throw $e;
    }

} catch (Exception $e) {
    // 버퍼 내용 정리
    ob_clean();
    
    // 에러 로깅
    error_log("cancel_auto_approval.php Error: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
    
    // 에러 응답
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'file' => basename($e->getFile()),
        'line' => $e->getLine()
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // DB 연결 정리
    if (isset($con)) {
        mysqli_close($con);
    }
    
    // 출력 버퍼 끝내기
    ob_end_flush();
}
?>