<?php
// 에러 표시 설정
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크
if (!isset($_SESSION['id'])) {
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => '파트너 권한이 필요합니다.'
    ]);
    exit;
}

$partner_user_id = $_SESSION['id'];  // 이것은 member.user_id 값

// POST 데이터 받기
$input = json_decode(file_get_contents('php://input'), true);
$withdrawal_id = isset($input['id']) ? intval($input['id']) : 0;

try {
    // 트랜잭션 시작
    mysqli_begin_transaction($con);
    
    // 1. 먼저 user_id로 member.id를 조회
    $getMemberIdQuery = "SELECT id FROM member WHERE user_id = ?";
    $getMemberIdStmt = mysqli_prepare($con, $getMemberIdQuery);
    
    if (!$getMemberIdStmt) {
        throw new Exception("회원 ID 조회 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($getMemberIdStmt, "s", $partner_user_id);
    mysqli_stmt_execute($getMemberIdStmt);
    $getMemberIdResult = mysqli_stmt_get_result($getMemberIdStmt);
    
    if (!$memberRow = mysqli_fetch_assoc($getMemberIdResult)) {
        throw new Exception("회원 정보를 찾을 수 없습니다.");
    }
    
    $partner_id = $memberRow['id'];  // member 테이블의 primary key
    mysqli_stmt_close($getMemberIdStmt);
    
    // 2. 출금 신청 정보 조회 및 잠금
    $checkQuery = "
        SELECT 
            t.id,
            t.user_id,
            t.amount,
            t.status
        FROM transfer t
        WHERE t.id = ?
        AND t.user_id = ?
        AND t.type = 'withdraw'
        FOR UPDATE
    ";
    
    $checkStmt = mysqli_prepare($con, $checkQuery);
    
    if (!$checkStmt) {
        throw new Exception("출금 정보 조회 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($checkStmt, "ii", $withdrawal_id, $partner_id);
    mysqli_stmt_execute($checkStmt);
    $checkResult = mysqli_stmt_get_result($checkStmt);
    
    if (!$row = mysqli_fetch_assoc($checkResult)) {
        throw new Exception("출금 신청 정보를 찾을 수 없습니다.");
    }
    
    mysqli_stmt_close($checkStmt);
    
    // 3. 상태 확인
    if ($row['status'] !== 'requested') {
        throw new Exception("신청 상태의 출금만 취소할 수 있습니다. 현재 상태: " . $row['status']);
    }
    
    // 4. 출금 신청 취소 처리
    $updateQuery = "
        UPDATE transfer 
        SET 
            status = 'cancelled',
            payment_status = 'cancelled',
            cancelled_at = NOW(),
            updated_at = NOW()
        WHERE id = ?
        AND user_id = ?
        AND status = 'requested'
    ";
    
    $updateStmt = mysqli_prepare($con, $updateQuery);
    
    if (!$updateStmt) {
        throw new Exception("출금 취소 처리 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($updateStmt, "ii", $withdrawal_id, $partner_id);
    
    if (!mysqli_stmt_execute($updateStmt)) {
        throw new Exception("출금 취소 업데이트 실패: " . mysqli_stmt_error($updateStmt));
    }
    
    $affected_rows = mysqli_stmt_affected_rows($updateStmt);
    mysqli_stmt_close($updateStmt);
    
    if ($affected_rows === 0) {
        throw new Exception("출금 취소 처리에 실패했습니다. 이미 처리된 신청일 수 있습니다.");
    }
    
    // 5. 활동 로그 기록 (테이블이 있는 경우)
    $logTableCheck = "SHOW TABLES LIKE 'activity_log'";
    $tableResult = mysqli_query($con, $logTableCheck);
    
    if (mysqli_num_rows($tableResult) > 0) {
        $logQuery = "
            INSERT INTO activity_log (
                user_id,
                action,
                details,
                ip_address,
                created_at
            ) VALUES (
                ?, 'withdrawal_cancel', ?, ?, NOW()
            )
        ";
        
        $logStmt = mysqli_prepare($con, $logQuery);
        
        if ($logStmt) {
            $details = json_encode([
                'withdrawal_id' => $withdrawal_id,
                'amount' => $row['amount'],
                'reason' => '파트너 직접 취소'
            ], JSON_UNESCAPED_UNICODE);
            
            $ip_address = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
            
            mysqli_stmt_bind_param($logStmt, "iss", $partner_id, $details, $ip_address);
            mysqli_stmt_execute($logStmt);
            mysqli_stmt_close($logStmt);
        }
    }
    
    // 트랜잭션 커밋
    mysqli_commit($con);
    
    // 성공 응답
    echo json_encode([
        'success' => true,
        'message' => '출금 신청이 취소되었습니다.',
        'withdrawal_id' => $withdrawal_id,
        'cancelled_amount' => $row['amount']
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    mysqli_rollback($con);
    
    error_log("Cancel Partner Withdrawal Error: " . $e->getMessage());
    
    // 에러 응답
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>