<?php
// 에러 표시 (디버깅용 - 운영환경에서는 주석처리)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크 - 로그인된 사용자만
if (!isset($_SESSION['id'])) {
    writeLog($con, 'BANK_ACCOUNT_DELETE_DENIED', null, null, 0, 
            '은행 계좌 삭제 시도 실패 - 미인증 사용자');
    
    http_response_code(403);
    echo json_encode([
        'success' => false, 
        'message' => '접근 권한이 없습니다.',
        'error_code' => 'AUTH_FAILED'
    ]);
    exit;
}

try {
    // 현재 로그인한 사용자 정보 조회
    $currentUserId = $_SESSION['id'];
    $currentUserQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $currentUserQuery);
    mysqli_stmt_bind_param($stmt, "s", $currentUserId);
    mysqli_stmt_execute($stmt);
    $currentUserResult = mysqli_stmt_get_result($stmt);
    $currentUserData = mysqli_fetch_assoc($currentUserResult);
    mysqli_stmt_close($stmt);

    if (!$currentUserData) {
        throw new Exception('사용자 정보를 찾을 수 없습니다.');
    }

    $currentMemberId = $currentUserData['id'];
    $currentUserName = $currentUserData['user_name'];

    // JSON 데이터 받기
    $inputData = json_decode(file_get_contents('php://input'), true);

    // 필수 파라미터 확인
    if (!isset($inputData['id']) || empty($inputData['id'])) {
        $requestData = ['bank_account_id' => null, 'deleted_by' => $currentUserName];
        writeLog($con, 'BANK_ACCOUNT_DELETE_FAILED', $requestData, null, 0, 
                '은행 계좌 삭제 실패 - 계좌 ID 누락', $currentMemberId);
        
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => '계좌 ID가 필요합니다.',
            'error_code' => 'MISSING_ID'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $id = intval($inputData['id']);

    // 입력 데이터 로그용
    $requestData = [
        'bank_account_id' => $id,
        'deleted_by' => $currentUserName
    ];

    // 트랜잭션 시작
    beginTransaction($con);
    
    // 1. 기존 데이터 조회 (로그용 및 존재 여부 확인)
    $selectQuery = "
        SELECT ba.*, COUNT(m.id) as member_count
        FROM bank_account ba
        LEFT JOIN member m ON ba.id = m.bank_account_id
        WHERE ba.id = ?
        GROUP BY ba.id
    ";
    $selectStmt = mysqli_prepare($con, $selectQuery);
    
    if (!$selectStmt) {
        throw new Exception("SELECT 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($selectStmt, "i", $id);
    mysqli_stmt_execute($selectStmt);
    $result = mysqli_stmt_get_result($selectStmt);
    $accountData = mysqli_fetch_assoc($result);
    
    if (!$accountData) {
        mysqli_stmt_close($selectStmt);
        writeLog($con, 'BANK_ACCOUNT_DELETE_FAILED', $requestData, null, 0, 
                "은행 계좌 삭제 실패 - 존재하지 않는 계좌 ID: {$id}", $currentMemberId);
        throw new Exception("존재하지 않는 은행 계좌입니다.");
    }
    
    mysqli_stmt_close($selectStmt);

    // 연결된 가맹점 목록 조회 (로그용)
    $connectedMembers = [];
    if ($accountData['member_count'] > 0) {
        $memberListQuery = "SELECT user_id, user_name FROM member WHERE bank_account_id = ?";
        $memberListStmt = mysqli_prepare($con, $memberListQuery);
        mysqli_stmt_bind_param($memberListStmt, "i", $id);
        mysqli_stmt_execute($memberListStmt);
        $memberListResult = mysqli_stmt_get_result($memberListStmt);
        
        while ($member = mysqli_fetch_assoc($memberListResult)) {
            $connectedMembers[] = $member['user_id'] . '(' . $member['user_name'] . ')';
        }
        mysqli_stmt_close($memberListStmt);
    }

    // 삭제 전 데이터 저장 (로그용)
    $logOldData = [
        'id' => $accountData['id'],
        'bank' => $accountData['bank'],
        'account_number' => $accountData['account_number'],
        'account_holder' => $accountData['account_holder'],
        'member_count' => $accountData['member_count'],
        'connected_members' => $connectedMembers,
        'created_at' => $accountData['created_at'],
        'deleted_by' => $currentUserName
    ];
    
    // 2. 연결된 가맹점이 있는 경우 연결 해제
    $affectedMembers = 0;
    if ($accountData['member_count'] > 0) {
        $updateMemberQuery = "UPDATE member SET bank_account_id = NULL WHERE bank_account_id = ?";
        $updateMemberStmt = mysqli_prepare($con, $updateMemberQuery);
        
        if (!$updateMemberStmt) {
            throw new Exception("가맹점 연결 해제 쿼리 준비 실패: " . mysqli_error($con));
        }
        
        mysqli_stmt_bind_param($updateMemberStmt, "i", $id);
        
        if (!mysqli_stmt_execute($updateMemberStmt)) {
            writeLog($con, 'BANK_ACCOUNT_DELETE_FAILED', $requestData, $logOldData, 0, 
                    "은행 계좌 삭제 실패 - 가맹점 연결 해제 오류: " . mysqli_stmt_error($updateMemberStmt), $currentMemberId);
            throw new Exception("가맹점 연결 해제 실패: " . mysqli_stmt_error($updateMemberStmt));
        }
        
        $affectedMembers = mysqli_stmt_affected_rows($updateMemberStmt);
        mysqli_stmt_close($updateMemberStmt);
    }
    
    // 3. 은행 계좌 삭제
    $deleteQuery = "DELETE FROM bank_account WHERE id = ?";
    $deleteStmt = mysqli_prepare($con, $deleteQuery);
    
    if (!$deleteStmt) {
        throw new Exception("DELETE 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($deleteStmt, "i", $id);
    
    if (!mysqli_stmt_execute($deleteStmt)) {
        writeLog($con, 'BANK_ACCOUNT_DELETE_FAILED', $requestData, $logOldData, 0, 
                "은행 계좌 삭제 실패 - 삭제 쿼리 오류: " . mysqli_stmt_error($deleteStmt), $currentMemberId);
        throw new Exception("데이터 삭제 실패: " . mysqli_stmt_error($deleteStmt));
    }
    
    $deletedRows = mysqli_stmt_affected_rows($deleteStmt);
    mysqli_stmt_close($deleteStmt);

    if ($deletedRows === 0) {
        writeLog($con, 'BANK_ACCOUNT_DELETE_FAILED', $requestData, $logOldData, 0, 
                "은행 계좌 삭제 실패 - 삭제된 행 없음", $currentMemberId);
        throw new Exception("삭제할 계좌를 찾을 수 없습니다.");
    }
    
    // 트랜잭션 커밋
    commitTransaction($con);
    
    // 성공 로그 기록 - currentMemberId 전달
    $detail = "은행 계좌 삭제 성공 - 은행: {$accountData['bank']}, 계좌: {$accountData['account_number']}, " .
             "예금주: {$accountData['account_holder']}, 삭제자: {$currentUserName}({$currentUserId})";
    if ($affectedMembers > 0) {
        $detail .= ", 연결 해제된 가맹점: {$affectedMembers}개 [" . implode(', ', $connectedMembers) . "]";
    }
    
    writeLog($con, 'BANK_ACCOUNT_DELETE_SUCCESS', $logOldData, null, 1, $detail, $currentMemberId);
    
    // 성공 응답
    $message = '은행 계좌가 성공적으로 삭제되었습니다.';
    if ($affectedMembers > 0) {
        $message .= " ({$affectedMembers}개의 가맹점 연결이 해제되었습니다.)";
    }
    
    echo json_encode([
        'success' => true,
        'message' => $message,
        'id' => $id,
        'affected_members' => $affectedMembers,
        'data' => [
            'deleted_account' => [
                'bank' => $accountData['bank'],
                'account_number' => $accountData['account_number'],
                'account_holder' => $accountData['account_holder']
            ],
            'disconnected_members' => $connectedMembers,
            'deleted_by' => $currentUserName
        ]
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    rollbackTransaction($con);
    
    $errorMessage = $e->getMessage();
    
    // 실패 로그 기록 - currentMemberId 전달
    if (isset($currentMemberId) && isset($requestData)) {
        writeLog($con, 'BANK_ACCOUNT_DELETE_FAILED', $requestData, $logOldData ?? null, 0, 
                "은행 계좌 삭제 실패 - " . $errorMessage, $currentMemberId);
    }
    
    // 에러 응답
    http_response_code(500);
    
    // 존재하지 않는 경우 404 Not Found
    if (strpos($errorMessage, '존재하지 않는') !== false) {
        http_response_code(404);
    }
    
    echo json_encode([
        'success' => false,
        'message' => $errorMessage,
        'error_code' => 'DELETE_FAILED'
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>