<?php
// 에러 표시 (디버깅용 - 운영환경에서는 주석처리)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크 - 로그인된 사용자만
if (!isset($_SESSION['id'])) {
    writeLog($con, 'KAKAO_ACCOUNT_DELETE_DENIED', null, null, 0, 
            '카카오 계정 삭제 시도 실패 - 미인증 사용자');
    
    http_response_code(403);
    echo json_encode([
        'success' => false, 
        'message' => '접근 권한이 없습니다.',
        'error_code' => 'AUTH_FAILED'
    ]);
    exit;
}

try {
    // 현재 로그인한 사용자 정보 조회
    $currentUserId = $_SESSION['id'];
    $currentUserQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $currentUserQuery);
    mysqli_stmt_bind_param($stmt, "s", $currentUserId);
    mysqli_stmt_execute($stmt);
    $currentUserResult = mysqli_stmt_get_result($stmt);
    $currentUserData = mysqli_fetch_assoc($currentUserResult);
    mysqli_stmt_close($stmt);

    if (!$currentUserData) {
        throw new Exception('사용자 정보를 찾을 수 없습니다.');
    }

    $currentMemberId = $currentUserData['id'];
    $currentUserName = $currentUserData['user_name'];

    // JSON 데이터 받기
    $inputData = json_decode(file_get_contents('php://input'), true);
    $id = isset($inputData['id']) ? intval($inputData['id']) : 0;

    // 입력 데이터 로그용
    $requestData = [
        'kakao_account_id' => $id,
        'deleted_by' => $currentUserName
    ];

    // 유효성 검사
    if (empty($id) || $id <= 0) {
        writeLog($con, 'KAKAO_ACCOUNT_DELETE_FAILED', $requestData, null, 0, 
                "카카오 계정 삭제 실패 - 유효하지 않은 ID: {$id}", $currentMemberId);
        
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => '유효하지 않은 계정 ID입니다.',
            'error_code' => 'INVALID_ID'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // 트랜잭션 시작
    beginTransaction($con);
    
    // 1. 카카오 계정 존재 여부 및 정보 확인
    $checkQuery = "
        SELECT 
            ka.id,
            ka.kakao_tid,
            ka.name,
            ka.qr_link,
            ka.friend_link,
            ka.created_at,
            COUNT(m.id) as member_count
        FROM kakao_account ka
        LEFT JOIN member m ON ka.id = m.kakao_account_id
        WHERE ka.id = ?
        GROUP BY ka.id
    ";
    
    $checkStmt = mysqli_prepare($con, $checkQuery);
    
    if (!$checkStmt) {
        throw new Exception("계정 확인 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($checkStmt, "i", $id);
    mysqli_stmt_execute($checkStmt);
    
    $result = mysqli_stmt_get_result($checkStmt);
    $kakaoAccount = mysqli_fetch_assoc($result);
    
    if (!$kakaoAccount) {
        mysqli_stmt_close($checkStmt);
        writeLog($con, 'KAKAO_ACCOUNT_DELETE_FAILED', $requestData, null, 0, 
                "카카오 계정 삭제 실패 - 존재하지 않는 계정 ID: {$id}", $currentMemberId);
        throw new Exception("존재하지 않는 카카오 계정입니다.");
    }
    
    $kakaoTid = $kakaoAccount['kakao_tid'];
    $accountName = $kakaoAccount['name'];
    $memberCount = $kakaoAccount['member_count'];
    
    mysqli_stmt_close($checkStmt);
    
    // 삭제 전 데이터 저장 (로그용)
    $logOldData = [
        'id' => $id,
        'kakao_tid' => $kakaoTid,
        'name' => $accountName,
        'qr_link' => $kakaoAccount['qr_link'],
        'friend_link' => $kakaoAccount['friend_link'],
        'created_at' => $kakaoAccount['created_at'],
        'member_count' => $memberCount,
        'deleted_by' => $currentUserName
    ];
    
    // 연결된 가맹점 목록 조회 (로그용)
    $connectedMembers = [];
    if ($memberCount > 0) {
        $memberListQuery = "SELECT user_id, user_name FROM member WHERE kakao_account_id = ?";
        $memberListStmt = mysqli_prepare($con, $memberListQuery);
        mysqli_stmt_bind_param($memberListStmt, "i", $id);
        mysqli_stmt_execute($memberListStmt);
        $memberListResult = mysqli_stmt_get_result($memberListStmt);
        
        while ($member = mysqli_fetch_assoc($memberListResult)) {
            $connectedMembers[] = $member['user_id'] . '(' . $member['user_name'] . ')';
        }
        mysqli_stmt_close($memberListStmt);
        
        $logOldData['connected_members'] = $connectedMembers;
    }
    
    // 2. 연결된 가맹점이 있는 경우 연결 해제
    if ($memberCount > 0) {
        $updateMemberQuery = "UPDATE member SET kakao_account_id = NULL, kakao_link_type = NULL WHERE kakao_account_id = ?";
        $updateStmt = mysqli_prepare($con, $updateMemberQuery);
        
        if (!$updateStmt) {
            throw new Exception("가맹점 연결 해제 쿼리 준비 실패: " . mysqli_error($con));
        }
        
        mysqli_stmt_bind_param($updateStmt, "i", $id);
        
        if (!mysqli_stmt_execute($updateStmt)) {
            writeLog($con, 'KAKAO_ACCOUNT_DELETE_FAILED', $requestData, $logOldData, 0, 
                    "카카오 계정 삭제 실패 - 가맹점 연결 해제 오류: " . mysqli_stmt_error($updateStmt), $currentMemberId);
            throw new Exception("가맹점 연결 해제 실패: " . mysqli_stmt_error($updateStmt));
        }
        
        mysqli_stmt_close($updateStmt);
    }
    
    // 3. 카카오 계정 삭제
    $deleteQuery = "DELETE FROM kakao_account WHERE id = ?";
    $deleteStmt = mysqli_prepare($con, $deleteQuery);
    
    if (!$deleteStmt) {
        throw new Exception("삭제 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($deleteStmt, "i", $id);
    
    if (!mysqli_stmt_execute($deleteStmt)) {
        writeLog($con, 'KAKAO_ACCOUNT_DELETE_FAILED', $requestData, $logOldData, 0, 
                "카카오 계정 삭제 실패 - 삭제 쿼리 오류: " . mysqli_stmt_error($deleteStmt), $currentMemberId);
        throw new Exception("카카오 계정 삭제 실패: " . mysqli_stmt_error($deleteStmt));
    }
    
    $affectedRows = mysqli_stmt_affected_rows($deleteStmt);
    mysqli_stmt_close($deleteStmt);
    
    if ($affectedRows === 0) {
        writeLog($con, 'KAKAO_ACCOUNT_DELETE_FAILED', $requestData, $logOldData, 0, 
                "카카오 계정 삭제 실패 - 삭제할 계정을 찾을 수 없음", $currentMemberId);
        throw new Exception("삭제할 계정을 찾을 수 없습니다.");
    }
    
    // 트랜잭션 커밋
    commitTransaction($con);
    
    // 성공 로그 기록 - currentMemberId 전달
    $detail = "카카오 계정 삭제 성공 - 계정명: {$accountName}, TID: {$kakaoTid}, " .
             "삭제자: {$currentUserName}({$currentUserId})";
    if ($memberCount > 0) {
        $detail .= ", 연결 해제된 가맹점: {$memberCount}개 [" . implode(', ', $connectedMembers) . "]";
    }
    
    writeLog($con, 'KAKAO_ACCOUNT_DELETE_SUCCESS', $logOldData, null, 1, $detail, $currentMemberId);
    
    // 성공 응답
    $responseData = [
        'success' => true,
        'message' => '카카오 계정이 성공적으로 삭제되었습니다.',
        'data' => [
            'deleted_id' => $id,
            'account_name' => $accountName,
            'kakao_tid' => $kakaoTid,
            'affected_member' => $memberCount,
            'deleted_by' => $currentUserName
        ]
    ];
    
    if ($memberCount > 0) {
        $responseData['message'] .= " {$memberCount}개의 가맹점 연결이 해제되었습니다.";
        $responseData['data']['disconnected_members'] = $connectedMembers;
    }
    
    echo json_encode($responseData, JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    rollbackTransaction($con);
    
    $errorMessage = $e->getMessage();
    
    // 실패 로그 기록 - currentMemberId 전달
    if (isset($currentMemberId) && isset($requestData)) {
        writeLog($con, 'KAKAO_ACCOUNT_DELETE_FAILED', $requestData, $logOldData ?? null, 0, 
                "카카오 계정 삭제 실패 - " . $errorMessage, $currentMemberId);
    }
    
    // 에러 응답
    http_response_code(500);
    
    // 존재하지 않는 계정인 경우 404
    if (strpos($errorMessage, '존재하지 않는') !== false) {
        http_response_code(404);
    }
    
    echo json_encode([
        'success' => false,
        'message' => $errorMessage,
        'error_code' => 'DELETE_FAILED'
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>