<?php
header('Content-Type: application/json; charset=UTF-8');

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// DB 연결
include '../inc/session.php';

// 권한 체크 - 로그인된 사용자만
if (!isset($_SESSION['id'])) {
    writeLog($con, 'MEMBER_DELETE_DENIED', null, null, 0, 
            '가맹점 삭제 시도 실패 - 미인증 사용자');
    
    http_response_code(403);
    echo json_encode([
        'success' => false, 
        'message' => '접근 권한이 없습니다.',
        'error_code' => 'AUTH_FAILED'
    ]);
    exit;
}

try {
    // 현재 로그인한 사용자 정보 조회
    $currentUserId = $_SESSION['id'];
    $currentUserQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $currentUserQuery);
    mysqli_stmt_bind_param($stmt, "s", $currentUserId);
    mysqli_stmt_execute($stmt);
    $currentUserResult = mysqli_stmt_get_result($stmt);
    $currentUserData = mysqli_fetch_assoc($currentUserResult);
    mysqli_stmt_close($stmt);

    if (!$currentUserData) {
        throw new Exception('사용자 정보를 찾을 수 없습니다.');
    }

    $currentMemberId = $currentUserData['id'];
    $currentUserName = $currentUserData['user_name'];

    // 관리자 권한 확인 (삭제는 관리자만)
    if ($currentUserData['grade'] !== 'admin') {
        writeLog($con, 'MEMBER_DELETE_DENIED', null, null, 0, 
                "가맹점 삭제 권한 없음 - 시도한 사용자: {$currentUserName}({$currentUserId}), 등급: {$currentUserData['grade']}", 
                $currentMemberId);
        
        http_response_code(403);
        echo json_encode([
            'success' => false,
            'message' => '가맹점 삭제 권한이 없습니다. 관리자만 삭제할 수 있습니다.'
        ]);
        exit;
    }

    // JSON 입력 받기
    $input = json_decode(file_get_contents('php://input'), true);
    $id = isset($input['id']) ? trim($input['id']) : '';

    // 입력 데이터 로그용
    $inputData = [
        'target_user_id' => $id,
        'deleted_by' => $currentUserName
    ];

    if (empty($id)) {
        writeLog($con, 'MEMBER_DELETE_FAILED', $inputData, null, 0, 
                '가맹점 삭제 실패 - 유효하지 않은 ID', $currentMemberId);
        
        echo json_encode([
            'success' => false,
            'message' => '유효하지 않은 ID입니다.'
        ]);
        exit;
    }

    // 트랜잭션 시작
    mysqli_autocommit($con, false);
    mysqli_begin_transaction($con);

    // 가맹점 존재 확인 및 데이터 조회
    $checkQuery = "SELECT * FROM member WHERE user_id = ?";
    $checkStmt = mysqli_prepare($con, $checkQuery);
    mysqli_stmt_bind_param($checkStmt, "s", $id);
    mysqli_stmt_execute($checkStmt);
    $checkResult = mysqli_stmt_get_result($checkStmt);
    $memberData = mysqli_fetch_assoc($checkResult);
    mysqli_stmt_close($checkStmt);

    if (!$memberData) {
        writeLog($con, 'MEMBER_DELETE_FAILED', $inputData, null, 0, 
                "가맹점 삭제 실패 - 존재하지 않는 가맹점: {$id}", $currentMemberId);
        
        echo json_encode([
            'success' => false,
            'message' => '존재하지 않는 가맹점입니다.'
        ]);
        exit;
    }

    $memberCode = $memberData['code'];
    $memberName = $memberData['user_name'];

    // 자기 자신 삭제 방지
    if ($id === $currentUserId) {
        writeLog($con, 'MEMBER_DELETE_FAILED', $inputData, $memberData, 0, 
                "가맹점 삭제 실패 - 자기 자신 삭제 시도: {$id}", $currentMemberId);
        
        echo json_encode([
            'success' => false,
            'message' => '자기 자신은 삭제할 수 없습니다.'
        ]);
        exit;
    }

    // 하위 가맹점 존재 확인
    $childQuery = "SELECT COUNT(*) as count FROM member WHERE JSON_CONTAINS(parents, ?, '$')";
    $childStmt = mysqli_prepare($con, $childQuery);
    $memberIdStr = (string)$memberData['id'];
    mysqli_stmt_bind_param($childStmt, "s", $memberIdStr);
    mysqli_stmt_execute($childStmt);
    $childResult = mysqli_stmt_get_result($childStmt);
    $childCount = mysqli_fetch_assoc($childResult)['count'];
    mysqli_stmt_close($childStmt);

    if ($childCount > 0) {
        writeLog($con, 'MEMBER_DELETE_FAILED', $inputData, $memberData, 0, 
                "가맹점 삭제 실패 - 하위 가맹점 존재: {$id} (하위 {$childCount}개)", $currentMemberId);
        
        echo json_encode([
            'success' => false,
            'message' => "해당 가맹점에 {$childCount}개의 하위 가맹점이 있어 삭제할 수 없습니다. 먼저 하위 가맹점을 삭제해주세요."
        ]);
        exit;
    }

    // 관련 데이터 확인 (입금 내역)
    $relatedQueries = [
        'transfer' => "SELECT COUNT(*) as count FROM transfer WHERE user_id = ?",
        'transaction_history' => "SELECT COUNT(*) as count FROM transaction_history WHERE user_id = ?",
        'rolling_history' => "SELECT COUNT(*) as count FROM rolling_history WHERE parent_member_id = ? OR child_member_id = ?",
        'manual_transactions' => "SELECT COUNT(*) as count FROM manual_transactions WHERE user_id = ?"
    ];

    $relatedData = [];
    foreach ($relatedQueries as $table => $query) {
        $stmt = mysqli_prepare($con, $query);
        if ($table === 'rolling_history') {
            mysqli_stmt_bind_param($stmt, "ii", $memberData['id'], $memberData['id']);
        } else {
            mysqli_stmt_bind_param($stmt, "i", $memberData['id']);
        }
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        $count = mysqli_fetch_assoc($result)['count'];
        mysqli_stmt_close($stmt);
        
        if ($count > 0) {
            $relatedData[$table] = $count;
        }
    }

    if (!empty($relatedData)) {
        $relatedInfo = [];
        foreach ($relatedData as $table => $count) {
            $relatedInfo[] = "{$table}: {$count}건";
        }
        $relatedText = implode(', ', $relatedInfo);
        
        writeLog($con, 'MEMBER_DELETE_FAILED', $inputData, $memberData, 0, 
                "가맹점 삭제 실패 - 관련 데이터 존재: {$id} ({$relatedText})", $currentMemberId);
        
        echo json_encode([
            'success' => false,
            'message' => "해당 가맹점과 연결된 거래 내역이 있어 삭제할 수 없습니다. 관련 데이터: {$relatedText}"
        ]);
        exit;
    }

    // 가맹점 삭제 실행
    $deleteQuery = "DELETE FROM member WHERE user_id = ?";
    $deleteStmt = mysqli_prepare($con, $deleteQuery);
    mysqli_stmt_bind_param($deleteStmt, "s", $id);
    $result = mysqli_stmt_execute($deleteStmt);
    
    if (!$result) {
        writeLog($con, 'MEMBER_DELETE_FAILED', $inputData, $memberData, 0, 
                "가맹점 삭제 실패 - DB 오류: " . mysqli_error($con), $currentMemberId);
        throw new Exception('가맹점 삭제 중 오류가 발생했습니다: ' . mysqli_error($con));
    }

    $affectedRows = mysqli_stmt_affected_rows($deleteStmt);
    mysqli_stmt_close($deleteStmt);

    if ($affectedRows > 0) {
        // 트랜잭션 커밋
        mysqli_commit($con);

        // 성공 로그 기록
        $detail = "가맹점 삭제 완료 - ID: {$id}, 가맹점명: {$memberName}, 코드: {$memberCode}, " .
                 "삭제자: {$currentUserName}({$currentUserId})";
        
        writeLog($con, 'MEMBER_DELETE_SUCCESS', $memberData, null, 1, $detail, $currentMemberId);

        echo json_encode([
            'success' => true,
            'message' => '가맹점이 삭제되었습니다.',
            'deleted_member' => [
                'user_id' => $id,
                'user_name' => $memberName,
                'code' => $memberCode
            ]
        ]);
    } else {
        writeLog($con, 'MEMBER_DELETE_FAILED', $inputData, $memberData, 0, 
                "가맹점 삭제 실패 - 삭제된 행 없음", $currentMemberId);
        
        echo json_encode([
            'success' => false,
            'message' => '가맹점 삭제에 실패했습니다.'
        ]);
    }

} catch (Exception $e) {
    // 트랜잭션 롤백
    if (isset($con)) {
        mysqli_rollback($con);
    }

    // 에러 로그 기록
    if (isset($currentMemberId) && isset($inputData)) {
        writeLog($con, 'MEMBER_DELETE_ERROR', $inputData, $memberData ?? null, 0, 
                "가맹점 삭제 오류: " . $e->getMessage(), $currentMemberId);
    }

    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);

} finally {
    // autocommit 복원
    if (isset($con)) {
        mysqli_autocommit($con, true);
        mysqli_close($con);
    }
}
?>