<?php
header('Content-Type: application/json');

// DB 연결
include '../inc/session.php';

// POST 데이터 받기
$userId = isset($_POST['user_id']) ? trim($_POST['user_id']) : '';
$userName = isset($_POST['user_name']) ? trim($_POST['user_name']) : '';
$code = isset($_POST['code']) ? trim($_POST['code']) : '';
$bank = isset($_POST['bank']) ? trim($_POST['bank']) : '';
$accountNumber = isset($_POST['account_number']) ? trim($_POST['account_number']) : '';
$accountHolder = isset($_POST['account_holder']) ? trim($_POST['account_holder']) : '';
$rate = isset($_POST['rate']) ? floatval($_POST['rate']) : 0.00;
$password = isset($_POST['password']) ? trim($_POST['password']) : '';

// 유효성 검사
if (empty($userId)) {
    echo json_encode([
        'success' => false,
        'message' => '가맹점 아이디가 필요합니다.'
    ]);
    exit;
}

if (empty($userName) || empty($code)) {
    echo json_encode([
        'success' => false,
        'message' => '가맹점명과 가맹점 코드는 필수 입력 항목입니다.'
    ]);
    exit;
}

try {
    // 가맹점 존재 확인
    $checkSql = "SELECT id, code FROM member WHERE user_id = ?";
    $checkStmt = $pdo->prepare($checkSql);
    $checkStmt->execute([$userId]);
    $member = $checkStmt->fetch();
    
    if (!$member) {
        echo json_encode([
            'success' => false,
            'message' => '존재하지 않는 가맹점입니다.'
        ]);
        exit;
    }
    
    // 코드 중복 체크 (자신을 제외한 다른 가맹점)
    if ($member['code'] !== $code) {
        $codeSql = "SELECT COUNT(*) as count FROM member WHERE code = ? AND user_id != ?";
        $codeStmt = $pdo->prepare($codeSql);
        $codeStmt->execute([$code, $userId]);
        $codeCount = $codeStmt->fetch()['count'];
        
        if ($codeCount > 0) {
            echo json_encode([
                'success' => false,
                'message' => '이미 사용 중인 가맹점 코드입니다.'
            ]);
            exit;
        }
    }
    
    // 업데이트 쿼리 준비
    $updateFields = [
        "user_name = ?",
        "code = ?",
        "bank = ?",
        "account_holder = ?",
        "account_number = ?",
        "rate = ?",
        "updated_at = NOW()"
    ];
    $updateParams = [
        $userName,
        $code,
        $bank,
        $accountHolder,
        $accountNumber,
        $rate
    ];
    
    // 비밀번호가 입력된 경우에만 업데이트
    if (!empty($password)) {
        $updateFields[] = "password = ?";
        $updateParams[] = password_hash($password, PASSWORD_DEFAULT);
    }
    
    // user_id를 마지막에 추가
    $updateParams[] = $userId;
    
    // 가맹점 정보 업데이트
    $sql = "UPDATE member SET " . implode(", ", $updateFields) . " WHERE user_id = ?";
    $stmt = $pdo->prepare($sql);
    $result = $stmt->execute($updateParams);
    
    if ($result) {
        echo json_encode([
            'success' => true,
            'message' => '가맹점 정보가 수정되었습니다.'
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => '가맹점 수정 중 오류가 발생했습니다.'
        ]);
    }
    
} catch (PDOException $e) {
    error_log('가맹점 수정 오류: ' . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => '데이터베이스 오류가 발생했습니다.'
    ]);
}
?>