<?php
session_start();

// 에러 출력 비활성화 (Excel 다운로드 시 출력 방지)
ini_set('display_errors', 0);
error_reporting(0);

try {
    // DB 연결
    require_once "../inc/db_connect.php";
    
    if (!isset($con)) {
        throw new Exception('DB 연결 실패');
    }

    // 세션 확인
    $userGrade = $_SESSION['grade'] ?? '';
    $sessionId = $_SESSION['id'] ?? '';
    
    if (empty($userGrade) || empty($sessionId)) {
        throw new Exception('로그인이 필요합니다');
    }
    
    // 엑셀 다운로드 요청 확인
    if (!isset($_GET['export']) || $_GET['export'] !== 'excel') {
        throw new Exception('잘못된 요청입니다');
    }
    
    // SimpleXLSXGen 라이브러리 포함
    if (!file_exists('../libs/SimpleXLSXGen.php')) {
        throw new Exception('Excel 라이브러리가 없습니다');
    }
    require_once '../libs/SimpleXLSXGen.php';
    
    // 파트너인 경우 member ID 찾기
    $memberId = null;
    if ($userGrade !== 'admin') {
        $memberQuery = "SELECT id FROM member WHERE user_id = ? LIMIT 1";
        $memberStmt = mysqli_prepare($con, $memberQuery);
        mysqli_stmt_bind_param($memberStmt, "s", $sessionId);
        mysqli_stmt_execute($memberStmt);
        $memberResult = mysqli_stmt_get_result($memberStmt);
        
        if ($memberRow = mysqli_fetch_assoc($memberResult)) {
            $memberId = $memberRow['id'];
        } else {
            throw new Exception('사용자 정보를 찾을 수 없습니다');
        }
    }

    // 검색 파라미터 받기
    $search = trim($_GET['search'] ?? '');
    $storeCode = $_GET['store_code'] ?? '';
    $kakaoCode = $_GET['kakao_code'] ?? '';
    $bankCode = $_GET['bank_code'] ?? '';
    
    // 날짜 파라미터
    $startDate = !empty(trim($_GET['start_date'] ?? '')) 
        ? trim($_GET['start_date']) 
        : date('Y-m-d', strtotime('-1 day'));
    $endDate = !empty(trim($_GET['end_date'] ?? '')) 
        ? trim($_GET['end_date']) 
        : date('Y-m-d');

    // WHERE 조건 구성
    $whereConditions = ["t.type = 'deposit'"];
    $params = [];
    $types = '';

    // 파트너는 자기 데이터만
    if ($userGrade !== 'admin' && $memberId !== null) {
        $whereConditions[] = "t.user_id = ?";
        $params[] = $memberId;
        $types .= 'i';
    }

    // 검색어 조건 (fetch_deposits.php와 동일한 로직)
    if (!empty($search)) {
        if (strpos($search, '*') !== false || strpos($search, ' ') !== false) {
            $searchParts = preg_split('/[\*\s]+/', $search);
            
            if (count($searchParts) == 2 && !empty($searchParts[0]) && !empty($searchParts[1])) {
                $whereConditions[] = "(t.depositor_name LIKE ? AND t.depositor_name LIKE ?)";
                $params[] = $searchParts[0] . '%';
                $params[] = '%' . $searchParts[1];
                $types .= 'ss';
            } else {
                $whereConditions[] = "t.depositor_name LIKE ?";
                $params[] = '%' . $search . '%';
                $types .= 's';
            }
        } 
        else if (mb_strlen($search, 'UTF-8') == 2) {
            $firstChar = mb_substr($search, 0, 1, 'UTF-8');
            $lastChar = mb_substr($search, 1, 1, 'UTF-8');
            
            $whereConditions[] = "(t.depositor_name LIKE ? AND t.depositor_name LIKE ?)";
            $params[] = $firstChar . '%';
            $params[] = '%' . $lastChar;
            $types .= 'ss';
        }
        else {
            $whereConditions[] = "t.depositor_name LIKE ?";
            $params[] = '%' . $search . '%';
            $types .= 's';
        }
    }

    // 날짜 범위
    if (!empty($startDate)) {
        $whereConditions[] = "DATE(t.created_at) >= ?";
        $params[] = $startDate;
        $types .= 's';
    }

    if (!empty($endDate)) {
        $whereConditions[] = "DATE(t.created_at) <= ?";
        $params[] = $endDate;
        $types .= 's';
    }

    // 관리자 전용 필터
    if ($userGrade === 'admin') {
        if (!empty($storeCode)) {
            $whereConditions[] = "t.user_id = ?";
            $params[] = $storeCode;
            $types .= 'i';
        }

        if (!empty($kakaoCode)) {
            if (is_array($kakaoCode)) {
                $kakaoIds = array_filter($kakaoCode);
                if (!empty($kakaoIds)) {
                    $placeholders = str_repeat('?,', count($kakaoIds) - 1) . '?';
                    $whereConditions[] = "m.kakao_account_id IN ($placeholders)";
                    foreach ($kakaoIds as $id) {
                        $params[] = $id;
                        $types .= 'i';
                    }
                }
            } else {
                $whereConditions[] = "m.kakao_account_id = ?";
                $params[] = $kakaoCode;
                $types .= 'i';
            }
        }

        if (!empty($bankCode)) {
            $whereConditions[] = "m.bank_account_id = ?";
            $params[] = $bankCode;
            $types .= 'i';
        }
    }

    $whereClause = implode(' AND ', $whereConditions);

    // 데이터 조회 쿼리 (fetch_deposits.php와 동일)
    $query = "SELECT 
                t.id,
                t.depositor_name,
                t.amount,
                t.kakao_tid,
                t.status,
                t.created_at,
                t.updated_at,
                t.user_id,
                t.trans_id,
                COALESCE(m.user_name, '미지정') as store_name,
                COALESCE(m.user_id, '-') as store_user_id,
                COALESCE(m.code, '-') as code,
                m.id as member_id,
                COALESCE(th.total_fee_rate, 0) as total_fee_rate,
                COALESCE(th.total_fee_amount, 0) as total_fee_amount,
                COALESCE(th.fee_rate, 0) as company_fee_rate,
                COALESCE(th.fee_amount, 0) as company_fee_amount,
                COALESCE(th.actual_amount, 0) as actual_amount,
                COALESCE(th.balance_after, 0) as balance_after,
                th.id as transaction_history_id
            FROM transfer t
            LEFT JOIN member m ON t.user_id = m.id
            LEFT JOIN transaction_history th ON t.trans_id = th.trans_id AND th.status = 'completed'
            WHERE " . $whereClause . "
            ORDER BY 
                CASE 
                    WHEN t.status IN ('requested', 'pending') THEN 0
                    ELSE 1
                END,
                COALESCE(t.updated_at, t.created_at) DESC, 
                t.id DESC";

    $stmt = mysqli_prepare($con, $query);
    
    if (!empty($params)) {
        mysqli_stmt_bind_param($stmt, $types, ...$params);
    }
    
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);

    // 엑셀 데이터 준비
    $excelData = [];
    
    // 헤더 행 - 관리자와 일반사용자 구분
    if ($userGrade === 'admin') {
        $excelData[] = ['ID', '입금자명', '요청금액', '처리금액', '총수수료', '회사수수료', '롤링수수료', '잔액', '업체명', '업체ID', '카카오계정', '상태', '신청일시', '처리일시'];
    } else {
        $excelData[] = ['ID', '입금자명', '요청금액', '처리금액', '잔액', '상태', '신청일시', '처리일시'];
    }
    
    // 데이터 행
    while ($row = mysqli_fetch_assoc($result)) {
        // 상태 한글 변환
        $status = '';
        switch ($row['status']) {
            case 'completed': $status = '승인'; break;
            case 'cancelled': $status = '취소'; break;
            case 'pending': $status = '보류'; break;
            case 'requested': $status = '대기'; break;
            case 'transferred': $status = '송금완료'; break;
            default: $status = $row['status'];
        }

        $hasTransaction = !empty($row['trans_id']) && $row['actual_amount'] > 0;
        $transactionHistoryId = $row['transaction_history_id'];
        
        // 롤링 수수료 계산
        $totalRollingAmount = 0;
        if ($hasTransaction && $transactionHistoryId) {
            $rollingQuery = "SELECT SUM(rolling_amount) as total_rolling 
                           FROM rolling_history 
                           WHERE original_transaction_id = ?";
            $rollingStmt = mysqli_prepare($con, $rollingQuery);
            mysqli_stmt_bind_param($rollingStmt, "i", $transactionHistoryId);
            mysqli_stmt_execute($rollingStmt);
            $rollingResult = mysqli_stmt_get_result($rollingStmt);
            
            if ($rollingRow = mysqli_fetch_assoc($rollingResult)) {
                $totalRollingAmount = floatval($rollingRow['total_rolling']);
            }
            mysqli_stmt_close($rollingStmt);
        }
        
        // 관리자와 일반사용자별로 다른 데이터 구성
        if ($userGrade === 'admin') {
            // 관리자용 엑셀 행 데이터
            $rowData = [
                $row['id'],
                $row['depositor_name'] ?: '미등록',
                number_format($row['amount']),
                $hasTransaction ? number_format($row['actual_amount']) : '-',
                $hasTransaction ? number_format($row['total_fee_amount']) : '-',
                $hasTransaction ? number_format($row['company_fee_amount']) : '-',
                $hasTransaction ? number_format($totalRollingAmount) : '-',
                $hasTransaction ? number_format($row['balance_after']) : '-',
                $row['store_name'],
                $row['store_user_id'],
                $row['kakao_tid'] ?: '-',
                $status,
                date('Y-m-d H:i:s', strtotime($row['created_at'])),
                $row['updated_at'] ? date('Y-m-d H:i:s', strtotime($row['updated_at'])) : '-'
            ];
        } else {
            // 일반사용자용 엑셀 행 데이터
            $rowData = [
                $row['id'],
                $row['depositor_name'] ?: '미등록',
                number_format($row['amount']),
                $hasTransaction ? number_format($row['actual_amount']) : '-',
                $hasTransaction ? number_format($row['balance_after']) : '-',
                $status,
                date('Y-m-d H:i:s', strtotime($row['created_at'])),
                $row['updated_at'] ? date('Y-m-d H:i:s', strtotime($row['updated_at'])) : '-'
            ];
        }
        
        $excelData[] = $rowData;
    }

    // 엑셀 파일 생성
    $xlsx = Shuchkin\SimpleXLSXGen::fromArray($excelData);
    
    // 파일명
    $filename = '입금내역_' . date('Y-m-d_H-i-s') . '.xlsx';
    
    // 다운로드 헤더 설정
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: max-age=0');
    
    // 파일 출력
    $xlsx->downloadAs($filename);

} catch (Exception $e) {
    error_log("Excel Export Error: " . $e->getMessage());
    
    // 에러 시 간단한 HTML 페이지 출력
    http_response_code(500);
    echo '<!DOCTYPE html>
    <html>
    <head><meta charset="UTF-8"><title>다운로드 오류</title></head>
    <body style="font-family:Arial; padding:50px; text-align:center;">
        <h2>엑셀 다운로드 오류</h2>
        <p style="color:#dc3545;">파일 생성 중 오류가 발생했습니다.</p>
        <p>잠시 후 다시 시도해주세요.</p>
        <button onclick="history.back()" style="padding:10px 20px; background:#007bff; color:white; border:none; border-radius:5px; cursor:pointer;">돌아가기</button>
    </body>
    </html>';
    
} finally {
    if (isset($con)) {
        mysqli_close($con);
    }
}
?>