<?php
// 에러 설정
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// 권한 체크
if (!isset($_SESSION['id'])) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => '접근 권한이 없습니다.']);
    exit;
}

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// PhpSpreadsheet 로드 (만약 사용한다면)
// require_once '../vendor/autoload.php';
// use PhpOffice\PhpSpreadsheet\Spreadsheet;
// use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

// GET 파라미터 받기 (기존 fetch_withdrawals.php와 동일한 로직)
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$storeCode = isset($_GET['store_code']) ? trim($_GET['store_code']) : '';
$status = isset($_GET['status']) ? trim($_GET['status']) : '';
$sortColumn = isset($_GET['sort']) ? trim($_GET['sort']) : 'created_at';
$sortOrder = isset($_GET['order']) ? strtoupper(trim($_GET['order'])) : 'DESC';

// 날짜 파라미터 처리
$startDate = isset($_GET['start_date']) && !empty(trim($_GET['start_date'])) 
    ? trim($_GET['start_date']) 
    : date('Y-m-d', strtotime('-1 day'));

$endDate = isset($_GET['end_date']) && !empty(trim($_GET['end_date'])) 
    ? trim($_GET['end_date']) 
    : date('Y-m-d');

try {
    // 데이터 조회 쿼리 (페이징 없이 모든 데이터)
    $query = "
        SELECT 
            t.id,
            t.user_id,
            t.amount,
            t.status,
            t.created_at,
            t.updated_at,
            t.completed_at,
            m.user_name,
            m.user_id as member_user_id,
            m.code as store_code,
            m.bank,
            m.account_holder,
            m.account_number,
            (t.amount - 1000) as actual_amount
        FROM transfer t
        LEFT JOIN member m ON t.user_id = m.id
        WHERE t.type = 'withdraw'
    ";
    
    $whereConditions = [];
    $params = [];
    $types = "";
    
    // 검색 조건 추가 (기존 로직과 동일)
    if (!empty($search)) {
        $whereConditions[] = "(m.user_name LIKE ? OR m.user_id LIKE ?)";
        $searchTerm = "%{$search}%";
        $params[] = $searchTerm;
        $params[] = $searchTerm;
        $types .= "ss";
    }
    
    if (!empty($startDate)) {
        $whereConditions[] = "DATE(t.created_at) >= ?";
        $params[] = $startDate;
        $types .= "s";
    }
    
    if (!empty($endDate)) {
        $whereConditions[] = "DATE(t.created_at) <= ?";
        $params[] = $endDate;
        $types .= "s";
    }
    
    if (!empty($storeCode)) {
        $whereConditions[] = "m.id = ?";
        $params[] = $storeCode;
        $types .= "i";
    }
    
    if (!empty($status)) {
        $whereConditions[] = "t.status = ?";
        $params[] = $status;
        $types .= "s";
    }
    
    if (!empty($whereConditions)) {
        $query .= " AND " . implode(" AND ", $whereConditions);
    }
    
    // 정렬 추가
    $allowedSortColumns = [
        'id' => 't.id',
        'user_name' => 'm.user_name',
        'member_user_id' => 'm.user_id',
        'amount' => 't.amount',
        'actual_amount' => '(t.amount - 1000)',
        'store_code' => 'm.code',
        'status' => 't.status',
        'created_at' => 't.created_at',
        'completed_at' => 't.completed_at'
    ];
    
    if (array_key_exists($sortColumn, $allowedSortColumns)) {
        $dbSortColumn = $allowedSortColumns[$sortColumn];
        $query .= " ORDER BY {$dbSortColumn} {$sortOrder}";
    }
    
    $stmt = mysqli_prepare($con, $query);
    
    if (!empty($params)) {
        mysqli_stmt_bind_param($stmt, $types, ...$params);
    }
    
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    // CSV 헤더 설정
    $filename = "withdrawals_" . date('Y-m-d_H-i-s') . ".csv";
    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: max-age=0');
    
    // BOM 추가 (Excel에서 한글 깨짐 방지)
    echo "\xEF\xBB\xBF";
    
    // CSV 출력 스트림 생성
    $output = fopen('php://output', 'w');
    
    // CSV 헤더
    $headers = [
        'ID',
        '회원명',
        '회원 ID',
        '신청금액',
        '수수료',
        '실제 출금액',
        '가맹점 코드',
        '은행',
        '계좌번호',
        '예금주',
        '상태',
        '신청일시',
        '처리일시'
    ];
    
    fputcsv($output, $headers);
    
    // 데이터 출력
    while ($row = mysqli_fetch_assoc($result)) {
        // 상태 한글화
        $statusText = '';
        switch ($row['status']) {
            case 'requested': $statusText = '신청'; break;
            case 'pending': $statusText = '대기'; break;
            case 'completed': $statusText = '완료'; break;
            case 'cancelled': $statusText = '취소'; break;
            default: $statusText = $row['status'];
        }
        
        $csvRow = [
            $row['id'],
            $row['user_name'] ?: '-',
            $row['member_user_id'] ?: '-',
            number_format($row['amount']),
            '1,000', // 고정 수수료
            number_format($row['actual_amount']),
            $row['store_code'] ?: '-',
            $row['bank'] ?: '-',
            $row['account_number'] ?: '-',
            $row['account_holder'] ?: '-',
            $statusText,
            $row['created_at'] ? date('Y-m-d H:i:s', strtotime($row['created_at'])) : '-',
            $row['completed_at'] ? date('Y-m-d H:i:s', strtotime($row['completed_at'])) : '-'
        ];
        
        fputcsv($output, $csvRow);
    }
    
    fclose($output);
    mysqli_stmt_close($stmt);

} catch (Exception $e) {
    error_log("Export Withdrawals Excel Error: " . $e->getMessage());
    
    // 에러 시 JSON 응답으로 변경
    header('Content-Type: application/json; charset=UTF-8');
    echo json_encode([
        'success' => false,
        'message' => '엑셀 다운로드 중 오류가 발생했습니다.',
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>