<?php
// 에러 표시 (디버깅용 - 운영환경에서는 주석처리)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크 - 로그인된 사용자만
if (!isset($_SESSION['id'])) {
    http_response_code(403);
    echo json_encode([
        'success' => false, 
        'message' => '접근 권한이 없습니다.',
        'error_code' => 'AUTH_FAILED'
    ]);
    exit;
}

// GET 파라미터 받기
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$bankFilter = isset($_GET['bank_filter']) ? trim($_GET['bank_filter']) : '';
$statusFilter = isset($_GET['status_filter']) ? trim($_GET['status_filter']) : '';
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = isset($_GET['per_page']) ? max(10, min(100, intval($_GET['per_page']))) : 20;

try {
    // 1. 전체 개수 조회 (검색 조건 포함)
    $countQuery = "
        SELECT COUNT(DISTINCT ba.id) as total_count
        FROM bank_account ba
        LEFT JOIN member m ON ba.id = m.bank_account_id
        WHERE 1=1
    ";
    
    $countParams = [];
    $countTypes = "";
    
    // 검색 조건 추가
    if (!empty($search)) {
        $countQuery .= " AND (
            ba.bank LIKE ? OR 
            ba.account_number LIKE ? OR 
            ba.account_holder LIKE ?
        )";
        $searchTerm = "%{$search}%";
        $countParams = [$searchTerm, $searchTerm, $searchTerm];
        $countTypes = "sss";
    }
    
    // 은행 필터
    if (!empty($bankFilter)) {
        $countQuery .= " AND ba.bank = ?";
        $countParams[] = $bankFilter;
        $countTypes .= "s";
    }
    
    $countStmt = mysqli_prepare($con, $countQuery);
    
    if (!$countStmt) {
        throw new Exception("카운트 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    if (!empty($countParams)) {
        mysqli_stmt_bind_param($countStmt, $countTypes, ...$countParams);
    }
    
    mysqli_stmt_execute($countStmt);
    $countResult = mysqli_stmt_get_result($countStmt);
    $totalCount = mysqli_fetch_assoc($countResult)['total_count'];
    mysqli_stmt_close($countStmt);
    
    // 페이징 계산
    $paging = calculatePagination($totalCount, $perPage, $page);
    
    // 2. 실제 데이터 조회
    $dataQuery = "
        SELECT 
            ba.id,
            ba.bank,
            ba.account_number,
            ba.account_holder,
            ba.created_at,
            COUNT(m.id) as member_count
        FROM bank_account ba
        LEFT JOIN member m ON ba.id = m.bank_account_id
        WHERE 1=1
    ";
    
    $dataParams = [];
    $dataTypes = "";
    
    // 검색 조건 추가
    if (!empty($search)) {
        $dataQuery .= " AND (
            ba.bank LIKE ? OR 
            ba.account_number LIKE ? OR 
            ba.account_holder LIKE ?
        )";
        $searchTerm = "%{$search}%";
        $dataParams = [$searchTerm, $searchTerm, $searchTerm];
        $dataTypes = "sss";
    }
    
    // 은행 필터
    if (!empty($bankFilter)) {
        $dataQuery .= " AND ba.bank = ?";
        $dataParams[] = $bankFilter;
        $dataTypes .= "s";
    }
    
    // GROUP BY 추가
    $dataQuery .= " GROUP BY ba.id";
    
    // 정렬 (최신순)
    $dataQuery .= " ORDER BY ba.created_at DESC, ba.id DESC";
    
    // LIMIT과 OFFSET 추가
    $dataQuery .= " LIMIT ? OFFSET ?";
    $dataParams[] = $paging['limit'];
    $dataParams[] = $paging['offset'];
    $dataTypes .= "ii";
    
    $dataStmt = mysqli_prepare($con, $dataQuery);
    
    if (!$dataStmt) {
        throw new Exception("데이터 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    if (!empty($dataParams)) {
        mysqli_stmt_bind_param($dataStmt, $dataTypes, ...$dataParams);
    }
    
    mysqli_stmt_execute($dataStmt);
    $dataResult = mysqli_stmt_get_result($dataStmt);
    
    $accounts = [];
    while ($row = mysqli_fetch_assoc($dataResult)) {
        // created_at 포맷팅
        if ($row['created_at']) {
            $row['created_at'] = date('Y-m-d H:i', strtotime($row['created_at']));
        }
        
        // is_active는 테이블에 없으므로 기본값 설정
        $row['is_active'] = 1;
        
        // bank_name으로 키 변경 (프론트엔드 호환성)
        $row['bank_name'] = $row['bank'];
        unset($row['bank']);
        
        $accounts[] = $row;
    }
    
    mysqli_stmt_close($dataStmt);
    
    // 성공 응답
    echo json_encode([
        'success' => true,
        'account' => $accounts,  // 프론트엔드에서 'accounts'가 아닌 'account'로 받고 있음
        'totalPages' => $paging['total_pages'],
        'currentPage' => $paging['current_page'],
        'totalCount' => $totalCount,
        'perPage' => $perPage,
        'search' => $search,
        'bankFilter' => $bankFilter,
        'statusFilter' => $statusFilter
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 에러 로깅
    error_log("Bank Account Fetch Error: " . $e->getMessage());
    error_log("User: " . $_SESSION['id']);
    
    // 에러 응답
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => '데이터 조회 중 오류가 발생했습니다.',
        'error' => $e->getMessage(),
        'error_code' => 'FETCH_FAILED'
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>