<?php
// fetch_deposits.php - 동적 자동승인 대기시간 적용

// 에러 보고 설정
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

session_start();

// JSON 헤더를 가장 먼저 설정
header('Content-Type: application/json; charset=utf-8');

// 출력 버퍼링 시작
ob_start();

try {
    // DB 연결
    require_once "../inc/db_connect.php";
    
    if (!isset($con)) {
        throw new Exception('DB 연결 객체가 존재하지 않습니다.');
    }

    // DB 연결 상태 확인
    if (mysqli_connect_errno()) {
        throw new Exception('DB 연결 실패: ' . mysqli_connect_error());
    }

    // 세션에서 사용자 정보 확인
    $userGrade = isset($_SESSION['grade']) ? $_SESSION['grade'] : '';
    $sessionId = isset($_SESSION['id']) ? $_SESSION['id'] : '';
    
    // 로그인 체크
    if (empty($userGrade) || empty($sessionId)) {
        throw new Exception('로그인이 필요합니다.');
    }
    
    // ============================================
    // settings 테이블에서 자동승인 대기시간 가져오기
    // ============================================
    $settingsQuery = "SELECT auto_approval_wait_time FROM settings LIMIT 1";
    $settingsResult = mysqli_query($con, $settingsQuery);
    
    $autoApprovalWaitTime = 60; // 기본값 60초
    
    if ($settingsResult && $settingsRow = mysqli_fetch_assoc($settingsResult)) {
        $autoApprovalWaitTime = intval($settingsRow['auto_approval_wait_time']);
        
        // 유효성 검사 (최소 10초, 최대 600초 = 10분)
        if ($autoApprovalWaitTime < 10) {
            $autoApprovalWaitTime = 10;
        } else if ($autoApprovalWaitTime > 600) {
            $autoApprovalWaitTime = 600;
        }
    }
    
    // 파트너인 경우 member 테이블에서 실제 id 찾기
    $memberId = null;
    if ($userGrade !== 'admin') {
        $memberQuery = "SELECT id FROM member WHERE user_id = ? LIMIT 1";
        $memberStmt = mysqli_prepare($con, $memberQuery);
        
        if (!$memberStmt) {
            throw new Exception('Member 쿼리 준비 실패: ' . mysqli_error($con));
        }
        
        mysqli_stmt_bind_param($memberStmt, "s", $sessionId);
        mysqli_stmt_execute($memberStmt);
        $memberResult = mysqli_stmt_get_result($memberStmt);
        
        if ($memberRow = mysqli_fetch_assoc($memberResult)) {
            $memberId = $memberRow['id'];
        } else {
            throw new Exception('사용자 정보를 찾을 수 없습니다.');
        }
        mysqli_stmt_close($memberStmt);
    }

    // GET 파라미터 받기 및 검증
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    $storeCode = isset($_GET['store_code']) && is_numeric($_GET['store_code']) ? intval($_GET['store_code']) : '';
    $kakaoCode = isset($_GET['kakao_code']) ? $_GET['kakao_code'] : '';
    $bankCode = isset($_GET['bank_code']) && is_numeric($_GET['bank_code']) ? intval($_GET['bank_code']) : '';
    $status = isset($_GET['status']) ? $_GET['status'] : '';
    $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
    $limit = 30;
    $offset = ($page - 1) * $limit;

    // 정렬 파라미터 검증
    $sortBy = isset($_GET['sort_by']) ? $_GET['sort_by'] : 'created_at';
    $sortOrder = isset($_GET['sort_order']) ? strtoupper($_GET['sort_order']) : 'DESC';
    
    $allowedSortColumns = [
        'id' => 't.id',
        'updated_at' => 't.updated_at',  
        'depositor_name' => 't.depositor_name',
        'amount' => 't.amount',
        'actual_amount' => 'th.actual_amount',
        'status' => 't.status',
        'created_at' => 't.created_at',
        'store_name' => 'm.user_name'
    ];
    
    if (!isset($allowedSortColumns[$sortBy])) {
        $sortBy = 'created_at';
    }
    
    if (!in_array($sortOrder, ['ASC', 'DESC'])) {
        $sortOrder = 'DESC';
    }

    // 날짜 파라미터 검증
    $startDate = isset($_GET['start_date']) && !empty(trim($_GET['start_date'])) 
        ? trim($_GET['start_date']) 
        : date('Y-m-d', strtotime('-1 day'));

    $endDate = isset($_GET['end_date']) && !empty(trim($_GET['end_date'])) 
        ? trim($_GET['end_date']) 
        : date('Y-m-d');

    // 날짜 형식 검증
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $startDate)) {
        $startDate = date('Y-m-d', strtotime('-1 day'));
    }
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $endDate)) {
        $endDate = date('Y-m-d');
    }

    // WHERE 조건 구성
    $whereConditions = ["t.type = 'deposit'"];
    $params = [];
    $types = '';

    // 파트너는 자기 내역만 조회 가능
    if ($userGrade !== 'admin' && $memberId !== null) {
        $whereConditions[] = "t.user_id = ?";
        $params[] = $memberId;
        $types .= 'i';
    }

    // 검색어 조건
    if (!empty($search)) {
        $search = htmlspecialchars($search, ENT_QUOTES, 'UTF-8');
        
        if (strpos($search, '*') !== false || strpos($search, ' ') !== false) {
            $searchParts = preg_split('/[\*\s]+/', $search);
            
            if (count($searchParts) == 2 && !empty($searchParts[0]) && !empty($searchParts[1])) {
                $whereConditions[] = "(t.depositor_name LIKE ? AND t.depositor_name LIKE ?)";
                $params[] = $searchParts[0] . '%';
                $params[] = '%' . $searchParts[1];
                $types .= 'ss';
            } else {
                $whereConditions[] = "t.depositor_name LIKE ?";
                $params[] = '%' . $search . '%';
                $types .= 's';
            }
        } else if (mb_strlen($search, 'UTF-8') == 2) {
            $firstChar = mb_substr($search, 0, 1, 'UTF-8');
            $lastChar = mb_substr($search, 1, 1, 'UTF-8');
            
            $whereConditions[] = "(t.depositor_name LIKE ? AND t.depositor_name LIKE ?)";
            $params[] = $firstChar . '%';
            $params[] = '%' . $lastChar;
            $types .= 'ss';
        } else {
            $whereConditions[] = "t.depositor_name LIKE ?";
            $params[] = '%' . $search . '%';
            $types .= 's';
        }
    }

    // 날짜 범위 조건
    $whereConditions[] = "DATE(t.created_at) >= ?";
    $params[] = $startDate;
    $types .= 's';

    $whereConditions[] = "DATE(t.created_at) <= ?";
    $params[] = $endDate;
    $types .= 's';

    // 상태 조건
    if (!empty($status) && is_array($status)) {
        $validStatuses = ['requested', 'pending', 'completed', 'cancelled', 'transferred'];
        $filteredStatuses = array_intersect($status, $validStatuses);
        
        if (!empty($filteredStatuses)) {
            $placeholders = str_repeat('?,', count($filteredStatuses) - 1) . '?';
            $whereConditions[] = "t.status IN ($placeholders)";
            foreach ($filteredStatuses as $statusValue) {
                $params[] = $statusValue;
                $types .= 's';
            }
        }
    } else if (!empty($status) && !is_array($status)) {
        $validStatuses = ['requested', 'pending', 'completed', 'cancelled', 'transferred'];
        if (in_array($status, $validStatuses)) {
            $whereConditions[] = "t.status = ?";
            $params[] = $status;
            $types .= 's';
        }
    }

    // 가맹점 코드 조건
    if (!empty($storeCode) && $userGrade === 'admin') {
        $whereConditions[] = "t.user_id = ?";
        $params[] = $storeCode;
        $types .= 'i';
    }

    // 카카오 연결 계정 조건
    if (!empty($kakaoCode) && $userGrade === 'admin') {
        if (is_array($kakaoCode)) {
            $kakaoIds = array_filter($kakaoCode, 'is_numeric');
            if (!empty($kakaoIds)) {
                $placeholders = str_repeat('?,', count($kakaoIds) - 1) . '?';
                $whereConditions[] = "m.kakao_account_id IN ($placeholders)";
                foreach ($kakaoIds as $id) {
                    $params[] = intval($id);
                    $types .= 'i';
                }
            }
        } else if (is_numeric($kakaoCode)) {
            $whereConditions[] = "m.kakao_account_id = ?";
            $params[] = intval($kakaoCode);
            $types .= 'i';
        }
    }

    // 은행 연결 계정 조건
    if (!empty($bankCode) && $userGrade === 'admin') {
        $whereConditions[] = "m.bank_account_id = ?";
        $params[] = $bankCode;
        $types .= 'i';
    }

    $whereClause = implode(' AND ', $whereConditions);

    // 전체 카운트 조회
    $countQuery = "SELECT COUNT(*) as total 
                   FROM transfer t
                   LEFT JOIN member m ON t.user_id = m.id
                   WHERE " . $whereClause;

    $countStmt = mysqli_prepare($con, $countQuery);
    if (!$countStmt) {
        throw new Exception('Count 쿼리 준비 실패: ' . mysqli_error($con));
    }
    
    if (!empty($params)) {
        mysqli_stmt_bind_param($countStmt, $types, ...$params);
    }
    
    if (!mysqli_stmt_execute($countStmt)) {
        throw new Exception('Count 쿼리 실행 실패: ' . mysqli_stmt_error($countStmt));
    }
    
    $countResult = mysqli_stmt_get_result($countStmt);
    $totalCount = mysqli_fetch_assoc($countResult)['total'];
    $totalPages = max(1, ceil($totalCount / $limit));
    mysqli_stmt_close($countStmt);
    
    // 페이지 검증
    if ($page > $totalPages && $totalPages > 0) {
        $page = 1;
        $offset = 0;
    }

    // ============================================
    // 메인 데이터 쿼리 - 동적 INTERVAL 적용
    // ============================================
    $dataQuery = "SELECT 
                t.id,
                t.depositor_name,
                t.amount,
                t.kakao_tid,
                t.status,
                t.created_at,
                t.updated_at,
                t.user_id,
                t.trans_id,
                COALESCE(m.user_name, '미지정') as store_name,
                COALESCE(m.user_id, '-') as store_user_id,
                COALESCE(m.code, '-') as code,
                m.id as member_id,
                COALESCE(th.total_fee_rate, 0) as total_fee_rate,
                COALESCE(th.total_fee_amount, 0) as total_fee_amount,
                COALESCE(th.fee_rate, 0) as company_fee_rate,
                COALESCE(th.fee_amount, 0) as company_fee_amount,
                COALESCE(th.actual_amount, 0) as actual_amount,
                COALESCE(th.balance_after, 0) as balance_after,
                th.id as transaction_history_id,
                th.approval_type as approval_type,
                -- 자동승인 정보 (동적 대기시간 적용)
                aaq.id as auto_approval_id,
                aaq.status as auto_approval_status,
                aaq.queued_at as auto_approval_queued_at,
                TIMESTAMPDIFF(SECOND, NOW(), DATE_ADD(aaq.queued_at, INTERVAL ? SECOND)) as seconds_until_auto_approval
            FROM transfer t
            LEFT JOIN member m ON t.user_id = m.id
            LEFT JOIN transaction_history th ON t.trans_id = th.trans_id AND th.status = 'completed'
            LEFT JOIN auto_approval_queue aaq ON t.id = aaq.transfer_id AND aaq.status = 'waiting'
            WHERE " . $whereClause . "
            ORDER BY 
                CASE WHEN t.status IN ('requested', 'pending') THEN 0 ELSE 1 END,
                CASE WHEN aaq.status = 'waiting' THEN 0 ELSE 1 END,
                COALESCE(t.updated_at, t.created_at) DESC, 
                t.id DESC
            LIMIT ? OFFSET ?";

    // 파라미터에 자동승인 대기시간 추가
    $dataParams = [];
    $dataParams[] = $autoApprovalWaitTime; // INTERVAL 값
    $dataParams = array_merge($dataParams, $params); // 기존 WHERE 파라미터들
    $dataParams[] = $limit;
    $dataParams[] = $offset;
    
    $dataTypes = 'i' . $types . 'ii'; // 맨 앞에 'i' 추가 (INTERVAL 용)

    $dataStmt = mysqli_prepare($con, $dataQuery);
    if (!$dataStmt) {
        throw new Exception('Data 쿼리 준비 실패: ' . mysqli_error($con));
    }
    
    if (!empty($dataParams)) {
        mysqli_stmt_bind_param($dataStmt, $dataTypes, ...$dataParams);
    }
    
    if (!mysqli_stmt_execute($dataStmt)) {
        throw new Exception('Data 쿼리 실행 실패: ' . mysqli_stmt_error($dataStmt));
    }
    
    $dataResult = mysqli_stmt_get_result($dataStmt);
    
    if (!$dataResult) {
        throw new Exception('데이터 결과 가져오기 실패: ' . mysqli_error($con));
    }

    $deposits = [];
    while ($row = mysqli_fetch_assoc($dataResult)) {
        // 상태 변환
        $status = '';
        $originalStatus = $row['status'];
        switch ($row['status']) {
            case 'completed': $status = '승인'; break;
            case 'cancelled': $status = '취소'; break;
            case 'pending': $status = '보류'; break;
            case 'requested': $status = '대기'; break;
            case 'transferred': $status = '송금완료'; break;
            default: $status = $row['status'];
        }

        $hasTransaction = !empty($row['trans_id']) && $row['actual_amount'] > 0;
        $transactionHistoryId = $row['transaction_history_id'];
        
        // 롤링 정보 조회
        $rollingInfo = [];
        $totalRollingAmount = 0;
        
        if ($hasTransaction && $transactionHistoryId) {
            $rollingQuery = "SELECT 
                                rh.rolling_rate,
                                rh.rolling_amount,
                                rh.parent_member_id,
                                COALESCE(pm.user_name, '미지정') as parent_name,
                                COALESCE(pm.user_id, '-') as parent_user_id
                             FROM rolling_history rh
                             LEFT JOIN member pm ON rh.parent_member_id = pm.id
                             WHERE rh.original_transaction_id = ?
                             ORDER BY rh.parent_member_id";
            
            $rollingStmt = mysqli_prepare($con, $rollingQuery);
            if ($rollingStmt) {
                mysqli_stmt_bind_param($rollingStmt, "i", $transactionHistoryId);
                if (mysqli_stmt_execute($rollingStmt)) {
                    $rollingResult = mysqli_stmt_get_result($rollingStmt);
                    
                    while ($rollingRow = mysqli_fetch_assoc($rollingResult)) {
                        $rollingInfo[] = [
                            'rolling_rate' => floatval($rollingRow['rolling_rate']),
                            'rolling_amount' => floatval($rollingRow['rolling_amount']),
                            'parent_member_id' => intval($rollingRow['parent_member_id']),
                            'parent_name' => $rollingRow['parent_name'],
                            'parent_user_id' => $rollingRow['parent_user_id']
                        ];
                        $totalRollingAmount += floatval($rollingRow['rolling_amount']);
                    }
                }
                mysqli_stmt_close($rollingStmt);
            }
        }
        
        $hasRolling = !empty($rollingInfo);

        $deposits[] = [
            'id' => intval($row['id']),
            'name' => $row['depositor_name'] ?: '미등록',
            'amount' => floatval($row['amount']),
            'kakao_tid' => $row['kakao_tid'],
            'store_name' => $row['store_name'],
            'store_user_id' => $row['store_user_id'],
            'code' => $row['code'],
            'member_id' => $row['member_id'] ? intval($row['member_id']) : null,
            'status' => $status,
            'original_status' => $originalStatus,
            'created_at' => $row['created_at'],
            'updated_at' => $row['updated_at'],
            'user_id' => intval($row['user_id']),
            'total_fee_rate' => $hasTransaction ? floatval($row['total_fee_rate']) : null,
            'total_fee_amount' => $hasTransaction ? floatval($row['total_fee_amount']) : null,
            'company_fee_rate' => $hasTransaction ? floatval($row['company_fee_rate']) : null,
            'company_fee_amount' => $hasTransaction ? floatval($row['company_fee_amount']) : null,
            'actual_amount' => $hasTransaction ? floatval($row['actual_amount']) : null,
            'balance_after' => $hasTransaction ? floatval($row['balance_after']) : null,
            'approval_type' => $row['approval_type'] ?: null,
            'rolling_info' => $rollingInfo,
            'total_rolling_amount' => $totalRollingAmount,
            'rolling_levels' => count($rollingInfo),
            'has_transaction' => $hasTransaction,
            'has_rolling' => $hasRolling,
            'rolling_rate' => $hasRolling ? $rollingInfo[0]['rolling_rate'] : null,
            'rolling_amount' => $hasRolling ? $rollingInfo[0]['rolling_amount'] : null,
            'parent_name' => $hasRolling ? $rollingInfo[0]['parent_name'] : null,
            'is_auto_approval_queued' => !empty($row['auto_approval_id']) && $row['auto_approval_status'] === 'waiting',
            'auto_approval_queued_at' => $row['auto_approval_queued_at'],
            'seconds_until_auto_approval' => $row['seconds_until_auto_approval'] > 0 ? $row['seconds_until_auto_approval'] : 0,
        ];
    }
    
    mysqli_stmt_close($dataStmt);

    // 중복 거래 탐지 쿼리
    $duplicateQuery = "SELECT DISTINCT t1.id
    FROM transfer t1
    INNER JOIN transfer t2 ON (
        t1.depositor_name = t2.depositor_name 
        AND t1.amount = t2.amount 
        AND t1.id != t2.id
        AND t1.type = 'deposit'
        AND t2.type = 'deposit'
        AND ABS(TIMESTAMPDIFF(MINUTE, t1.created_at, t2.created_at)) <= 10
    )";

    $duplicateStmt = mysqli_prepare($con, $duplicateQuery);
    $duplicateIds = [];

    if ($duplicateStmt && mysqli_stmt_execute($duplicateStmt)) {
        $duplicateResult = mysqli_stmt_get_result($duplicateStmt);
        while ($dupRow = mysqli_fetch_assoc($duplicateResult)) {
            $duplicateIds[] = intval($dupRow['id']);
        }
        mysqli_stmt_close($duplicateStmt);
    }

    // deposits 배열에 중복 여부 추가
    foreach ($deposits as &$deposit) {
        $deposit['is_duplicate'] = in_array($deposit['id'], $duplicateIds);
    }

    // 버퍼 내용 정리
    ob_clean();

    // 성공 응답 (자동승인 대기시간 포함)
    echo json_encode([
        'success' => true,
        'deposits' => $deposits,
        'totalPages' => $totalPages,
        'currentPage' => $page,
        'totalCount' => $totalCount,
        'userGrade' => $userGrade,
        'autoApprovalWaitTime' => $autoApprovalWaitTime, // 클라이언트에 전달
        'appliedDateRange' => [
            'startDate' => $startDate,
            'endDate' => $endDate
        ],
        'appliedFilters' => [
            'status' => $status,
            'search' => $search,
            'storeCode' => $storeCode,
            'kakaoCode' => $kakaoCode,
            'bankCode' => $bankCode
        ],
        'sortInfo' => [
            'sortBy' => $sortBy,
            'sortOrder' => $sortOrder
        ]
    ], JSON_UNESCAPED_UNICODE);

} catch (Exception $e) {
    ob_clean();
    
    error_log("fetch_deposits.php Error: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => '서버 오류가 발생했습니다: ' . $e->getMessage(),
        'file' => basename($e->getFile()),
        'line' => $e->getLine(),
        'debug' => [
            'userGrade' => $userGrade ?? 'not set',
            'sessionId' => $sessionId ?? 'not set',
            'memberId' => $memberId ?? 'not set'
        ]
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    if (isset($con)) {
        mysqli_close($con);
    }
    
    ob_end_flush();
}
?>