<?php
// 에러 표시 (디버깅용 - 운영환경에서는 주석처리)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크 - 로그인된 사용자만
if (!isset($_SESSION['id'])) {
    http_response_code(403);
    echo json_encode([
        'success' => false, 
        'message' => '접근 권한이 없습니다.',
        'error_code' => 'AUTH_FAILED'
    ]);
    exit;
}

// GET 파라미터 받기
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = isset($_GET['per_page']) ? max(10, min(100, intval($_GET['per_page']))) : 20;

try {
    // 1. 전체 개수 조회 (검색 조건 포함)
    $countQuery = "
        SELECT COUNT(DISTINCT ka.id) as total_count
        FROM kakao_account ka
        LEFT JOIN member m ON ka.id = m.kakao_account_id
        WHERE 1=1
    ";
    
    $countParams = [];
    $countTypes = "";
    
    // 검색 조건 추가 (client_id 제거)
    if (!empty($search)) {
        $countQuery .= " AND (
            ka.kakao_tid LIKE ? OR 
            ka.name LIKE ?
        )";
        $searchTerm = "%{$search}%";
        $countParams = [$searchTerm, $searchTerm];
        $countTypes = "ss";
    }
    
    $countStmt = mysqli_prepare($con, $countQuery);
    
    if (!$countStmt) {
        throw new Exception("카운트 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    if (!empty($countParams)) {
        mysqli_stmt_bind_param($countStmt, $countTypes, ...$countParams);
    }
    
    mysqli_stmt_execute($countStmt);
    $countResult = mysqli_stmt_get_result($countStmt);
    $totalCount = mysqli_fetch_assoc($countResult)['total_count'];
    mysqli_stmt_close($countStmt);
    
    // 페이징 계산
    $paging = calculatePagination($totalCount, $perPage, $page);
    
    // 2. 실제 데이터 조회 (client_id, client_secret, secret_key 제거)
    $dataQuery = "
        SELECT 
            ka.id,
            ka.kakao_tid,
            ka.name,
            ka.qr_link,
            ka.friend_link,
            ka.created_at,
            COUNT(m.id) as member_count
        FROM kakao_account ka
        LEFT JOIN member m ON ka.id = m.kakao_account_id
        WHERE 1=1
    ";
    
    $dataParams = [];
    $dataTypes = "";
    
    // 검색 조건 추가 (client_id 제거)
    if (!empty($search)) {
        $dataQuery .= " AND (
            ka.kakao_tid LIKE ? OR 
            ka.name LIKE ?
        )";
        $searchTerm = "%{$search}%";
        $dataParams = [$searchTerm, $searchTerm];
        $dataTypes = "ss";
    }
    
    // GROUP BY 추가
    $dataQuery .= " GROUP BY ka.id";
    
    // 정렬 (최신순)
    $dataQuery .= " ORDER BY ka.created_at DESC, ka.id DESC";
    
    // LIMIT과 OFFSET 추가
    $dataQuery .= " LIMIT ? OFFSET ?";
    $dataParams[] = $paging['limit'];
    $dataParams[] = $paging['offset'];
    $dataTypes .= "ii";
    
    $dataStmt = mysqli_prepare($con, $dataQuery);
    
    if (!$dataStmt) {
        throw new Exception("데이터 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    if (!empty($dataParams)) {
        mysqli_stmt_bind_param($dataStmt, $dataTypes, ...$dataParams);
    }
    
    mysqli_stmt_execute($dataStmt);
    $dataResult = mysqli_stmt_get_result($dataStmt);
    
    $accounts = [];
    while ($row = mysqli_fetch_assoc($dataResult)) {
        // created_at 포맷팅
        if ($row['created_at']) {
            $row['created_at'] = date('Y-m-d H:i', strtotime($row['created_at']));
        }
        
        // qr_link NULL 또는 빈 값 처리
        if (empty($row['qr_link']) || $row['qr_link'] === null) {
            // 빈 값이면 기본 구조 제공
            $row['qr_link'] = json_encode([
                "10000" => "",
                "30000" => "",
                "50000" => "",
                "100000" => "",
                "200000" => "",
                "300000" => "",
                "500000" => "",
                "1000000" => "",
                "2000000" => ""
            ]);
        } else {
            // JSON 유효성 검사
            $decoded = json_decode($row['qr_link'], true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                // JSON 파싱 실패시 기본 구조 제공
                error_log("Invalid JSON in qr_link for kakao_account id: " . $row['id']);
                $row['qr_link'] = json_encode([
                    "10000" => "",
                    "30000" => "",
                    "50000" => "",
                    "100000" => "",
                    "200000" => "",
                    "300000" => "",
                    "500000" => "",
                    "1000000" => "",
                    "2000000" => ""
                ]);
            }
        }
        $hasDefaultLink = false;
        if (!empty($row['qr_link'])) {
            $decoded = json_decode($row['qr_link'], true);
            if (is_array($decoded) && isset($decoded['default']) && !empty($decoded['default'])) {
                $hasDefaultLink = true;
            }
        }
        $row['has_default_link'] = $hasDefaultLink;
        
        $accounts[] = $row;
    }
    
    mysqli_stmt_close($dataStmt);
    
    // 성공 응답
    echo json_encode([
        'success' => true,
        'accounts' => $accounts,
        'totalPages' => $paging['total_pages'],
        'currentPage' => $paging['current_page'],
        'totalCount' => $totalCount,
        'perPage' => $perPage,
        'search' => $search
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 에러 로깅
    error_log("Kakao Accounts Fetch Error: " . $e->getMessage());
    error_log("User: " . $_SESSION['id']);
    
    // 에러 응답
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => '데이터 조회 중 오류가 발생했습니다.',
        'error' => $e->getMessage(),
        'error_code' => 'FETCH_FAILED'
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>