<?php
// 경고 메시지 숨기기
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', 0);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크 (로그인 사용자만)
if (!isset($_SESSION['id'])) {
    http_response_code(403);
    echo json_encode([
        'success' => false, 
        'message' => '접근 권한이 없습니다.',
        'error_code' => 'AUTH_FAILED'
    ]);
    exit;
}

// GET 파라미터 받기
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$type_filter = isset($_GET['type_filter']) ? trim($_GET['type_filter']) : '';
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$itemsPerPage = 50; // 로그는 많으므로 50개씩

// 날짜 파라미터 처리 - 없으면 기본값(어제~오늘) 설정
$start_date = isset($_GET['start_date']) && !empty(trim($_GET['start_date'])) 
    ? trim($_GET['start_date']) 
    : date('Y-m-d', strtotime('-1 day'));

$end_date = isset($_GET['end_date']) && !empty(trim($_GET['end_date'])) 
    ? trim($_GET['end_date']) 
    : date('Y-m-d');

// 만약 두 날짜가 모두 비어있었다면 기본값 사용
if (empty($_GET['start_date']) && empty($_GET['end_date'])) {
    $start_date = date('Y-m-d', strtotime('-1 day'));
    $end_date = date('Y-m-d');
}

if ($page < 1) $page = 1;
$offset = ($page - 1) * $itemsPerPage;

try {
    // 1. 전체 카운트 조회
    $countQuery = "SELECT COUNT(*) as total FROM all_log l LEFT JOIN member m ON l.user_id = m.id WHERE 1=1";
    $whereConditions = [];
    
    // 검색 조건 구성 - IP와 처리자 정보 추가
    if (!empty($search)) {
        $search_escaped = mysqli_real_escape_string($con, $search);
        $whereConditions[] = "(
            l.user_id LIKE '%$search_escaped%' OR 
            l.type LIKE '%$search_escaped%' OR 
            l.detail LIKE '%$search_escaped%' OR
            l.old_data LIKE '%$search_escaped%' OR
            l.new_data LIKE '%$search_escaped%' OR
            l.ip_address LIKE '%$search_escaped%' OR
            processor.user_id LIKE '%$search_escaped%' OR
            processor.user_name LIKE '%$search_escaped%'
        )";
    }
    
    // 타입 필터
    if (!empty($type_filter)) {
        $type_escaped = mysqli_real_escape_string($con, $type_filter);
        $whereConditions[] = "l.type = '$type_escaped'";
    }
    
    // 날짜 필터 - 항상 적용 (테이블 별칭 사용)
    if (!empty($start_date)) {
        $start_escaped = mysqli_real_escape_string($con, $start_date);
        $whereConditions[] = "DATE(l.created_at) >= '$start_escaped'";
    }
    
    if (!empty($end_date)) {
        $end_escaped = mysqli_real_escape_string($con, $end_date);
        $whereConditions[] = "DATE(l.created_at) <= '$end_escaped'";
    }
    
    // WHERE 절 추가
    if (!empty($whereConditions)) {
        $countQuery .= " AND " . implode(" AND ", $whereConditions);
    }
    
    $countResult = mysqli_query($con, $countQuery);
    if (!$countResult) {
        throw new Exception("Count query failed: " . mysqli_error($con));
    }
    
    $countRow = mysqli_fetch_assoc($countResult);
    $totalCount = $countRow['total'];
    $totalPages = ceil($totalCount / $itemsPerPage);
    
    // 페이지가 총 페이지 수보다 크면 1페이지로
    if ($page > $totalPages && $totalPages > 0) {
        $page = 1;
        $offset = 0;
    }
    
    // 2. 실제 데이터 조회 - JOIN 추가
    $dataQuery = "
        SELECT 
            l.*,
            m.user_name,
            processor.user_id as processor_user_id,
            processor.user_name as processor_name,
            processor.grade as processor_level,
            @row_number := @row_number + 1 AS row_num
        FROM all_log l
        LEFT JOIN member m ON l.user_id = m.id
        LEFT JOIN member processor ON l.user_id = processor.id
        CROSS JOIN (SELECT @row_number := 0) AS t
        WHERE 1=1
    ";
    
    // WHERE 절 추가
    if (!empty($whereConditions)) {
        $dataQuery .= " AND " . implode(" AND ", $whereConditions);
    }
    
    $dataQuery .= " ORDER BY l.id DESC LIMIT $itemsPerPage OFFSET $offset";
    
    $dataResult = mysqli_query($con, $dataQuery);
    if (!$dataResult) {
        throw new Exception("Data query failed: " . mysqli_error($con));
    }
    
    $logs = [];
    $logs_count = 0;
    
    while ($row = mysqli_fetch_assoc($dataResult)) {
        // 날짜 포맷
        if (!empty($row['created_at'])) {
            $row['created_at'] = date('Y-m-d H:i:s', strtotime($row['created_at']));
        }
        
        // NULL 값 처리
        $row['old_data'] = $row['old_data'] ?? '';
        $row['new_data'] = $row['new_data'] ?? '';
        $row['detail'] = $row['detail'] ?? '';
        
        // 사용자 정보 추가
        if (!empty($row['user_name'])) {
            $row['user_id'] = $row['user_id'] . ' (' . $row['user_name'] . ')';
        }
        
        // display_id 계산 (전체에서의 순번)
        $row['display_id'] = $totalCount - $offset - $logs_count;
        $logs_count++;
        
        $logs[] = $row;
    }
    
    // 성공 응답
    $response = [
        'success' => true,
        'logs' => $logs,
        'totalPages' => $totalPages,
        'currentPage' => $page,
        'totalCount' => $totalCount,
        'appliedDateRange' => [
            'start_date' => $start_date,
            'end_date' => $end_date
        ],
        'filters' => [
            'search' => $search,
            'type' => $type_filter,
            'start_date' => $start_date,
            'end_date' => $end_date
        ]
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 에러 로깅
    error_log("Log Fetch Error: " . $e->getMessage());
    
    // 에러 응답
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => '데이터 조회 중 오류가 발생했습니다.',
        'error_code' => 'DB_ERROR',
        'debug' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

// 데이터베이스 연결 종료
if (isset($con)) {
    mysqli_close($con);
}
?>