<?php
// ajax/fetch_manual_transactions.php
header('Content-Type: application/json; charset=UTF-8');

// 세션 시작
session_start();

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 응답 데이터 초기화
$response = [
    'success' => false,
    'message' => '',
    'transactions' => [],
    'summary' => [
        'todayDeposit' => 0,
        'todayWithdraw' => 0,
        'todayNet' => 0,
        'totalBalance' => 0,
        'periodDeposit' => 0,
        'periodWithdraw' => 0,
        'periodNet' => 0,
        'totalCount' => 0,
        'depositCount' => 0,
        'withdrawCount' => 0
    ],
    'currentPage' => 1,
    'totalPages' => 1,
    'totalRecords' => 0
];

try {
    // 세션 체크
    if (!isset($_SESSION['id'])) {
        throw new Exception('로그인이 필요합니다.');
    }
    
    $admin_id = $_SESSION['id'];
    
    // 파라미터 받기
    $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
    $per_page = isset($_GET['per_page']) ? intval($_GET['per_page']) : 20;
    $per_page = min(100, max(10, $per_page)); // 10~100 사이로 제한
    
    // 검색 조건
    $member_user_id = isset($_GET['member_user_id']) ? escapeString($con, $_GET['member_user_id']) : '';
    $type = isset($_GET['type']) ? escapeString($con, $_GET['type']) : '';
    $status = isset($_GET['status']) ? escapeString($con, $_GET['status']) : '';
    $start_date = isset($_GET['start_date']) ? escapeString($con, $_GET['start_date']) : '';
    $end_date = isset($_GET['end_date']) ? escapeString($con, $_GET['end_date']) : '';
    $min_amount = isset($_GET['min_amount']) ? floatval($_GET['min_amount']) : 0;
    $max_amount = isset($_GET['max_amount']) ? floatval($_GET['max_amount']) : 0;
    
    // 날짜 기본값 설정 (오늘과 7일 전)
    if (empty($start_date) && empty($end_date)) {
        $end_date = date('Y-m-d');
        $start_date = date('Y-m-d', strtotime('-7 days'));
    }
    
    // WHERE 조건 생성
    $where_conditions = [];
    
    if (!empty($member_user_id)) {
        $where_conditions[] = "(mt.member_user_id LIKE '%$member_user_id%' OR m.user_name LIKE '%$member_user_id%')";
    }
    
    if (!empty($type) && in_array($type, ['deposit', 'withdraw'])) {
        $where_conditions[] = "mt.type = '$type'";
    }
    
    if (!empty($status) && in_array($status, ['pending', 'completed', 'cancelled'])) {
        $where_conditions[] = "mt.status = '$status'";
    }
    
    if (!empty($start_date)) {
        $where_conditions[] = "DATE(mt.created_at) >= '$start_date'";
    }
    
    if (!empty($end_date)) {
        $where_conditions[] = "DATE(mt.created_at) <= '$end_date'";
    }
    
    if ($min_amount > 0) {
        $where_conditions[] = "mt.amount >= $min_amount";
    }
    
    if ($max_amount > 0) {
        $where_conditions[] = "mt.amount <= $max_amount";
    }
    
    $where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
    
    // ================== 통계 데이터 조회 ==================
    
    // 1. 검색 기간 통계 (필터 적용)
    $period_stats_query = "
        SELECT 
            SUM(CASE WHEN mt.type = 'deposit' AND mt.status = 'completed' THEN mt.amount ELSE 0 END) as period_deposit,
            SUM(CASE WHEN mt.type = 'withdraw' AND mt.status = 'completed' THEN mt.amount ELSE 0 END) as period_withdraw,
            COUNT(CASE WHEN mt.type = 'deposit' THEN 1 END) as deposit_count,
            COUNT(CASE WHEN mt.type = 'withdraw' THEN 1 END) as withdraw_count,
            COUNT(*) as total_count
        FROM manual_transactions mt
        LEFT JOIN member m ON mt.member_user_id COLLATE utf8mb4_general_ci = m.user_id COLLATE utf8mb4_general_ci
        $where_clause
    ";
    
    $period_result = mysqli_query($con, $period_stats_query);
    if ($period_result && $row = mysqli_fetch_assoc($period_result)) {
        $response['summary']['periodDeposit'] = floatval($row['period_deposit'] ?? 0);
        $response['summary']['periodWithdraw'] = floatval($row['period_withdraw'] ?? 0);
        $response['summary']['periodNet'] = $response['summary']['periodDeposit'] - $response['summary']['periodWithdraw'];
        $response['summary']['depositCount'] = intval($row['deposit_count'] ?? 0);
        $response['summary']['withdrawCount'] = intval($row['withdraw_count'] ?? 0);
        $response['summary']['totalCount'] = intval($row['total_count'] ?? 0);
    }
    
    // 2. 오늘 통계 (오늘 날짜가 검색 범위에 포함된 경우만)
    $today = date('Y-m-d');
    $show_today_stats = false;
    
    // 오늘 날짜가 검색 범위에 포함되는지 확인
    if ((!empty($start_date) && !empty($end_date) && $today >= $start_date && $today <= $end_date) ||
        (empty($start_date) && empty($end_date))) {
        $show_today_stats = true;
    }
    
    if ($show_today_stats) {
        // WHERE 조건을 오늘 날짜로 재구성
        $today_where_conditions = [];
        $today_where_conditions[] = "DATE(mt.created_at) = '$today'";
        
        // 다른 필터 조건 추가 (날짜 제외)
        if (!empty($member_user_id)) {
            $today_where_conditions[] = "(mt.member_user_id LIKE '%$member_user_id%' OR m.user_name LIKE '%$member_user_id%')";
        }
        if (!empty($type) && in_array($type, ['deposit', 'withdraw'])) {
            $today_where_conditions[] = "mt.type = '$type'";
        }
        if (!empty($status) && in_array($status, ['pending', 'completed', 'cancelled'])) {
            $today_where_conditions[] = "mt.status = '$status'";
        }
        if ($min_amount > 0) {
            $today_where_conditions[] = "mt.amount >= $min_amount";
        }
        if ($max_amount > 0) {
            $today_where_conditions[] = "mt.amount <= $max_amount";
        }
        
        $today_where_clause = 'WHERE ' . implode(' AND ', $today_where_conditions);
        
        $today_stats_query = "
            SELECT 
                SUM(CASE WHEN mt.type = 'deposit' AND mt.status = 'completed' THEN mt.amount ELSE 0 END) as today_deposit,
                SUM(CASE WHEN mt.type = 'withdraw' AND mt.status = 'completed' THEN mt.amount ELSE 0 END) as today_withdraw
            FROM manual_transactions mt
            LEFT JOIN member m ON mt.member_user_id COLLATE utf8mb4_general_ci = m.user_id COLLATE utf8mb4_general_ci
            $today_where_clause
        ";
        
        $today_result = mysqli_query($con, $today_stats_query);
        if ($today_result && $row = mysqli_fetch_assoc($today_result)) {
            $response['summary']['todayDeposit'] = floatval($row['today_deposit'] ?? 0);
            $response['summary']['todayWithdraw'] = floatval($row['today_withdraw'] ?? 0);
            $response['summary']['todayNet'] = $response['summary']['todayDeposit'] - $response['summary']['todayWithdraw'];
        }
    } else {
        // 오늘 날짜가 검색 범위에 없으면 0으로 설정
        $response['summary']['todayDeposit'] = 0;
        $response['summary']['todayWithdraw'] = 0;
        $response['summary']['todayNet'] = 0;
    }
    
    // 3. 전체 잔액 합계 (모든 가맹점의 현재 잔액 합)
    $total_balance_query = "SELECT SUM(amount) as total_balance FROM member WHERE amount > 0";
    $balance_result = mysqli_query($con, $total_balance_query);
    if ($balance_result && $row = mysqli_fetch_assoc($balance_result)) {
        $response['summary']['totalBalance'] = floatval($row['total_balance'] ?? 0);
    }
    
    // ================== 거래 내역 조회 ==================
    
    // 전체 레코드 수 조회 (COLLATE 명시)
    $count_query = "
        SELECT COUNT(*) as total 
        FROM manual_transactions mt
        LEFT JOIN member m ON mt.member_user_id COLLATE utf8mb4_general_ci = m.user_id COLLATE utf8mb4_general_ci
        $where_clause
    ";
    
    // 전체 레코드 수 조회 부분
    $count_result = mysqli_query($con, $count_query);
    if (!$count_result) {
        throw new Exception('데이터 조회 실패: ' . mysqli_error($con));
    }

    $total_records = 0;
    if ($row = mysqli_fetch_assoc($count_result)) {
        $total_records = intval($row['total']);
    }

    // 페이지네이션 계산 - 수정된 부분
    if ($total_records == 0) {
        // 데이터가 없을 때
        $total_pages = 1;  // 0이 아닌 1로 설정
        $page = 1;
        $offset = 0;
    } else {
        // 데이터가 있을 때
        $total_pages = ceil($total_records / $per_page);
        $page = min(max(1, $page), $total_pages);  // 최소 1, 최대 total_pages
        $offset = max(0, ($page - 1) * $per_page);  // offset은 최소 0
    }
    
    // 거래 내역 조회 (COLLATE 명시)
    $query = "
        SELECT 
            mt.*,
            m.user_name as member_name
        FROM manual_transactions mt
        LEFT JOIN member m ON mt.member_user_id COLLATE utf8mb4_general_ci = m.user_id COLLATE utf8mb4_general_ci
        $where_clause
        ORDER BY mt.created_at DESC, mt.id DESC
        LIMIT $offset, $per_page
    ";
    
    $result = mysqli_query($con, $query);
    if (!$result) {
        throw new Exception('데이터 조회 실패: ' . mysqli_error($con));
    }
    
    $transactions = [];
    while ($row = mysqli_fetch_assoc($result)) {
        // 데이터 정리
        $transaction = [
            'id' => intval($row['id']),
            'user_id' => intval($row['user_id'] ?? 0),
            'member_user_id' => $row['member_user_id'],
            'member_name' => $row['member_name'] ?? '-',
            'admin_id' => $row['admin_id'] ?? 'admin',
            'type' => $row['type'],
            'amount' => floatval($row['amount']),
            'balance_before' => floatval($row['balance_before'] ?? 0),
            'balance_after' => floatval($row['balance_after'] ?? 0),
            'memo' => $row['memo'],
            'status' => $row['status'] ?? 'completed',
            'ip_address' => $row['ip_address'],
            'created_at' => $row['created_at'],
            'updated_at' => $row['updated_at']
        ];
        
        $transactions[] = $transaction;
    }
    
    // 응답 데이터 설정
    $response['success'] = true;
    $response['message'] = '데이터 조회 성공';
    $response['transactions'] = $transactions;
    $response['currentPage'] = $page;
    $response['totalPages'] = $total_pages;
    $response['totalRecords'] = $total_records;
    
    // 추가 통계 정보
    $response['searchInfo'] = [
        'start_date' => $start_date,
        'end_date' => $end_date,
        'filters_applied' => !empty($where_conditions)
    ];
    
} catch (Exception $e) {
    $response['success'] = false;
    $response['message'] = $e->getMessage();
    
    // 에러 로그
    error_log("fetch_manual_transactions.php Error: " . $e->getMessage());
}

// JSON 응답 출력
echo json_encode($response, JSON_UNESCAPED_UNICODE);

// 데이터베이스 연결 종료
if (isset($con)) {
    mysqli_close($con);
}
?>