<?php
// 에러 표시 설정
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크
if (!isset($_SESSION['id'])) {
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => '파트너 권한이 필요합니다.'
    ]);
    exit;
}

$partner_id = $_SESSION['id'];

try {
    // 파트너 정보 조회
    $query = "
        SELECT 
            m.id,
            m.user_id,
            m.user_name,
            m.code,
            m.bank,
            m.account_number,
            m.account_holder,
            m.amount,
            (
                SELECT COALESCE(SUM(amount), 0) 
                FROM transfer 
                WHERE user_id = m.id 
                AND type = 'withdraw' 
                AND status IN ('requested', 'pending')
            ) as pending_withdrawal
        FROM member m
        WHERE m.user_id = ?
        AND m.grade != 'admin' 
    ";
    
    $stmt = mysqli_prepare($con, $query);
    
    if (!$stmt) {
        throw new Exception("쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($stmt, "s", $partner_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if ($row = mysqli_fetch_assoc($result)) {
        // 출금 가능 금액 계산 (잔액 - 대기중인 출금)
        $available_amount = $row['amount'] - $row['pending_withdrawal'];
        
        // 계좌 정보
        $bankInfo = null;
        if ($row['bank'] && $row['account_number'] && $row['account_holder']) {
            $bankInfo = [
                'bank' => $row['bank'],
                'account_number' => $row['account_number'],
                'account_holder' => $row['account_holder']
            ];
        }
        
        // 응답 데이터
        $response = [
            'success' => true,
            'amount' => $available_amount,
            'totalamount' => $row['amount'],
            'pendingWithdrawal' => $row['pending_withdrawal'],
            'bankInfo' => $bankInfo,
            'partnerInfo' => [
                'name' => $row['user_name'],
                'code' => $row['code']
            ]
        ];
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
        
    } else {
        throw new Exception("파트너 정보를 찾을 수 없습니다.");
    }
    
    mysqli_stmt_close($stmt);
    
} catch (Exception $e) {
    error_log("Fetch Partner Info Error: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => '서버 오류가 발생했습니다.',
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>