<?php
header('Content-Type: application/json');

try {
    // session.php 포함 - MySQLi 연결 사용
    include '../inc/session.php';
    
    // MySQLi 연결 확인
    if (!isset($con) || !$con) {
        throw new Exception('데이터베이스 연결이 없습니다.');
    }

    // 로그인 체크 및 현재 사용자 정보 가져오기
    if (!isset($_SESSION['id'])) {
        throw new Exception('로그인이 필요합니다.');
    }

    $current_user_id = $_SESSION['id']; // 현재 로그인한 파트너의 user_id
    
    // 현재 파트너의 정보 조회 (ID 가져오기)
    $partner_query = "SELECT id, level, parents FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $partner_query);
    mysqli_stmt_bind_param($stmt, "s", $current_user_id);
    mysqli_stmt_execute($stmt);
    $partner_result = mysqli_stmt_get_result($stmt);
    
    if (!$partner_result || mysqli_num_rows($partner_result) == 0) {
        throw new Exception('파트너 정보를 찾을 수 없습니다.');
    }
    
    $partner_data = mysqli_fetch_assoc($partner_result);
    $partner_id = $partner_data['id'];
    $partner_level = $partner_data['level'];
    $partner_parents = $partner_data['parents'];

    // 뷰 타입 확인 (tree 또는 list)
    $view_type = isset($_GET['view']) ? $_GET['view'] : 'tree';
    
    // 검색 조건
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    
    // WHERE 절 구성 - 현재 파트너와 그 하위 가맹점들만 조회
    $whereConditions = [];
    
    // 현재 파트너 본인 포함
    $whereConditions[] = "m.id = $partner_id";
    
    // 현재 파트너를 parents에 포함하는 하위 가맹점들
    $whereConditions[] = "JSON_CONTAINS(m.parents, '$partner_id')";
    
    // OR 조건으로 연결
    $parentCondition = "(" . implode(" OR ", $whereConditions) . ")";
    
    $finalWhereConditions = [$parentCondition];
    
    if (!empty($search)) {
        $search_escaped = mysqli_real_escape_string($con, $search);
        
        // 일반 검색 (단계 검색은 파트너용에서 제거)
        $finalWhereConditions[] = "(m.user_name LIKE '%$search_escaped%' 
                                  OR m.user_id LIKE '%$search_escaped%' 
                                  OR m.code LIKE '%$search_escaped%')";
    }
    
    $whereClause = "WHERE " . implode(" AND ", $finalWhereConditions);

    // 파트너 권한으로 조회할 수 있는 데이터만 가져오기
    $sql = "SELECT 
                m.id, 
                m.user_id, 
                m.user_name, 
                m.code,
                m.kakao_account_id,
                ka.name as kakao_account_name,
                ka.kakao_tid,
                m.bank_account_id,
                ba.bank as bank_account_bank,
                ba.account_holder as bank_account_holder,
                ba.account_number as bank_account_number,
                m.bank, 
                m.account_number, 
                m.account_holder, 
                m.rate, 
                m.amount,
                m.grade,
                m.level,
                m.parents,
                DATE_FORMAT(m.created_at, '%Y-%m-%d %H:%i') as created_at,
                DATE_FORMAT(m.updated_at, '%Y-%m-%d %H:%i') as updated_at
            FROM member m
            LEFT JOIN kakao_account ka ON m.kakao_account_id = ka.id
            LEFT JOIN bank_account ba ON m.bank_account_id = ba.id
            $whereClause 
            ORDER BY m.level ASC, m.id ASC";
    
    $result = mysqli_query($con, $sql);
    
    if (!$result) {
        throw new Exception('데이터 조회 쿼리 실행 실패: ' . mysqli_error($con));
    }
    
    $members = [];
    $membersByParent = [];
    $rootMembers = [];
    
    while ($row = mysqli_fetch_assoc($result)) {
        // 데이터 포맷팅
        $row['rate'] = floatval($row['rate']);
        $row['amount'] = $row['amount'] ?: 0;
        $row['level'] = (int)($row['level'] ?? 1);
        
        // parents 데이터 파싱
        if (!empty($row['parents'])) {
            $row['parents_array'] = json_decode($row['parents'], true);
            if (!is_array($row['parents_array'])) {
                $row['parents_array'] = [];
            }
        } else {
            $row['parents_array'] = [];
        }
        
        // 직속 상위 ID 찾기
        $parentId = null;
        if (!empty($row['parents_array'])) {
            // parents 배열의 마지막 요소가 직속 상위
            $parentId = end($row['parents_array']);
        }
        
        // 현재 파트너가 최상위로 표시되도록 처리
        if ($row['id'] == $partner_id) {
            $parentId = null; // 현재 파트너는 트리의 루트가 됨
        }
        
        // 카카오/은행 정보 null 처리
        if (!$row['kakao_account_id']) {
            $row['kakao_account_name'] = null;
            $row['kakao_tid'] = null;
        }
        
        if (!$row['bank_account_id']) {
            $row['bank_account_bank'] = null;
            $row['bank_account_holder'] = null;
            $row['bank_account_number'] = null;
        }
        
        $members[$row['id']] = $row;
        
        // 부모별로 그룹화
        if ($parentId) {
            if (!isset($membersByParent[$parentId])) {
                $membersByParent[$parentId] = [];
            }
            $membersByParent[$parentId][] = $row['id'];
        } else {
            // 최상위 멤버 (현재 파트너)
            $rootMembers[] = $row['id'];
        }
    }
    
    // 트리 구조 생성 함수
    function buildTree($memberId, &$members, &$membersByParent) {
        if (!isset($members[$memberId])) {
            return null;
        }
        
        $node = $members[$memberId];
        $node['children'] = [];
        
        // 자식 노드들 추가
        if (isset($membersByParent[$memberId])) {
            foreach ($membersByParent[$memberId] as $childId) {
                $childNode = buildTree($childId, $members, $membersByParent);
                if ($childNode) {
                    $node['children'][] = $childNode;
                }
            }
        }
        
        return $node;
    }
    
    // 트리 구조 생성 (현재 파트너를 루트로)
    $treeData = [];
    foreach ($rootMembers as $rootId) {
        $treeNode = buildTree($rootId, $members, $membersByParent);
        if ($treeNode) {
            $treeData[] = $treeNode;
        }
    }
    
    // 전체 멤버 수
    $totalCount = count($members);
    
    echo json_encode([
        'success' => true,
        'view_type' => $view_type,
        'tree_data' => $treeData,
        'flat_data' => array_values($members),
        'totalCount' => $totalCount,
        'partner_info' => [
            'user_id' => $current_user_id,
            'id' => $partner_id,
            'level' => $partner_level
        ]
    ]);

} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => '오류: ' . $e->getMessage()
    ]);
}
?>