<?php
// 에러 표시 설정
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크
if (!isset($_SESSION['id'])) {
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => '파트너 권한이 필요합니다.'
    ]);
    exit;
}

$partner_user_id = $_SESSION['id'];  // 이것은 member.user_id 값

// GET 파라미터
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$status = isset($_GET['status']) ? trim($_GET['status']) : '';
$startDate = isset($_GET['start_date']) ? trim($_GET['start_date']) : '';
$endDate = isset($_GET['end_date']) ? trim($_GET['end_date']) : '';
$perPage = 10;

// 날짜 유효성 검사 및 기본값 설정
if (empty($startDate) || empty($endDate)) {
    // 기본값: 오늘 날짜
    $today = date('Y-m-d');
    $startDate = empty($startDate) ? $today : $startDate;
    $endDate = empty($endDate) ? $today : $endDate;
}

// 날짜 형식 검증
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $startDate) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $endDate)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => '올바른 날짜 형식이 아닙니다. (YYYY-MM-DD)'
    ]);
    exit;
}

// 시작 날짜가 종료 날짜보다 늦지 않은지 확인
if (strtotime($startDate) > strtotime($endDate)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => '시작 날짜는 종료 날짜보다 늦을 수 없습니다.'
    ]);
    exit;
}

// 검색 범위를 최대 1년으로 제한
$daysDiff = (strtotime($endDate) - strtotime($startDate)) / (60 * 60 * 24);
if ($daysDiff > 365) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => '검색 기간은 최대 1년까지 가능합니다.'
    ]);
    exit;
}

try {
    // 먼저 user_id로 member.id를 조회
    $getMemberIdQuery = "SELECT id FROM member WHERE user_id = ?";
    $getMemberIdStmt = mysqli_prepare($con, $getMemberIdQuery);
    
    if (!$getMemberIdStmt) {
        throw new Exception("회원 ID 조회 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($getMemberIdStmt, "s", $partner_user_id);
    mysqli_stmt_execute($getMemberIdStmt);
    $getMemberIdResult = mysqli_stmt_get_result($getMemberIdStmt);
    
    if (!$memberRow = mysqli_fetch_assoc($getMemberIdResult)) {
        throw new Exception("회원 정보를 찾을 수 없습니다.");
    }
    
    $partner_id = $memberRow['id'];  // member 테이블의 primary key
    mysqli_stmt_close($getMemberIdStmt);
    
    // 기본 WHERE 조건
    $whereConditions = [
        "t.user_id = ?", 
        "t.type = 'withdraw'",
        "DATE(t.created_at) >= ?",
        "DATE(t.created_at) <= ?"
    ];
    $params = [$partner_id, $startDate, $endDate];
    $types = "iss";
    
    // 상태 조건 추가
    if (!empty($status)) {
        $whereConditions[] = "t.status = ?";
        $params[] = $status;
        $types .= "s";
    }
    
    // WHERE 절 생성
    $whereClause = " WHERE " . implode(" AND ", $whereConditions);
    
    // 1. 전체 개수 조회
    $countQuery = "SELECT COUNT(*) as total_count FROM transfer t" . $whereClause;
    
    $countStmt = mysqli_prepare($con, $countQuery);
    
    if (!$countStmt) {
        throw new Exception("카운트 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($countStmt, $types, ...$params);
    mysqli_stmt_execute($countStmt);
    $countResult = mysqli_stmt_get_result($countStmt);
    $totalCount = mysqli_fetch_assoc($countResult)['total_count'];
    mysqli_stmt_close($countStmt);
    
    // 페이징 계산
    $totalPages = ceil($totalCount / $perPage);
    $offset = ($page - 1) * $perPage;
    
    // 2. 실제 데이터 조회
    $dataQuery = "
        SELECT 
            t.id,
            t.amount,
            t.status,
            t.payment_status,
            t.created_at,
            t.updated_at,
            t.completed_at
        FROM transfer t" . $whereClause . " 
        ORDER BY t.created_at DESC 
        LIMIT ? OFFSET ?";
    
    // 파라미터에 LIMIT, OFFSET 추가
    $params[] = $perPage;
    $params[] = $offset;
    $types .= "ii";
    
    $dataStmt = mysqli_prepare($con, $dataQuery);
    
    if (!$dataStmt) {
        throw new Exception("데이터 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($dataStmt, $types, ...$params);
    mysqli_stmt_execute($dataStmt);
    $dataResult = mysqli_stmt_get_result($dataStmt);
    
    $withdrawals = [];
    while ($row = mysqli_fetch_assoc($dataResult)) {
        // 날짜 포맷팅
        $row['created_at'] = date('Y-m-d H:i:s', strtotime($row['created_at']));
        
        if ($row['completed_at']) {
            $row['completed_at'] = date('Y-m-d H:i:s', strtotime($row['completed_at']));
        }
        
        // 금액을 숫자로 변환
        $row['amount'] = intval($row['amount']);
        
        $withdrawals[] = $row;
    }
    
    mysqli_stmt_close($dataStmt);
    
    // 통계 정보 추가 조회
    $statsQuery = "
        SELECT 
            COUNT(*) as total_requests,
            IFNULL(SUM(CASE WHEN t.status = 'completed' THEN t.amount ELSE 0 END), 0) as total_completed_amount,
            IFNULL(SUM(CASE WHEN t.status = 'requested' THEN t.amount ELSE 0 END), 0) as total_pending_amount,
            COUNT(CASE WHEN t.status = 'completed' THEN 1 END) as completed_count,
            COUNT(CASE WHEN t.status = 'requested' THEN 1 END) as pending_count,
            COUNT(CASE WHEN t.status = 'cancelled' THEN 1 END) as cancelled_count
        FROM transfer t" . $whereClause;
    
    $statsStmt = mysqli_prepare($con, $statsQuery);
    mysqli_stmt_bind_param($statsStmt, $types_without_limit = substr($types, 0, -2), ...array_slice($params, 0, -2));
    mysqli_stmt_execute($statsStmt);
    $statsResult = mysqli_stmt_get_result($statsStmt);
    $stats = mysqli_fetch_assoc($statsResult);
    mysqli_stmt_close($statsStmt);
    
    // 숫자 타입으로 변환
    foreach ($stats as $key => $value) {
        $stats[$key] = intval($value);
    }
    
    // 성공 응답
    echo json_encode([
        'success' => true,
        'withdrawals' => $withdrawals,
        'totalCount' => $totalCount,
        'totalPages' => $totalPages,
        'currentPage' => $page,
        'perPage' => $perPage,
        'searchParams' => [
            'start_date' => $startDate,
            'end_date' => $endDate,
            'status' => $status,
            'days_range' => intval($daysDiff) + 1
        ],
        'stats' => $stats
    ], JSON_UNESCAPED_UNICODE | JSON_NUMERIC_CHECK);
    
} catch (Exception $e) {
    error_log("Fetch Partner Withdrawals Error: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => '데이터 조회 중 오류가 발생했습니다.',
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>