<?php
header('Content-Type: application/json');

try {
    // session.php 포함 - MySQLi 연결 사용
    include '../inc/session.php';
    
    // MySQLi 연결 확인
    if (!isset($con) || !$con) {
        throw new Exception('데이터베이스 연결이 없습니다.');
    }

    // 뷰 타입 확인 (tree 또는 list)
    $view_type = isset($_GET['view']) ? $_GET['view'] : 'tree';
    
    // 검색 조건
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    
    // WHERE 절 구성
    $whereConditions = ["1=1"];
    
    if (!empty($search)) {
        $search_escaped = mysqli_real_escape_string($con, $search);
        
        // 단계 검색 처리
        if (preg_match('/(\d+)단계/', $search, $matches)) {
            $searchLevel = (int)$matches[1];
            $whereConditions[] = "m.level = $searchLevel";
        } else {
            // 일반 검색
            $whereConditions[] = "(m.user_name LIKE '%$search_escaped%' 
                                  OR m.user_id LIKE '%$search_escaped%' 
                                  OR m.code LIKE '%$search_escaped%')";
        }
    }
    
    $whereClause = "WHERE " . implode(" AND ", $whereConditions);

    // 모든 데이터 조회 (트리뷰는 페이지네이션 없음)
    $sql = "SELECT 
                m.id, 
                m.user_id, 
                m.user_name, 
                m.code,
                m.kakao_account_id,
                m.kakao_link_type,
                ka.name as kakao_account_name,
                ka.kakao_tid,
                m.bank_account_id,
                ba.bank as bank_account_bank,
                ba.account_holder as bank_account_holder,
                ba.account_number as bank_account_number,
                m.bank, 
                m.account_number, 
                m.account_holder, 
                m.rate, 
                m.amount,
                m.grade,
                m.level,
                m.parents,
                DATE_FORMAT(m.created_at, '%Y-%m-%d %H:%i') as created_at,
                DATE_FORMAT(m.updated_at, '%Y-%m-%d %H:%i') as updated_at
            FROM member m
            LEFT JOIN kakao_account ka ON m.kakao_account_id = ka.id
            LEFT JOIN bank_account ba ON m.bank_account_id = ba.id
            $whereClause 
            ORDER BY m.level ASC, m.id ASC";
    
    $result = mysqli_query($con, $sql);
    
    if (!$result) {
        throw new Exception('데이터 조회 쿼리 실행 실패: ' . mysqli_error($con));
    }
    
    $members = [];
    $membersByParent = [];
    $rootMembers = [];
    
    while ($row = mysqli_fetch_assoc($result)) {
        // 데이터 포맷팅
        $row['rate'] = floatval($row['rate']);
        $row['amount'] = $row['amount'] ?: 0;
        $row['level'] = (int)($row['level'] ?? 1);
        
        // 등급에 따른 배지 정보 추가
        $gradeBadges = [
            'admin' => [
                'text' => '관리자',
                'class' => 'bg-red-500 text-white'
            ],
            'partner' => [
                'text' => '총판',
                'class' => 'bg-blue-500 text-white'
            ],
            'basic' => [
                'text' => '가맹점',
                'class' => 'bg-green-500 text-white'
            ]
        ];
        
        $row['grade_badge'] = $gradeBadges[$row['grade']] ?? [
            'text' => '가맹점',
            'class' => 'bg-green-500 text-white'
        ];
        
        // 카카오 링크 타입 배지 정보 추가
        if ($row['kakao_account_id'] && $row['kakao_link_type']) {
            $linkTypeBadges = [
                'qr' => [
                    'text' => '코드송금',
                    'class' => 'bg-yellow-500 text-white'
                ],
                'friend' => [
                    'text' => '친구톡',
                    'class' => 'bg-purple-500 text-white'
                ]
            ];
            
            $row['kakao_link_badge'] = $linkTypeBadges[$row['kakao_link_type']] ?? null;
        } else {
            $row['kakao_link_badge'] = null;
        }
        
        // parents 데이터 파싱
        if (!empty($row['parents'])) {
            $row['parents_array'] = json_decode($row['parents'], true);
            if (!is_array($row['parents_array'])) {
                $row['parents_array'] = [];
            }
        } else {
            $row['parents_array'] = [];
        }
        
        // 직속 상위 ID 찾기
        $parentId = null;
        if (!empty($row['parents_array'])) {
            // parents 배열의 마지막 요소가 직속 상위
            $parentId = end($row['parents_array']);
        }
        
        // 카카오/은행 정보 null 처리
        if (!$row['kakao_account_id']) {
            $row['kakao_account_name'] = null;
            $row['kakao_tid'] = null;
            $row['kakao_link_type'] = null;
        }
        
        if (!$row['bank_account_id']) {
            $row['bank_account_bank'] = null;
            $row['bank_account_holder'] = null;
            $row['bank_account_number'] = null;
        }
        
        $members[$row['id']] = $row;
        
        // 부모별로 그룹화
        if ($parentId) {
            if (!isset($membersByParent[$parentId])) {
                $membersByParent[$parentId] = [];
            }
            $membersByParent[$parentId][] = $row['id'];
        } else {
            // 최상위 멤버
            $rootMembers[] = $row['id'];
        }
    }
    
    // 트리 구조 생성 함수
    function buildTree($memberId, &$members, &$membersByParent) {
        if (!isset($members[$memberId])) {
            return null;
        }
        
        $node = $members[$memberId];
        $node['children'] = [];
        
        // 자식 노드들 추가
        if (isset($membersByParent[$memberId])) {
            foreach ($membersByParent[$memberId] as $childId) {
                $childNode = buildTree($childId, $members, $membersByParent);
                if ($childNode) {
                    $node['children'][] = $childNode;
                }
            }
        }
        
        return $node;
    }
    
    // 트리 구조 생성
    $treeData = [];
    foreach ($rootMembers as $rootId) {
        $treeNode = buildTree($rootId, $members, $membersByParent);
        if ($treeNode) {
            $treeData[] = $treeNode;
        }
    }
    
    // 전체 멤버 수
    $totalCount = count($members);
    
    echo json_encode([
        'success' => true,
        'view_type' => $view_type,
        'tree_data' => $treeData,
        'flat_data' => array_values($members),
        'totalCount' => $totalCount
    ]);

} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => '오류: ' . $e->getMessage()
    ]);
}
?>