<?php
header('Content-Type: application/json');

try {
    include '../inc/session.php';
    
    if (!isset($con)) {
        throw new Exception('DB 연결 없음');
    }

    // 파라미터 받기
    $search = $_GET['search'] ?? '';
    $startDate = $_GET['start_date'] ?? date('Y-m-d');
    $endDate = $_GET['end_date'] ?? date('Y-m-d');
    $grade = $_GET['grade'] ?? '';
    $page = intval($_GET['page'] ?? 1);
    $perPage = 20;
    $offset = ($page - 1) * $perPage;
    
    // 정렬 파라미터 추가
    $sortBy = $_GET['sort_by'] ?? 'id';
    $sortOrder = strtoupper($_GET['sort_order'] ?? 'DESC');
    
    // 정렬 필드 검증 (SQL 인젝션 방지)
    $allowedSortFields = [
        'id', 'user_name', 'user_id', 'grade', 'rate', 'amount',
        'total_deposit', 'total_withdraw', 'total_deposit_fee', 
        'total_withdraw_fee', 'total_actual_fee', 'total_rolling_paid',
        'total_rolling_received', 'last_balance'
    ];
    
    if (!in_array($sortBy, $allowedSortFields)) {
        $sortBy = 'id';
    }
    
    if (!in_array($sortOrder, ['ASC', 'DESC'])) {
        $sortOrder = 'DESC';
    }

    // 날짜를 timestamp로 변환
    $startTimestamp = strtotime($startDate . ' 00:00:00');
    $endTimestamp = strtotime($endDate . ' 23:59:59');

    // history 테이블용 기간 조건
    $historyPeriodCondition = "AND h.stat_date >= $startTimestamp AND h.stat_date <= $endTimestamp";

    // 검색 조건 (활성 가맹점용)
    $searchCondition = '';
    if (!empty($search)) {
        $search_escaped = mysqli_real_escape_string($con, $search);
        $searchCondition = "AND (m.user_name LIKE '%$search_escaped%' OR m.user_id LIKE '%$search_escaped%')";
    }

    // 등급 조건 (활성 가맹점용)
    $gradeCondition = '';
    if (!empty($grade)) {
        $grade_escaped = mysqli_real_escape_string($con, $grade);
        $gradeCondition = "AND m.grade = '$grade_escaped'";
    }

    // ===== 1. 활성 가맹점 기본 정보 (테이블용) =====
    $summaryBaseSql = "
        SELECT 
            COUNT(*) as total_stores,
            SUM(amount) as total_balance
        FROM member m
        WHERE m.level IN (1, 2) 
        $searchCondition
        $gradeCondition
    ";
    $summaryBaseResult = mysqli_query($con, $summaryBaseSql);
    $summaryBase = mysqli_fetch_assoc($summaryBaseResult);

    // ===== 2. 전체 history 데이터 (상단 카드용 - 삭제된 가맹점 포함) =====
    if (!empty($search) || !empty($grade)) {
        // 검색이나 등급 필터가 있는 경우
        $cardHistorySql = "
            SELECT 
                SUM(h.total_deposit) as total_deposit,
                SUM(h.total_withdraw - h.total_withdraw_fee) as total_withdraw,
                SUM(h.total_deposit_fee) as total_deposit_fee,
                SUM(h.total_withdraw_fee) as total_withdraw_fee,
                SUM(h.total_actual_fee) as total_actual_fee,
                SUM(h.total_rolling_paid) as total_rolling_paid,
                SUM(h.total_rolling_received) as total_rolling_received
            FROM history h
            INNER JOIN member m ON h.user_id = m.id
            WHERE 1=1
            $historyPeriodCondition
            $searchCondition
            $gradeCondition
        ";
    } else {
        // 검색/필터가 없는 경우
        $cardHistorySql = "
            SELECT 
                SUM(h.total_deposit) as total_deposit,
                SUM(h.total_withdraw - h.total_withdraw_fee) as total_withdraw,
                SUM(h.total_deposit_fee) as total_deposit_fee,
                SUM(h.total_withdraw_fee) as total_withdraw_fee,
                SUM(h.total_actual_fee) as total_actual_fee,
                SUM(h.total_rolling_paid) as total_rolling_paid,
                SUM(h.total_rolling_received) as total_rolling_received
            FROM history h
            WHERE h.stat_date >= $startTimestamp 
            AND h.stat_date <= $endTimestamp
        ";
    }
    
    $cardHistoryResult = mysqli_query($con, $cardHistorySql);
    $cardHistorySum = mysqli_fetch_assoc($cardHistoryResult);

    // ===== 3. 페이지네이션 정보 =====
    $totalRecords = $summaryBase['total_stores'];
    $totalPages = ceil($totalRecords / $perPage);

    // ===== 4. 개별 가맹점 데이터 (테이블용 - 활성 가맹점만) =====
    
    // 정렬 필드가 집계 필드인 경우와 일반 필드인 경우 구분
    $orderByClause = '';
    if (in_array($sortBy, ['total_deposit', 'total_withdraw', 'total_deposit_fee', 
                           'total_withdraw_fee', 'total_actual_fee', 'total_rolling_paid',
                           'total_rolling_received', 'last_balance'])) {
        // 집계 필드로 정렬
        $orderByClause = "ORDER BY $sortBy $sortOrder";
    } elseif (in_array($sortBy, ['user_name', 'user_id', 'grade', 'rate', 'amount'])) {
        // member 테이블 필드로 정렬
        $orderByClause = "ORDER BY m.$sortBy $sortOrder";
    } else {
        // 기본값 (id로 정렬)
        $orderByClause = "ORDER BY m.id $sortOrder";
    }
    
    $membersSql = "
        SELECT 
            m.id,
            m.user_name,
            m.user_id,
            m.grade,
            m.rate,
            m.amount,
            IFNULL(SUM(h.total_deposit), 0) as total_deposit,
            IFNULL(SUM(h.total_withdraw - h.total_withdraw_fee), 0) as total_withdraw,
            IFNULL(SUM(h.total_deposit_fee), 0) as total_deposit_fee,
            IFNULL(SUM(h.total_withdraw_fee), 0) as total_withdraw_fee,
            IFNULL(SUM(h.total_actual_fee), 0) as total_actual_fee,
            IFNULL(SUM(h.total_rolling_paid), 0) as total_rolling_paid,
            IFNULL(SUM(h.total_rolling_received), 0) as total_rolling_received,
            (
                SELECT IFNULL(h2.last_balance, 0)
                FROM history h2 
                WHERE h2.user_id = m.id 
                AND DATE(FROM_UNIXTIME(h2.stat_date)) = '$endDate'
                LIMIT 1
            ) as last_balance
        FROM member m
        LEFT JOIN history h ON m.id = h.user_id
            AND h.stat_date >= $startTimestamp 
            AND h.stat_date <= $endTimestamp
        WHERE m.level IN (1, 2)
        $searchCondition
        $gradeCondition
        GROUP BY m.id
        $orderByClause
        LIMIT $offset, $perPage
    ";

    $membersResult = mysqli_query($con, $membersSql);
    
    if (!$membersResult) {
        throw new Exception('쿼리 실행 실패: ' . mysqli_error($con));
    }

    $members = [];
    $rowNum = $offset + 1;
    
    while ($row = mysqli_fetch_assoc($membersResult)) {
        $members[] = [
            'display_id' => $rowNum++,
            'user_name' => $row['user_name'],
            'user_id' => $row['user_id'],
            'grade' => $row['grade'],
            'rate' => $row['rate'],
            'amount' => intval($row['amount']),
            'total_deposit' => intval($row['total_deposit']),
            'total_withdraw' => intval($row['total_withdraw']),
            'total_deposit_fee' => intval($row['total_deposit_fee']),
            'total_withdraw_fee' => intval($row['total_withdraw_fee']),
            'total_actual_fee' => intval($row['total_actual_fee']),
            'total_rolling_paid' => intval($row['total_rolling_paid']),
            'total_rolling_received' => intval($row['total_rolling_received']),
            'last_balance' => intval($row['last_balance'])
        ];
    }

    // 응답 데이터 준비
    $responseData = [
        'success' => true,
        'summary' => [
            'total_stores' => intval($summaryBase['total_stores']),
            'total_balance' => intval($summaryBase['total_balance'] ?? 0),
            'total_deposit' => intval($cardHistorySum['total_deposit'] ?? 0),
            'total_withdraw' => intval($cardHistorySum['total_withdraw'] ?? 0),
            'total_deposit_fee' => intval($cardHistorySum['total_deposit_fee'] ?? 0),
            'total_withdraw_fee' => intval($cardHistorySum['total_withdraw_fee'] ?? 0),
            'total_actual_fee' => intval($cardHistorySum['total_actual_fee'] ?? 0),
            'total_rolling_paid' => intval($cardHistorySum['total_rolling_paid'] ?? 0),
            'total_rolling_received' => intval($cardHistorySum['total_rolling_received'] ?? 0)
        ],
        'members' => $members,
        'totalPages' => $totalPages,
        'currentPage' => $page,
        'totalRecords' => $totalRecords,
        'dateRange' => [
            'start' => $startDate,
            'end' => $endDate
        ],
        'sortInfo' => [
            'field' => $sortBy,
            'order' => $sortOrder
        ]
    ];

    echo json_encode($responseData);

} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>