<?php
// 에러 표시 설정
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크
if (!isset($_SESSION['id'])) {
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => '접근 권한이 없습니다.'
    ]);
    exit;
}

// GET 파라미터 받기
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$storeCode = isset($_GET['store_code']) ? trim($_GET['store_code']) : '';
$status = isset($_GET['status']) ? trim($_GET['status']) : '';
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = isset($_GET['per_page']) ? max(20, min(100, intval($_GET['per_page']))) : 20;

// 정렬 파라미터 처리
$sortColumn = isset($_GET['sort']) ? trim($_GET['sort']) : 'created_at';
$sortOrder = isset($_GET['order']) ? strtoupper(trim($_GET['order'])) : 'DESC';

// 허용된 정렬 컬럼 확인
$allowedSortColumns = [
    'id' => 't.id',
    'user_name' => 'm.user_name',
    'member_user_id' => 'm.user_id',
    'amount' => 't.amount',
    'actual_amount' => '(t.amount - 1000)', // 실제 출금액 (수수료 차감)
    'store_code' => 'm.code',
    'status' => 't.status',
    'created_at' => 't.created_at',
    'completed_at' => 't.completed_at'
];

// 정렬 컬럼 검증
if (!array_key_exists($sortColumn, $allowedSortColumns)) {
    $sortColumn = 'created_at';
}
$dbSortColumn = $allowedSortColumns[$sortColumn];

// 정렬 순서 검증
if (!in_array($sortOrder, ['ASC', 'DESC'])) {
    $sortOrder = 'DESC';
}

// 날짜 파라미터 처리 - 없으면 기본값(어제~오늘) 설정
$startDate = isset($_GET['start_date']) && !empty(trim($_GET['start_date'])) 
    ? trim($_GET['start_date']) 
    : date('Y-m-d', strtotime('-1 day'));

$endDate = isset($_GET['end_date']) && !empty(trim($_GET['end_date'])) 
    ? trim($_GET['end_date']) 
    : date('Y-m-d');

// 만약 두 날짜가 모두 비어있었다면 기본값 사용
if (empty($_GET['start_date']) && empty($_GET['end_date'])) {
    $startDate = date('Y-m-d', strtotime('-1 day'));
    $endDate = date('Y-m-d');
}

try {
    // 1. 전체 개수 조회 쿼리
    $countQuery = "
        SELECT COUNT(*) as total_count
        FROM transfer t
        LEFT JOIN member m ON t.user_id = m.id
        WHERE t.type = 'withdraw'
    ";
    
    $whereConditions = [];
    $params = [];
    $types = "";
    
    // 검색 조건 추가
    if (!empty($search)) {
        $whereConditions[] = "(m.user_name LIKE ? OR m.user_id LIKE ?)";
        $searchTerm = "%{$search}%";
        $params[] = $searchTerm;
        $params[] = $searchTerm;
        $types .= "ss";
    }
    
    // 날짜 범위 검색 - 항상 적용
    if (!empty($startDate)) {
        $whereConditions[] = "DATE(t.created_at) >= ?";
        $params[] = $startDate;
        $types .= "s";
    }
    
    if (!empty($endDate)) {
        $whereConditions[] = "DATE(t.created_at) <= ?";
        $params[] = $endDate;
        $types .= "s";
    }
    
    // 가맹점 코드 검색
    if (!empty($storeCode)) {
        $whereConditions[] = "m.id = ?";
        $params[] = $storeCode;
        $types .= "i";
    }
    
    // 상태 필터
    if (!empty($status)) {
        $whereConditions[] = "t.status = ?";
        $params[] = $status;
        $types .= "s";
    }
    
    // WHERE 조건 추가
    if (!empty($whereConditions)) {
        $countQuery .= " AND " . implode(" AND ", $whereConditions);
    }
    
    // 카운트 쿼리 실행
    $countStmt = mysqli_prepare($con, $countQuery);
    
    if (!$countStmt) {
        throw new Exception("카운트 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    if (!empty($params)) {
        mysqli_stmt_bind_param($countStmt, $types, ...$params);
    }
    
    mysqli_stmt_execute($countStmt);
    $countResult = mysqli_stmt_get_result($countStmt);
    $totalCount = mysqli_fetch_assoc($countResult)['total_count'];
    mysqli_stmt_close($countStmt);
    
    // 페이징 계산
    $totalPages = ceil($totalCount / $perPage);
    $offset = ($page - 1) * $perPage;
    
    // 2. 실제 데이터 조회 쿼리
    $dataQuery = "
        SELECT 
            t.id,
            t.user_id,
            t.amount,
            t.type,
            t.kakao_tid,
            t.transfer_link,
            t.status,
            t.payment_status,
            t.created_at,
            t.updated_at,
            t.completed_at,
            m.user_name,
            m.user_id as member_user_id,
            m.code as store_code,
            m.bank,
            m.account_holder,
            m.account_number,
            (t.amount - 1000) as actual_amount,
            th.balance_before,
            th.balance_after
        FROM transfer t
        LEFT JOIN member m ON t.user_id = m.id
        LEFT JOIN transaction_history th ON t.id = th.transfer_id AND th.status = 'completed'
        WHERE t.type = 'withdraw'
    ";
    
    // WHERE 조건 추가
    if (!empty($whereConditions)) {
        $dataQuery .= " AND " . implode(" AND ", $whereConditions);
    }
    
    // 정렬 추가
    $dataQuery .= " ORDER BY {$dbSortColumn} {$sortOrder}";
    
    // 페이징 추가
    $dataQuery .= " LIMIT ? OFFSET ?";
    
    // 파라미터에 LIMIT, OFFSET 추가
    $params[] = $perPage;
    $params[] = $offset;
    $types .= "ii";
    
    $dataStmt = mysqli_prepare($con, $dataQuery);
    
    if (!$dataStmt) {
        throw new Exception("데이터 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    if (!empty($params)) {
        mysqli_stmt_bind_param($dataStmt, $types, ...$params);
    }
    
    mysqli_stmt_execute($dataStmt);
    $dataResult = mysqli_stmt_get_result($dataStmt);
    
    $withdrawals = [];
    while ($row = mysqli_fetch_assoc($dataResult)) {
        // 날짜 포맷팅
        if ($row['created_at']) {
            $row['created_at'] = date('Y-m-d H:i:s', strtotime($row['created_at']));
        }
        if ($row['updated_at']) {
            $row['updated_at'] = date('Y-m-d H:i:s', strtotime($row['updated_at']));
        }
        if ($row['completed_at']) {
            $row['completed_at'] = date('Y-m-d H:i:s', strtotime($row['completed_at']));
        }
        
        // 상태 한글화
        $statusText = '';
        switch ($row['status']) {
            case 'requested':
                $statusText = '신청';
                break;
            case 'pending':
                $statusText = '대기';
                break;
            case 'completed':
                $statusText = '완료';
                break;
            case 'cancelled':
                $statusText = '취소';
                break;
            default:
                $statusText = $row['status'];
        }
        $row['status_text'] = $statusText;
        
        $withdrawals[] = $row;
    }
    
    mysqli_stmt_close($dataStmt);
    
    // 성공 응답
    echo json_encode([
        'success' => true,
        'withdrawals' => $withdrawals,
        'totalCount' => $totalCount,
        'totalPages' => $totalPages,
        'currentPage' => $page,
        'perPage' => $perPage,
        'appliedDateRange' => [
            'startDate' => $startDate,
            'endDate' => $endDate
        ],
        'sortInfo' => [
            'column' => $sortColumn,
            'order' => $sortOrder
        ]
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 에러 로깅
    error_log("Fetch Withdrawals Error: " . $e->getMessage());
    
    // 에러 응답
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => '데이터 조회 중 오류가 발생했습니다.',
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>