<?php
// ajax/get_transaction_history.php
header('Content-Type: application/json; charset=UTF-8');

// 세션 시작
session_start();

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 응답 데이터 초기화
$response = [
    'success' => false,
    'message' => '',
    'transactions' => [],
    'balance' => 0,
    'total_deposit' => 0,
    'total_withdraw' => 0,
    'total_deposit_fee' => 0,
    'total_withdraw_fee' => 0,
    'total_pages' => 0,
    'current_page' => 1
];

try {
    // 로그인 체크
    if (!isset($_SESSION['id'])) {
        throw new Exception('로그인이 필요합니다.');
    }

    // 파라미터 검증
    if (!isset($_GET['user_id']) || empty($_GET['user_id'])) {
        throw new Exception('가맹점 ID가 필요합니다.');
    }

    $user_id = mysqli_real_escape_string($con, $_GET['user_id']);
    $type_filter = isset($_GET['type']) ? mysqli_real_escape_string($con, $_GET['type']) : '';
    $default_days = 2;

    // 날짜 파라미터 처리
    $date_to = isset($_GET['date_to']) && !empty($_GET['date_to']) 
        ? mysqli_real_escape_string($con, $_GET['date_to']) 
        : date('Y-m-d');

    $date_from = isset($_GET['date_from']) && !empty($_GET['date_from']) 
        ? mysqli_real_escape_string($con, $_GET['date_from']) 
        : date('Y-m-d', strtotime("-{$default_days} days"));
    
    $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
    $per_page = 20;
    $offset = ($page - 1) * $per_page;

    // 1. 가맹점 정보 조회
    $member_query = "
        SELECT id, user_id, user_name, amount 
        FROM member 
        WHERE user_id = '$user_id'
    ";
    
    $member_result = mysqli_query($con, $member_query);
    
    if (!$member_result || mysqli_num_rows($member_result) == 0) {
        throw new Exception('존재하지 않는 가맹점입니다.');
    }
    
    $member = mysqli_fetch_assoc($member_result);
    $member_id = $member['id'];
    $current_balance = floatval($member['amount']);

    // 2. WHERE 조건 생성
    $th_where_clause = "";
    $mt_where_clause = "";

    // 날짜 필터
    if ($date_from) {
        $th_where_clause .= " AND DATE(th.created_at) >= '$date_from'";
        $mt_where_clause .= " AND DATE(mt.created_at) >= '$date_from'";
    }
    if ($date_to) {
        $th_where_clause .= " AND DATE(th.created_at) <= '$date_to'";
        $mt_where_clause .= " AND DATE(mt.created_at) <= '$date_to'";
    }
    
    // 유형 필터
    if ($type_filter === 'deposit') {
        $th_where_clause .= " AND th.type = 'deposit'";
        $mt_where_clause .= " AND mt.type = 'deposit'";
    }
    elseif ($type_filter === 'withdraw') {
        $th_where_clause .= " AND th.type = 'withdraw'";
        $mt_where_clause .= " AND mt.type = 'withdraw'";
    }

    // 3. 통합 거래내역 조회
    $transactions_query = "
        (
            -- transaction_history 테이블 (일반 입금/출금) + transfer 테이블 조인
            SELECT 
                'normal' as source,
                th.id as trans_id,
                th.type as type,
                th.actual_amount as amount,
                th.balance_before,
                th.balance_after,
                '' as memo,
                th.status as status,
                th.created_at,
                CASE 
                    WHEN th.type = 'deposit' THEN '입금'
                    WHEN th.type = 'withdraw' THEN '출금'
                END as type_text,
                'SYSTEM' as admin_name,
                COALESCE(t.depositor_name, '') as depositor_name,
                CASE 
                    WHEN t.updated_at IS NOT NULL THEN t.updated_at 
                    ELSE th.created_at 
                END as processed_at
            FROM transaction_history th
            LEFT JOIN transfer t ON th.transfer_id = t.id
            WHERE th.user_id = $member_id 
                AND th.status = 'completed'
                $th_where_clause
        )
        UNION ALL
        (
            -- manual_transactions 테이블 (수동 지급/회수)
            SELECT 
                'manual' as source,
                mt.id as trans_id,
                mt.type as type,
                mt.amount,
                mt.balance_before,
                mt.balance_after,
                IFNULL(mt.memo, '') as memo,
                mt.status as status,
                mt.created_at,
                CASE 
                    WHEN mt.type = 'deposit' THEN '수동입금'
                    WHEN mt.type = 'withdraw' THEN '수동출금'
                END as type_text,
                mt.admin_id as admin_name,
                '-' as depositor_name,
                mt.created_at as processed_at
            FROM manual_transactions mt
            WHERE mt.member_user_id = '$user_id'
                AND mt.status = 'completed'
                $mt_where_clause
        )
        ORDER BY created_at DESC
        LIMIT $offset, $per_page
    ";
    
    $transactions_result = mysqli_query($con, $transactions_query);
    
    if (!$transactions_result) {
        throw new Exception('거래내역 조회 실패: ' . mysqli_error($con));
    }
    
    // 거래내역 데이터 정리
    $transactions = [];
    while ($row = mysqli_fetch_assoc($transactions_result)) {
        $transactions[] = [
            'id' => $row['trans_id'],
            'source' => $row['source'],
            'type' => $row['type'],
            'type_text' => $row['type_text'],
            'amount' => floatval($row['amount']),
            'balance_before' => floatval($row['balance_before']),
            'balance_after' => floatval($row['balance_after']),
            'memo' => $row['memo'] ?: '-',
            'status' => $row['status'],
            'admin_name' => $row['admin_name'],
            'depositor_name' => $row['depositor_name'],
            'processed_at' => $row['processed_at'] ? date('Y-m-d H:i:s', strtotime($row['processed_at'])) : '-',
            'created_at' => date('Y-m-d H:i:s', strtotime($row['created_at']))
        ];
    }

    // 4. 전체 건수 조회 (페이지네이션용)
    $count_query = "
        SELECT COUNT(*) as total FROM (
            (
                SELECT th.id
                FROM transaction_history th
                WHERE th.user_id = $member_id 
                    AND th.status = 'completed'
                    $th_where_clause
            )
            UNION ALL
            (
                SELECT mt.id
                FROM manual_transactions mt
                WHERE mt.member_user_id = '$user_id'
                    AND mt.status = 'completed'
                    $mt_where_clause
            )
        ) as combined
    ";
    
    $count_result = mysqli_query($con, $count_query);
    $total_count = mysqli_fetch_assoc($count_result)['total'];
    $total_pages = ceil($total_count / $per_page);

    // 5. 날짜 범위에 맞는 통계 조회 - 수정된 부분
    // WHERE 조건을 alias 없이 다시 생성
    $date_where_th = "";
    $date_where_mt = "";
    
    if ($date_from) {
        $date_where_th .= " AND DATE(created_at) >= '$date_from'";
        $date_where_mt .= " AND DATE(created_at) >= '$date_from'";
    }
    if ($date_to) {
        $date_where_th .= " AND DATE(created_at) <= '$date_to'";
        $date_where_mt .= " AND DATE(created_at) <= '$date_to'";
    }

    // 총 입금액 (실제 입금액)
    $deposit_query = "
        SELECT 
            COALESCE(SUM(amount), 0) as total 
        FROM (
            SELECT actual_amount as amount 
            FROM transaction_history 
            WHERE user_id = $member_id 
                AND type = 'deposit' 
                AND status = 'completed'
                $date_where_th
            UNION ALL
            SELECT amount 
            FROM manual_transactions 
            WHERE member_user_id = '$user_id' 
                AND type = 'deposit' 
                AND status = 'completed'
                $date_where_mt
        ) as deposits
    ";
    $deposit_result = mysqli_query($con, $deposit_query);
    $total_deposit = floatval(mysqli_fetch_assoc($deposit_result)['total']);

    // 총 출금액 (actual_amount 사용)
    $withdraw_query = "
        SELECT 
            COALESCE(SUM(amount), 0) as total 
        FROM (
            SELECT actual_amount as amount 
            FROM transaction_history 
            WHERE user_id = $member_id 
                AND type = 'withdraw' 
                AND status = 'completed'
                $date_where_th
            UNION ALL
            SELECT amount 
            FROM manual_transactions 
            WHERE member_user_id = '$user_id' 
                AND type = 'withdraw' 
                AND status = 'completed'
                $date_where_mt
        ) as withdrawals
    ";
    $withdraw_result = mysqli_query($con, $withdraw_query);
    $total_withdraw = floatval(mysqli_fetch_assoc($withdraw_result)['total']);

    // 입금 수수료 총합
    $deposit_fee_query = "
        SELECT 
            COALESCE(SUM(fee_amount), 0) as total_fee
        FROM transaction_history 
        WHERE user_id = $member_id 
            AND type = 'deposit' 
            AND status = 'completed'
            AND fee_amount > 0
            $date_where_th
    ";
    $deposit_fee_result = mysqli_query($con, $deposit_fee_query);
    $total_deposit_fee = floatval(mysqli_fetch_assoc($deposit_fee_result)['total_fee']);

    // 출금 수수료 총합
    $withdraw_fee_query = "
        SELECT 
            COALESCE(SUM(fee_amount), 0) as total_fee
        FROM transaction_history 
        WHERE user_id = $member_id 
            AND type = 'withdraw' 
            AND status = 'completed'
            AND fee_amount > 0
            $date_where_th
    ";
    $withdraw_fee_result = mysqli_query($con, $withdraw_fee_query);
    $total_withdraw_fee = floatval(mysqli_fetch_assoc($withdraw_fee_result)['total_fee']);

    $response['success'] = true;
    $response['transactions'] = $transactions;
    $response['balance'] = $current_balance;
    $response['total_deposit'] = $total_deposit;
    $response['total_withdraw'] = $total_withdraw;
    $response['total_deposit_fee'] = $total_deposit_fee;
    $response['total_withdraw_fee'] = $total_withdraw_fee;
    $response['total_pages'] = $total_pages;
    $response['current_page'] = $page;
    $response['total_count'] = $total_count;

} catch (Exception $e) {
    $response['success'] = false;
    $response['message'] = $e->getMessage();
    error_log("get_transaction_history.php Error: " . $e->getMessage());
}

echo json_encode($response, JSON_UNESCAPED_UNICODE);

if (isset($con)) {
    mysqli_close($con);
}
?>