<?php
// 에러 표시 설정 (디버깅용)
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 트랜잭션 ID 생성 함수
function generateTransactionId() {
    return 'TXN' . date('YmdHis') . sprintf('%06d', mt_rand(0, 999999));
}

// JSON 응답 함수
function jsonResponse($success, $message, $data = null, $httpCode = 200) {
    http_response_code($httpCode);
    $response = [
        'success' => $success,
        'message' => $message
    ];
    if ($data !== null) {
        $response['data'] = $data;
    }
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit;
}

// 권한 체크 - 관리자만 접근 가능
if (!isset($_SESSION['id'])) {
    jsonResponse(false, '로그인이 필요합니다.', null, 403);
}

// 관리자 권한 확인
$adminCheckQuery = "SELECT id, grade FROM member WHERE user_id = ?";
$adminCheckStmt = mysqli_prepare($con, $adminCheckQuery);
mysqli_stmt_bind_param($adminCheckStmt, "s", $_SESSION['id']);
mysqli_stmt_execute($adminCheckStmt);
$adminResult = mysqli_stmt_get_result($adminCheckStmt);
$adminData = mysqli_fetch_assoc($adminResult);
mysqli_stmt_close($adminCheckStmt);

if (!$adminData || $adminData['grade'] !== 'admin') {
    jsonResponse(false, '관리자 권한이 필요합니다.', null, 403);
}

$adminMemberId = $adminData['id'];
$adminUserId = $_SESSION['id'];

// POST 데이터 받기
$input = json_decode(file_get_contents('php://input'), true);

if (!isset($input['id']) || !isset($input['action'])) {
    jsonResponse(false, '필수 파라미터가 누락되었습니다.');
}

$transferId = intval($input['id']);
$action = $input['action'];

// 허용된 액션 확인
$allowedActions = ['requested', 'completed', 'cancelled', 'transferred'];
if (!in_array($action, $allowedActions)) {
    jsonResponse(false, '유효하지 않은 액션입니다.');
}

try {
    // autocommit 비활성화 (중요!)
    mysqli_autocommit($con, FALSE);
    
    // 트랜잭션 시작 - READ WRITE 모드로 명시
    mysqli_begin_transaction($con, MYSQLI_TRANS_START_READ_WRITE);
    
    // 1. 현재 transfer 정보 및 사용자 정보 조회 (FOR UPDATE로 잠금)
    $checkQuery = "SELECT t.id, t.user_id, t.amount, t.status, t.type, t.created_at, t.trans_id,
                          m.amount as current_balance, m.user_name, m.user_id as member_user_id, m.rate,
                          m.parents
                   FROM transfer t 
                   JOIN member m ON t.user_id = m.id 
                   WHERE t.id = ? AND t.type = 'deposit'
                   FOR UPDATE";
    
    $checkStmt = mysqli_prepare($con, $checkQuery);
    mysqli_stmt_bind_param($checkStmt, "i", $transferId);
    mysqli_stmt_execute($checkStmt);
    $result = mysqli_stmt_get_result($checkStmt);
    
    if (!$transfer = mysqli_fetch_assoc($result)) {
        throw new Exception('입금 신청을 찾을 수 없습니다.');
    }
    mysqli_stmt_close($checkStmt);
    
    $oldStatus = $transfer['status'];
    $memberId = $transfer['user_id'];
    $memberUserId = $transfer['member_user_id'];
    $depositAmount = $transfer['amount'];
    $currentBalance = $transfer['current_balance'];
    $userName = $transfer['user_name'];
    $memberRate = $transfer['rate'] ?? 0;
    $existingTransId = $transfer['trans_id'] ?? null;
    $parentsJson = $transfer['parents'] ?? '[]';
    
    // 상태 체크
    if ($oldStatus === $action) {
        throw new Exception('이미 해당 상태입니다.');
    }
    
    // 중복 처리 방지 체크
    if ($action === 'completed') {
        // trans_id가 이미 있는지 확인
        if (!empty($existingTransId)) {
            // transaction_history에서도 확인
            $historyCheckQuery = "SELECT id FROM transaction_history WHERE trans_id = ? OR transfer_id = ?";
            $historyCheckStmt = mysqli_prepare($con, $historyCheckQuery);
            mysqli_stmt_bind_param($historyCheckStmt, "si", $existingTransId, $transferId);
            mysqli_stmt_execute($historyCheckStmt);
            $historyCheckResult = mysqli_stmt_get_result($historyCheckStmt);
            
            if (mysqli_num_rows($historyCheckResult) > 0) {
                mysqli_stmt_close($historyCheckStmt);
                throw new Exception('이미 처리된 거래입니다. (Transaction ID: ' . $existingTransId . ')');
            }
            mysqli_stmt_close($historyCheckStmt);
        }
        
        // transfer_id로도 중복 체크
        $dupCheckQuery = "SELECT id FROM transaction_history WHERE transfer_id = ?";
        $dupCheckStmt = mysqli_prepare($con, $dupCheckQuery);
        mysqli_stmt_bind_param($dupCheckStmt, "i", $transferId);
        mysqli_stmt_execute($dupCheckStmt);
        $dupResult = mysqli_stmt_get_result($dupCheckStmt);
        
        if (mysqli_num_rows($dupResult) > 0) {
            mysqli_stmt_close($dupCheckStmt);
            throw new Exception('이미 처리 내역이 존재하는 거래입니다.');
        }
        mysqli_stmt_close($dupCheckStmt);
    }
    
    // 변수 초기화
    $newBalance = null;
    $actualAmount = null;
    $totalFeeAmount = null;
    $companyFeeAmount = null;
    $totalRollingAmount = 0;
    $transId = null;
    $transactionHistoryId = null;
    
    // ====== 입금 승인 처리 ======
    if ($action === 'completed' && $oldStatus !== 'completed') {
        // 트랜잭션 ID 생성
        $transId = generateTransactionId();
        
        // 총 수수료 계산
        $totalFeeRate = $memberRate;
        $totalFeeAmount = round($depositAmount * ($totalFeeRate / 100), 2);
        $actualAmount = $depositAmount - $totalFeeAmount;
        
        // member 테이블 잔액 증가 (FOR UPDATE로 잠금)
        $lockMemberQuery = "SELECT id, amount, callback_url FROM member WHERE id = ? FOR UPDATE";
        $lockMemberStmt = mysqli_prepare($con, $lockMemberQuery);
        mysqli_stmt_bind_param($lockMemberStmt, "i", $memberId);
        mysqli_stmt_execute($lockMemberStmt);
        $lockResult = mysqli_stmt_get_result($lockMemberStmt);
        $lockedMember = mysqli_fetch_assoc($lockResult);
        mysqli_stmt_close($lockMemberStmt);
        
        $currentBalance = $lockedMember['amount'];
        $callbackUrl = $lockedMember['callback_url']; // 콜백 URL 미리 저장
        
        $balanceQuery = "UPDATE member SET amount = amount + ? WHERE id = ?";
        $balanceStmt = mysqli_prepare($con, $balanceQuery);
        mysqli_stmt_bind_param($balanceStmt, "di", $actualAmount, $memberId);
        
        if (!mysqli_stmt_execute($balanceStmt)) {
            throw new Exception('잔액 업데이트 실패: ' . mysqli_error($con));
        }
        mysqli_stmt_close($balanceStmt);
        
        $newBalance = $currentBalance + $actualAmount;
        
        // ===== 다단계 롤링 처리 =====
        $rollingDetails = [];
        $parentsArray = json_decode($parentsJson, true);
        if (!is_array($parentsArray)) {
            $parentsArray = [];
        }
        
        if (count($parentsArray) > 0) {
            $currentMemberId = $memberId;
            $currentRate = $memberRate;
            $reversedParents = array_reverse($parentsArray);
            
            foreach ($reversedParents as $parentId) {
                // 상위 회원 정보 조회 (FOR UPDATE로 잠금)
                $parentInfoQuery = "SELECT id, rate, user_name, user_id, amount FROM member WHERE id = ? FOR UPDATE";
                $parentInfoStmt = mysqli_prepare($con, $parentInfoQuery);
                mysqli_stmt_bind_param($parentInfoStmt, "i", $parentId);
                mysqli_stmt_execute($parentInfoStmt);
                $parentInfoResult = mysqli_stmt_get_result($parentInfoStmt);
                
                if ($parentInfo = mysqli_fetch_assoc($parentInfoResult)) {
                    $parentRate = floatval($parentInfo['rate']);
                    $parentName = $parentInfo['user_name'];
                    $parentUserId = $parentInfo['user_id'];
                    
                    if ($currentRate > $parentRate) {
                        $rollingRate = $currentRate - $parentRate;
                        $rollingAmount = round($depositAmount * ($rollingRate / 100), 2);
                        
                        if ($rollingAmount > 0) {
                            // 상위 회원 잔액 증가
                            $parentBalanceQuery = "UPDATE member SET amount = amount + ? WHERE id = ?";
                            $parentBalanceStmt = mysqli_prepare($con, $parentBalanceQuery);
                            mysqli_stmt_bind_param($parentBalanceStmt, "di", $rollingAmount, $parentId);
                            
                            if (!mysqli_stmt_execute($parentBalanceStmt)) {
                                throw new Exception("상위 회원({$parentName}) 잔액 업데이트 실패");
                            }
                            mysqli_stmt_close($parentBalanceStmt);
                            
                            $rollingDetails[] = [
                                'parent_member_id' => $parentId,
                                'parent_name' => $parentName,
                                'parent_user_id' => $parentUserId,
                                'child_member_id' => $currentMemberId,
                                'rolling_rate' => $rollingRate,
                                'rolling_amount' => $rollingAmount,
                                'current_rate' => $currentRate,
                                'parent_rate' => $parentRate
                            ];
                            
                            $totalRollingAmount += $rollingAmount;
                        }
                    }
                    
                    $currentMemberId = $parentId;
                    $currentRate = $parentRate;
                }
                mysqli_stmt_close($parentInfoStmt);
            }
        }
        
        // 회사 수수료 계산
        $companyFeeAmount = $totalFeeAmount - $totalRollingAmount;
        $companyFeeRate = ($depositAmount > 0) ? ($companyFeeAmount / $depositAmount) * 100 : 0;
        
        // transaction_history 기록
        $historyQuery = "INSERT INTO transaction_history 
                        (trans_id, transfer_id, user_id, type, requested_amount, 
                        total_fee_rate, total_fee_amount, fee_rate, fee_amount, 
                        actual_amount, balance_before, balance_after, status, 
                        processed_by, created_at) 
                        VALUES (?, ?, ?, 'deposit', ?, ?, ?, ?, ?, ?, ?, ?, 'completed', ?, NOW())";
        $historyStmt = mysqli_prepare($con, $historyQuery);
        mysqli_stmt_bind_param($historyStmt, "siiddddddddi", 
            $transId, $transferId, $memberId, $depositAmount, 
            $totalFeeRate, $totalFeeAmount, $companyFeeRate, $companyFeeAmount,
            $actualAmount, $currentBalance, $newBalance, $adminMemberId
        );
        
        if (!mysqli_stmt_execute($historyStmt)) {
            throw new Exception('트랜잭션 내역 저장 실패: ' . mysqli_error($con));
        }
        $transactionHistoryId = mysqli_insert_id($con);
        mysqli_stmt_close($historyStmt);
        
        // rolling_history 저장
        if (!empty($rollingDetails)) {
            foreach ($rollingDetails as $rollingDetail) {
                $rollingHistoryQuery = "INSERT INTO rolling_history 
                                    (parent_member_id, child_member_id, rolling_rate, rolling_amount, 
                                    original_transaction_id, created_at) 
                                    VALUES (?, ?, ?, ?, ?, NOW())";
                $rollingHistoryStmt = mysqli_prepare($con, $rollingHistoryQuery);
                mysqli_stmt_bind_param($rollingHistoryStmt, "iiddi", 
                    $rollingDetail['parent_member_id'], 
                    $rollingDetail['child_member_id'],
                    $rollingDetail['rolling_rate'], 
                    $rollingDetail['rolling_amount'], 
                    $transactionHistoryId
                );
                
                if (!mysqli_stmt_execute($rollingHistoryStmt)) {
                    throw new Exception('롤링 내역 저장 실패');
                }
                mysqli_stmt_close($rollingHistoryStmt);
            }
        }
        
        // transfer 상태 업데이트
        $updateQuery = "UPDATE transfer SET 
                    status = ?, 
                    trans_id = ?, 
                    updated_at = NOW() 
                    WHERE id = ?";
        $updateStmt = mysqli_prepare($con, $updateQuery);
        mysqli_stmt_bind_param($updateStmt, "ssi", $action, $transId, $transferId);
        
        if (!mysqli_stmt_execute($updateStmt)) {
            throw new Exception('상태 업데이트 실패: ' . mysqli_error($con));
        }
        mysqli_stmt_close($updateStmt);
        
        // 로그 기록
        if (function_exists('writeLog')) {
            $detail = "입금 승인 완료 - Trans ID: {$transId}, Transfer ID: {$transferId}, " .
                    "사용자: {$userName}({$memberUserId}), 요청액: " . number_format($depositAmount) . "원, " .
                    "수수료: " . number_format($totalFeeAmount) . "원, 실지급: " . number_format($actualAmount) . "원";
            writeLog($con, 'DEPOSIT_APPROVED', null, ['trans_id' => $transId, 'transfer_id' => $transferId], 1, $detail, $adminMemberId);
        }
    }

    // 트랜잭션 커밋
    // mysqli_commit($con);

    // ===== 트랜잭션 커밋 후 콜백 호출 =====
    // 승인 처리가 완료된 후에만 콜백 호출
    if ($action === 'completed' && isset($transId) && isset($callbackUrl) && !empty($callbackUrl)) {
        // 콜백 호출을 위한 데이터 준비
        $callbackData = [
            'trans_id' => $transId,
            'transfer_id' => $transferId,
            'user_id' => $memberUserId,
            'amount' => $depositAmount,
            'actual_amount' => $actualAmount,
            'fee_amount' => $totalFeeAmount,
            'status' => 'completed'
        ];
        
        // 로그 정보 준비
        $logInfo = [
            'trans_id' => $transId,
            'transfer_id' => $transferId,
            'member_id' => $memberId
        ];
        
        // 콜백 URL 호출 (0은 성공 코드)
        $callbackResult = callWebhook($con, $callbackUrl, 0, $callbackData, $logInfo, $adminMemberId);
        
        // 콜백 결과 로깅
        if (!$callbackResult['success']) {
            error_log("콜백 호출 완료 (실패) - Transfer ID: {$transferId}, URL: {$callbackUrl}, Message: " . $callbackResult['message']);
        } else {
            error_log("콜백 호출 완료 (성공) - Transfer ID: {$transferId}, URL: {$callbackUrl}, HTTP Code: " . $callbackResult['http_code']);
        }
    }
    
    // ====== 입금 취소 처리 ======
    else if ($action === 'cancelled' && $oldStatus === 'completed') {
        // trans_id와 transaction_history 조회
        if (empty($existingTransId)) {
            throw new Exception('취소할 거래 정보를 찾을 수 없습니다.');
        }
        
        $getHistoryQuery = "SELECT * FROM transaction_history WHERE trans_id = ?";
        $getHistoryStmt = mysqli_prepare($con, $getHistoryQuery);
        mysqli_stmt_bind_param($getHistoryStmt, "s", $existingTransId);
        mysqli_stmt_execute($getHistoryStmt);
        $historyResult = mysqli_stmt_get_result($getHistoryStmt);
        $historyData = mysqli_fetch_assoc($historyResult);
        mysqli_stmt_close($getHistoryStmt);
        
        if (!$historyData) {
            throw new Exception('거래 내역을 찾을 수 없습니다.');
        }
        
        $actualAmountToRevert = $historyData['actual_amount'];
        $transactionHistoryId = $historyData['id'];
        
        // member 잔액 확인 및 차감 (FOR UPDATE로 잠금)
        $lockMemberQuery = "SELECT id, amount FROM member WHERE id = ? FOR UPDATE";
        $lockMemberStmt = mysqli_prepare($con, $lockMemberQuery);
        mysqli_stmt_bind_param($lockMemberStmt, "i", $memberId);
        mysqli_stmt_execute($lockMemberStmt);
        $lockResult = mysqli_stmt_get_result($lockMemberStmt);
        $lockedMember = mysqli_fetch_assoc($lockResult);
        mysqli_stmt_close($lockMemberStmt);
        
        $currentBalance = $lockedMember['amount'];
        
        if ($currentBalance < $actualAmountToRevert) {
            throw new Exception("취소 불가: 잔액 부족 (현재: " . number_format($currentBalance) . 
                              "원, 필요: " . number_format($actualAmountToRevert) . "원)");
        }
        
        // 롤링 내역 조회 및 복구
        $getRollingQuery = "SELECT * FROM rolling_history WHERE original_transaction_id = ?";
        $getRollingStmt = mysqli_prepare($con, $getRollingQuery);
        mysqli_stmt_bind_param($getRollingStmt, "i", $transactionHistoryId);
        mysqli_stmt_execute($getRollingStmt);
        $rollingResult = mysqli_stmt_get_result($getRollingStmt);
        
        $rollingDataArray = [];
        while ($rollingRow = mysqli_fetch_assoc($rollingResult)) {
            $rollingDataArray[] = $rollingRow;
            
            // 상위 회원 잔액 차감 (FOR UPDATE로 잠금)
            $parentLockQuery = "SELECT amount FROM member WHERE id = ? FOR UPDATE";
            $parentLockStmt = mysqli_prepare($con, $parentLockQuery);
            mysqli_stmt_bind_param($parentLockStmt, "i", $rollingRow['parent_member_id']);
            mysqli_stmt_execute($parentLockStmt);
            mysqli_stmt_close($parentLockStmt);
            
            $parentBalanceQuery = "UPDATE member SET amount = amount - ? WHERE id = ?";
            $parentBalanceStmt = mysqli_prepare($con, $parentBalanceQuery);
            mysqli_stmt_bind_param($parentBalanceStmt, "di", $rollingRow['rolling_amount'], $rollingRow['parent_member_id']);
            
            if (!mysqli_stmt_execute($parentBalanceStmt)) {
                throw new Exception("상위 회원 잔액 차감 실패");
            }
            mysqli_stmt_close($parentBalanceStmt);
        }
        mysqli_stmt_close($getRollingStmt);
        
        // 회원 잔액 차감
        $balanceQuery = "UPDATE member SET amount = amount - ? WHERE id = ?";
        $balanceStmt = mysqli_prepare($con, $balanceQuery);
        mysqli_stmt_bind_param($balanceStmt, "di", $actualAmountToRevert, $memberId);
        
        if (!mysqli_stmt_execute($balanceStmt)) {
            throw new Exception("잔액 차감 실패");
        }
        mysqli_stmt_close($balanceStmt);
        
        $newBalance = $currentBalance - $actualAmountToRevert;
        
        // rolling_history 삭제
        if (!empty($rollingDataArray)) {
            $deleteRollingQuery = "DELETE FROM rolling_history WHERE original_transaction_id = ?";
            $deleteRollingStmt = mysqli_prepare($con, $deleteRollingQuery);
            mysqli_stmt_bind_param($deleteRollingStmt, "i", $transactionHistoryId);
            mysqli_stmt_execute($deleteRollingStmt);
            mysqli_stmt_close($deleteRollingStmt);
        }
        
        // transaction_history 삭제
        $deleteHistoryQuery = "DELETE FROM transaction_history WHERE id = ?";
        $deleteHistoryStmt = mysqli_prepare($con, $deleteHistoryQuery);
        mysqli_stmt_bind_param($deleteHistoryStmt, "i", $transactionHistoryId);
        mysqli_stmt_execute($deleteHistoryStmt);
        mysqli_stmt_close($deleteHistoryStmt);
        
        // transfer 상태 업데이트 (마지막에)
        $updateQuery = "UPDATE transfer SET 
                       status = ?, 
                       trans_id = NULL, 
                       cancelled_at = NOW(), 
                       updated_at = NOW() 
                       WHERE id = ?";
        $updateStmt = mysqli_prepare($con, $updateQuery);
        mysqli_stmt_bind_param($updateStmt, "si", $action, $transferId);
        
        if (!mysqli_stmt_execute($updateStmt)) {
            throw new Exception('상태 업데이트 실패');
        }
        mysqli_stmt_close($updateStmt);
        
        // 로그 기록
        if (function_exists('writeLog')) {
            $detail = "입금 취소 완료 - Trans ID: {$existingTransId}, Transfer ID: {$transferId}, " .
                     "사용자: {$userName}({$memberUserId}), 취소액: " . number_format($actualAmountToRevert) . "원";
            writeLog($con, 'DEPOSIT_CANCELLED', null, ['trans_id' => $existingTransId, 'transfer_id' => $transferId], 1, $detail, $adminMemberId);
        }
    }
    
    // ====== 기타 상태 변경 ======
    else {
        // 상태 변경 로그 기록 (모든 상태 변경에 대해)
        $oldData = [
            'transfer_id' => $transferId,
            'status' => $oldStatus,
            'amount' => $depositAmount,
            'user_name' => $userName,
            'member_user_id' => $memberUserId,
            'balance' => $currentBalance
        ];
        
        $newData = [
            'transfer_id' => $transferId,
            'status' => $action,
            'amount' => $depositAmount,
            'user_name' => $userName,
            'member_user_id' => $memberUserId,
            'processed_at' => date('Y-m-d H:i:s')
        ];
        
        $statusMessages = [
            'completed' => '승인',
            'cancelled' => '취소',
            'transferred' => '송금완료',
            'requested' => '대기'
        ];
        
        $statusMessage = $statusMessages[$action] ?? $action;
        $detail = "입금 신청 {$statusMessage} 처리 - ID: {$transferId}, 사용자: {$userName}({$memberUserId}), 금액: " . number_format($depositAmount) . "원, 변경: {$oldStatus} → {$action}";
        
        if (function_exists('writeLog')) {
            writeLog($con, 'DEPOSIT_STATUS_CHANGE', $oldData, $newData, 1, $detail, $adminMemberId);
        }
        
        // transfer 테이블 업데이트 - status와 updated_at만 변경
        $updateQuery = "UPDATE transfer SET status = ?, updated_at = NOW() WHERE id = ?";
        $updateStmt = mysqli_prepare($con, $updateQuery);
        mysqli_stmt_bind_param($updateStmt, "si", $action, $transferId);
        
        if (!mysqli_stmt_execute($updateStmt)) {
            throw new Exception('상태 업데이트 실패');
        }
        mysqli_stmt_close($updateStmt);
    }
    
    // 트랜잭션 커밋
    mysqli_commit($con);
    
    // 성공 응답 데이터 구성
    $responseData = [
        'id' => $transferId,
        'old_status' => $oldStatus,
        'new_status' => $action,
        'processed_at' => date('Y-m-d H:i:s')
    ];
    
    if ($transId) {
        $responseData['trans_id'] = $transId;
    }
    
    if ($newBalance !== null) {
        $responseData['new_balance'] = $newBalance;
        $responseData['old_balance'] = $currentBalance;
    }
    
    if ($action === 'completed' && $actualAmount !== null) {
        $responseData['deposit_details'] = [
            'requested_amount' => $depositAmount,
            'total_fee_rate' => $totalFeeRate,
            'total_fee_amount' => $totalFeeAmount,
            'company_fee_amount' => $companyFeeAmount,
            'rolling_amount' => $totalRollingAmount,
            'rolling_levels' => count($rollingDetails),
            'actual_amount' => $actualAmount
        ];
    }
    
    jsonResponse(true, '입금 신청이 성공적으로 처리되었습니다.', $responseData);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    mysqli_rollback($con);
    
    // 에러 로깅
    error_log("Process Deposit Error: " . $e->getMessage());
    
    if (function_exists('writeLog')) {
        $errorData = [
            'transfer_id' => $transferId ?? null,
            'action' => $action ?? null,
            'error' => $e->getMessage()
        ];
        writeLog($con, 'DEPOSIT_PROCESS_ERROR', null, $errorData, 0, $e->getMessage(), $adminMemberId ?? 0);
    }
    
    jsonResponse(false, $e->getMessage(), null, 500);
    
} finally {
    // autocommit 복원
    mysqli_autocommit($con, TRUE);
    
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}

/**
 * 콜백 URL 호출 함수 (bind_param 개수 오류 수정)
 */
function callWebhook($con, $callbackUrl, $code, $data, $logInfo, $adminMemberId = null) {
    // 반환할 결과 배열 초기화
    $result = [
        'success' => false,
        'http_code' => null,
        'response_code' => null,
        'message' => ''
    ];
    
    // 콜백 URL이 비어있으면 바로 반환
    if (empty($callbackUrl)) {
        $result['message'] = '콜백 URL이 없음';
        return $result;
    }
    
    // 필수 정보 확인
    if (!isset($logInfo['trans_id']) || !isset($logInfo['transfer_id']) || !isset($logInfo['member_id'])) {
        $result['message'] = '필수 로그 정보 누락';
        return $result;
    }
    
    // 콜백 데이터에 code 추가
    $callbackData = array_merge(['code' => $code], $data);
    $requestDataJson = json_encode($callbackData);
    
    // 변수들을 미리 정의 (참조 전달을 위해)
    $transId = $logInfo['trans_id'];
    $transferId = (string)$logInfo['transfer_id'];  // 문자열로 변환
    $memberId = (int)$logInfo['member_id'];         // 정수로 변환
    $responseCode = null;
    $httpCode = 0;
    $responseMessage = '';
    
    try {
        // cURL 세션 초기화
        $ch = curl_init($callbackUrl);
        
        // cURL 옵션 설정
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($callbackData));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: PaymentSystem/1.0',
            'Accept: application/json, text/plain, */*'
        ]);
        
        // cURL 요청 실행
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        $curlErrno = curl_errno($ch);
        
        // cURL 세션 종료
        curl_close($ch);
        
        // 응답 처리
        $logStatus = 'SUCCESS';
        
        if ($response === false) {
            // cURL 요청 실패
            $logStatus = 'CURL_FAILED';
            $responseMessage = "cURL Error [{$curlErrno}]: {$curlError}";
            $result['message'] = $responseMessage;
        } elseif ($httpCode === 0) {
            // HTTP 응답 없음
            $logStatus = 'NO_RESPONSE';
            $responseMessage = "No HTTP response (connection failed or timeout)";
            if (!empty($curlError)) {
                $responseMessage .= " - cURL Error: {$curlError}";
            }
            $result['message'] = $responseMessage;
        } elseif ($httpCode >= 400) {
            // HTTP 에러 응답
            $logStatus = 'HTTP_ERROR';
            $responseMessage = "HTTP {$httpCode} Error";
            if (!empty($response)) {
                $responseMessage .= " - Response: " . substr($response, 0, 500);
            }
            $result['message'] = $responseMessage;
        } else {
            // 정상 응답
            $logStatus = 'SUCCESS';
            
            // JSON 응답 파싱 시도
            $responseData = json_decode($response, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($responseData)) {
                if (isset($responseData['code'])) {
                    $responseCode = $responseData['code'];
                }
                if (isset($responseData['message'])) {
                    $responseMessage = $responseData['message'];
                } else {
                    $responseMessage = "JSON response received";
                }
                $responseMessage .= " - Full Response: " . $response;
            } else {
                $responseMessage = "Non-JSON response: " . substr($response, 0, 500);
            }
            
            $result['message'] = $responseMessage;
            $result['success'] = true;
        }
        
        // 결과 업데이트
        $result['http_code'] = $httpCode;
        $result['response_code'] = $responseCode;
        
        // 콜백 로그 기록 - 타입과 변수 개수 정확히 맞추기
        $callbackLogQuery = "INSERT INTO callback_log 
                           (trans_id, transfer_id, member_id, callback_url, 
                           request_data, response_code, http_status, 
                           response_message, created_at) 
                           VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())";
        $callbackLogStmt = mysqli_prepare($con, $callbackLogQuery);
        
        if ($callbackLogStmt) {
            // response_code가 null이면 logStatus 사용
            $responseCodeForLog = ($responseCode !== null) ? (string)$responseCode : $logStatus;
            
            // 타입 문자열: s s i s s s i s (8개)
            // 변수: transId, transferId, memberId, callbackUrl, requestDataJson, responseCodeForLog, httpCode, responseMessage (8개)
            mysqli_stmt_bind_param($callbackLogStmt, "ssisssis", 
                $transId,                // s - string
                $transferId,             // s - string  
                $memberId,               // i - integer
                $callbackUrl,            // s - string
                $requestDataJson,        // s - string
                $responseCodeForLog,     // s - string
                $httpCode,               // i - integer
                $responseMessage         // s - string
            );
            
            if (!mysqli_stmt_execute($callbackLogStmt)) {
                error_log("콜백 로그 저장 실패: " . mysqli_error($con));
            }
            mysqli_stmt_close($callbackLogStmt);
        }
        
        // 시스템 로그에 추가
        if (function_exists('writeLog') && $adminMemberId !== null) {
            $logDetail = "콜백 URL 호출 완료 - URL: {$callbackUrl}, " .
                       "HTTP 상태: " . ($httpCode ?: 'NO_RESPONSE') . ", " .
                       "응답 코드: " . ($responseCode ?: 'N/A') . ", " .
                       "상태: {$logStatus}";
            
            if (!empty($curlError)) {
                $logDetail .= ", cURL Error: {$curlError}";
            }
            
            $logParams = [
                'trans_id' => $transId, 
                'transfer_id' => $transferId,
                'callback_url' => $callbackUrl,
                'http_code' => $httpCode,
                'status' => $logStatus
            ];
            
            writeLog(
                $con, 
                'CALLBACK_EXECUTED', 
                null, 
                $logParams, 
                1,
                $logDetail, 
                $adminMemberId
            );
        }
        
    } catch (Exception $e) {
        // 예외 발생 시에도 로그 기록
        $result['success'] = false;
        $result['message'] = "콜백 호출 중 예외 발생: " . $e->getMessage();
        
        // 예외 상황도 데이터베이스에 기록
        try {
            $callbackLogQuery = "INSERT INTO callback_log 
                               (trans_id, transfer_id, member_id, callback_url, 
                               request_data, response_code, http_status, 
                               response_message, created_at) 
                               VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())";
            $callbackLogStmt = mysqli_prepare($con, $callbackLogQuery);
            
            if ($callbackLogStmt) {
                $exceptionMessage = "Exception: " . $e->getMessage();
                $exceptionCode = 'EXCEPTION';
                $zeroHttpCode = 0;
                
                // 타입 문자열: s s i s s s i s (8개)
                mysqli_stmt_bind_param($callbackLogStmt, "ssisssis", 
                    $transId,            // s - string
                    $transferId,         // s - string
                    $memberId,           // i - integer  
                    $callbackUrl,        // s - string
                    $requestDataJson,    // s - string
                    $exceptionCode,      // s - string
                    $zeroHttpCode,       // i - integer
                    $exceptionMessage    // s - string
                );
                
                mysqli_stmt_execute($callbackLogStmt);
                mysqli_stmt_close($callbackLogStmt);
            }
        } catch (Exception $logException) {
            error_log("콜백 예외 로그 저장 실패: " . $logException->getMessage());
        }
    }
    
    return $result;
}
?>