<?php
// 에러 표시 설정
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크
if (!isset($_SESSION['id'])) {
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => '파트너 권한이 필요합니다.'
    ]);
    exit;
}

$partner_user_id = $_SESSION['id'];  // 이것이 member.user_id 값

// POST 데이터 받기
$input = json_decode(file_get_contents('php://input'), true);
$amount = isset($input['amount']) ? intval($input['amount']) : 0;

// 고정 수수료
$FEE = 1000;

// 시간 범위 체크 함수 (자정 넘나드는 시간 포함)
function isTimeInRange($current, $start, $end) {
    $current = strtotime($current);
    $start = strtotime($start);
    $end = strtotime($end);
    
    if ($start <= $end) {
        // 같은 날 시간 (예: 12:00 ~ 13:00)
        return ($current >= $start && $current <= $end);
    } else {
        // 자정 넘나드는 시간 (예: 23:00 ~ 06:00)
        return ($current >= $start || $current <= $end);
    }
}

// 매시간 차단 시간 체크 함수 (새로 추가)
function isHourlyWithdrawBlocked() {
    $currentMinute = (int)date('i');
    // 50분~59분 사이면 차단
    return ($currentMinute >= 50);
}

// 시간 범위 설명 생성
function getTimeRangeDescription($start, $end) {
    $startHour = intval(explode(':', $start)[0]);
    $endHour = intval(explode(':', $end)[0]);
    
    if ($endHour <= $startHour) {
        return $start . ' ~ ' . $end . ' (다음날)';
    }
    return $start . ' ~ ' . $end;
}

try {
    // 트랜잭션 시작
    mysqli_begin_transaction($con);
    
    // 1. 먼저 user_id로 member.id와 등급을 조회
    $getMemberQuery = "SELECT id, grade FROM member WHERE user_id = ?";
    $getMemberStmt = mysqli_prepare($con, $getMemberQuery);
    
    if (!$getMemberStmt) {
        throw new Exception("회원 정보 조회 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($getMemberStmt, "s", $partner_user_id);
    mysqli_stmt_execute($getMemberStmt);
    $getMemberResult = mysqli_stmt_get_result($getMemberStmt);
    
    if (!$memberRow = mysqli_fetch_assoc($getMemberResult)) {
        throw new Exception("회원 정보를 찾을 수 없습니다.");
    }
    
    $partner_id = $memberRow['id'];  // member 테이블의 primary key
    $user_grade = $memberRow['grade'];
    mysqli_stmt_close($getMemberStmt);
    
    // 2. 출금 제한 시간 체크
    $settingsQuery = "SELECT * FROM settings LIMIT 1";
    $settingsResult = mysqli_query($con, $settingsQuery);
    $settings = mysqli_fetch_assoc($settingsResult);
    
    if ($settings) {
        $current_time = date('H:i');
        $current_minute = (int)date('i');
        
        // 매시간 차단 확인 (모든 등급에 적용)
        if ($settings['hourly_withdraw_block_enabled'] && isHourlyWithdrawBlocked()) {
            throw new Exception('매시간 50분~정각까지는 출금이 제한됩니다. (현재: ' . date('H:i') . ')');
        }
        
        // 등급별 차단 시간 확인
        if ($user_grade === 'partner' && $settings['partner_withdraw_block_enabled']) {
            $start = $settings['partner_withdraw_block_start'];
            $end = $settings['partner_withdraw_block_end'];
            
            if ($start && $end && isTimeInRange($current_time, $start, $end)) {
                throw new Exception('현재 출금 제한 시간입니다. (' . getTimeRangeDescription($start, $end) . ')');
            }
        } elseif ($user_grade === 'basic' && $settings['basic_withdraw_block_enabled']) {
            $start = $settings['basic_withdraw_block_start'];
            $end = $settings['basic_withdraw_block_end'];
            
            if ($start && $end && isTimeInRange($current_time, $start, $end)) {
                throw new Exception('현재 출금 제한 시간입니다. (' . getTimeRangeDescription($start, $end) . ')');
            }
        }
    }
    
    // 3. 이미 진행 중인 출금 신청이 있는지 확인
    $pendingCheckQuery = "
        SELECT COUNT(*) as pending_count
        FROM transfer
        WHERE user_id = ?
        AND type = 'withdraw'
        AND status = 'requested'
    ";
    
    $pendingCheckStmt = mysqli_prepare($con, $pendingCheckQuery);
    
    if (!$pendingCheckStmt) {
        throw new Exception("대기 중인 출금 확인 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($pendingCheckStmt, "i", $partner_id);
    mysqli_stmt_execute($pendingCheckStmt);
    $pendingResult = mysqli_stmt_get_result($pendingCheckStmt);
    $pendingRow = mysqli_fetch_assoc($pendingResult);
    mysqli_stmt_close($pendingCheckStmt);
    
    if ($pendingRow['pending_count'] > 0) {
        throw new Exception("이미 처리 대기 중인 출금 신청이 있습니다. 기존 신청이 완료된 후 다시 시도해주세요.");
    }
    
    // 4. 파트너 정보 및 잔액 조회 (FOR UPDATE로 잠금)
    $checkQuery = "
        SELECT 
            m.id,
            m.user_id,
            m.amount,
            m.bank,
            m.account_number,
            m.account_holder,
            m.grade,
            (
                SELECT COALESCE(SUM(amount), 0) 
                FROM transfer 
                WHERE user_id = m.id 
                AND type = 'withdraw' 
                AND status IN ('requested', 'pending')
            ) as pending_withdrawal
        FROM member m
        WHERE m.user_id = ?
        AND m.grade IN ('basic', 'partner')
        FOR UPDATE
    ";
    
    $checkStmt = mysqli_prepare($con, $checkQuery);
    
    if (!$checkStmt) {
        throw new Exception("잔액 조회 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($checkStmt, "s", $partner_user_id);
    mysqli_stmt_execute($checkStmt);
    $checkResult = mysqli_stmt_get_result($checkStmt);
    
    if (!$row = mysqli_fetch_assoc($checkResult)) {
        throw new Exception("파트너 정보를 찾을 수 없습니다.");
    }
    
    mysqli_stmt_close($checkStmt);
    
    // 5. 유효성 검사
    if ($amount < 10000) {
        throw new Exception("최소 출금 금액은 10,000원입니다.");
    }
    
    if (!$row['bank'] || !$row['account_number'] || !$row['account_holder']) {
        throw new Exception("계좌 정보가 등록되지 않았습니다. 먼저 계좌 정보를 등록해주세요.");
    }
    
    // 출금 가능 금액 계산
    $available_amount = $row['amount'] - $row['pending_withdrawal'];
    
    if ($amount > $available_amount) {
        throw new Exception("출금 가능 금액을 초과했습니다. 현재 출금 가능 금액: " . number_format($available_amount) . "원");
    }
    
    // 실제 입금액 계산 (수수료 차감)
    $actual_amount = $amount - $FEE;
    
    if ($actual_amount <= 0) {
        throw new Exception("수수료를 제외한 실제 입금액이 0원 이하입니다.");
    }
    
    // 6. 출금 신청 기록 생성
    $insertQuery = "
        INSERT INTO transfer (
            user_id, 
            amount, 
            type, 
            status, 
            payment_status,
            created_at,
            updated_at
        ) VALUES (
            ?, ?, 'withdraw', 'requested', 'pending', NOW(), NOW()
        )
    ";
    
    $insertStmt = mysqli_prepare($con, $insertQuery);
    
    if (!$insertStmt) {
        throw new Exception("출금 신청 생성 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($insertStmt, "ii", $partner_id, $amount);
    
    if (!mysqli_stmt_execute($insertStmt)) {
        throw new Exception("출금 신청 저장 실패: " . mysqli_stmt_error($insertStmt));
    }
    
    $withdrawal_id = mysqli_insert_id($con);
    mysqli_stmt_close($insertStmt);
    
    // 7. 활동 로그 기록 (테이블이 있는 경우)
    $logTableCheck = "SHOW TABLES LIKE 'activity_log'";
    $tableResult = mysqli_query($con, $logTableCheck);
    
    if (mysqli_num_rows($tableResult) > 0) {
        $logQuery = "
            INSERT INTO activity_log (
                user_id,
                action,
                details,
                ip_address,
                created_at
            ) VALUES (
                ?, 'withdrawal_request', ?, ?, NOW()
            )
        ";
        
        $logStmt = mysqli_prepare($con, $logQuery);
        
        if ($logStmt) {
            $details = json_encode([
                'withdrawal_id' => $withdrawal_id,
                'amount' => $amount,
                'fee' => $FEE,
                'actual_amount' => $actual_amount,
                'user_grade' => $user_grade,
                'time_checked' => date('H:i'),
                'hourly_block_checked' => $settings['hourly_withdraw_block_enabled'] ?? 0
            ], JSON_UNESCAPED_UNICODE);
            
            $ip_address = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
            
            mysqli_stmt_bind_param($logStmt, "iss", $partner_id, $details, $ip_address);
            mysqli_stmt_execute($logStmt);
            mysqli_stmt_close($logStmt);
        }
    }
    
    // 8. 시스템 로그 기록 (보안 감사용)
    error_log("출금 신청 성공 - User ID: {$partner_user_id}, Member ID: {$partner_id}, Grade: {$user_grade}, Amount: {$amount}, Withdrawal ID: {$withdrawal_id}, Time: " . date('H:i'));
    
    // 트랜잭션 커밋
    mysqli_commit($con);
    
    // 성공 응답
    echo json_encode([
        'success' => true,
        'message' => '출금 신청이 완료되었습니다.',
        'withdrawal_id' => $withdrawal_id,
        'amount' => $amount,
        'fee' => $FEE,
        'actual_amount' => $actual_amount,
        'new_available_amount' => $available_amount - $amount,
        'user_grade' => $user_grade
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    mysqli_rollback($con);
    
    // 보안 로그 기록 (실패한 시도도 기록)
    error_log("출금 신청 실패 - User ID: {$partner_user_id}, Amount: {$amount}, Error: " . $e->getMessage() . ", IP: " . ($_SERVER['REMOTE_ADDR'] ?? 'unknown') . ", Time: " . date('H:i'));
    
    // 에러 응답
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>