<?php
// ajax/process_transaction.php
header('Content-Type: application/json; charset=UTF-8');

// 세션 시작
session_start();

// 데이터베이스 연결 (writeLog 함수 포함)
require_once '../inc/db_connect.php';

// 응답 데이터 초기화
$response = [
    'success' => false,
    'message' => '',
    'data' => null
];

try {
    // 세션 체크
    if (!isset($_SESSION['id'])) {
        throw new Exception('로그인이 필요합니다.');
    }
    
    $admin_id = $_SESSION['id'];

    // 현재 로그인한 관리자 정보 조회
    $admin_query = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $admin_query);
    mysqli_stmt_bind_param($stmt, "s", $admin_id);
    mysqli_stmt_execute($stmt);
    $admin_result = mysqli_stmt_get_result($stmt);
    $admin_data = mysqli_fetch_assoc($admin_result);
    mysqli_stmt_close($stmt);

    if (!$admin_data) {
        throw new Exception('관리자 정보를 찾을 수 없습니다.');
    }

    $admin_member_id = $admin_data['id'];
    $admin_name = $admin_data['user_name'];

    // POST 요청 확인
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('잘못된 요청 방식입니다.');
    }

    // 필수 파라미터 검증
    if (!isset($_POST['user_id']) || empty($_POST['user_id'])) {
        throw new Exception('가맹점 ID가 필요합니다.');
    }
    
    if (!isset($_POST['type']) || !in_array($_POST['type'], ['deposit', 'withdraw'])) {
        throw new Exception('올바른 거래 유형을 선택해주세요.');
    }
    
    if (!isset($_POST['amount']) || !is_numeric($_POST['amount']) || $_POST['amount'] <= 0) {
        throw new Exception('올바른 금액을 입력해주세요.');
    }
    
    // 메모 필수 체크
    if (!isset($_POST['memo']) || trim($_POST['memo']) === '') {
        throw new Exception('메모를 입력해주세요. (거래 사유 필수)');
    }

    // 데이터 정리
    $user_id = escapeString($con, $_POST['user_id']);
    $type = escapeString($con, $_POST['type']);
    $amount = floatval($_POST['amount']);
    $memo = escapeString($con, trim($_POST['memo']));
    $ip_address = $_SERVER['REMOTE_ADDR'];

    // 입력 데이터 로그용
    $inputData = [
        'target_user_id' => $user_id,
        'type' => $type,
        'amount' => $amount,
        'memo' => $memo,
        'admin_id' => $admin_id,
        'admin_name' => $admin_name
    ];

    // 트랜잭션 시작
    beginTransaction($con);

    // 1. 가맹점 정보 조회 및 잠금
    $member_query = "
        SELECT id, user_id, user_name, amount 
        FROM member 
        WHERE user_id = '$user_id' 
        FOR UPDATE
    ";
    
    $member_result = mysqli_query($con, $member_query);
    
    if (!$member_result) {
        throw new Exception('데이터베이스 오류: ' . mysqli_error($con));
    }
    
    if (mysqli_num_rows($member_result) == 0) {
        writeLog($con, 'MANUAL_TRANSACTION_FAILED', $inputData, null, 0, 
                "수동거래 실패 - 존재하지 않는 가맹점: {$user_id}", $admin_member_id);
        throw new Exception('존재하지 않는 가맹점입니다.');
    }
    
    $member = mysqli_fetch_assoc($member_result);
    $member_id = $member['id'];
    $current_balance = floatval($member['amount']);
    
    // 2. 잔액 검증 (회수의 경우)
    if ($type === 'withdraw' && $amount > $current_balance) {
        writeLog($con, 'MANUAL_TRANSACTION_FAILED', $inputData, null, 0, 
                "수동거래 실패 - 잔액 부족: 요청({$amount}) > 현재({$current_balance})", $admin_member_id);
        throw new Exception('잔액이 부족합니다. 현재 잔액: ' . number_format($current_balance) . '원');
    }
    
    // 3. 새로운 잔액 계산
    if ($type === 'deposit') {
        $new_balance = $current_balance + $amount;
        $type_text = '지급';
        $log_type = 'MANUAL_DEPOSIT';
    } else {
        $new_balance = $current_balance - $amount;
        $type_text = '회수';
        $log_type = 'MANUAL_WITHDRAW';
    }
    
    // 4. member 테이블 잔액 업데이트
    $update_member_query = "
        UPDATE member 
        SET amount = $new_balance
        WHERE id = $member_id
    ";
    
    if (!mysqli_query($con, $update_member_query)) {
        writeLog($con, 'MANUAL_TRANSACTION_FAILED', $inputData, null, 0, 
                "수동거래 실패 - 잔액 업데이트 오류: " . mysqli_error($con), $admin_member_id);
        throw new Exception('잔액 업데이트 실패: ' . mysqli_error($con));
    }
    
    // 5. manual_transactions 테이블에 거래 내역 저장
    $transaction_id = null;
    $check_table = "SHOW TABLES LIKE 'manual_transactions'";
    $table_result = mysqli_query($con, $check_table);
    
    if (mysqli_num_rows($table_result) > 0) {
        $insert_transaction_query = "
            INSERT INTO manual_transactions (
                user_id,
                member_user_id,
                admin_id,
                type,
                amount,
                balance_before,
                balance_after,
                memo,
                status,
                ip_address,
                created_at
            ) VALUES (
                $member_id,
                '$user_id',
                '$admin_id',
                '$type',
                $amount,
                $current_balance,
                $new_balance,
                '$memo',
                'completed',
                '$ip_address',
                NOW()
            )
        ";
        
        if (mysqli_query($con, $insert_transaction_query)) {
            $transaction_id = mysqli_insert_id($con);
        }
    }
    
    // 6. all_log 테이블에 로그 저장
    $log_detail = sprintf(
        "[수동%s] 가맹점: %s (%s), 금액: %s원, 메모: %s, 처리자: %s",
        $type_text,
        $member['user_name'],
        $user_id,
        number_format($amount),
        $memo,
        $admin_name
    );
    
    // 이전 데이터와 새 데이터
    $old_data = [
        'balance' => $current_balance,
        'member_id' => $member_id,
        'member_name' => $member['user_name'],
        'member_user_id' => $user_id
    ];
    
    $new_data = [
        'balance' => $new_balance,
        'amount' => $amount,
        'type' => $type,
        'memo' => $memo,
        'transaction_id' => $transaction_id,
        'processed_by' => $admin_name,
        'processed_at' => date('Y-m-d H:i:s')
    ];
    
    // 로그 기록 - 중요: admin_member_id를 userId로 전달
    writeLog($con, $log_type, $old_data, $new_data, 1, $log_detail, $admin_member_id);
    
    // 7. 트랜잭션 커밋
    commitTransaction($con);
    
    // 성공 응답
    $response['success'] = true;
    $response['message'] = sprintf(
        "%s 처리가 완료되었습니다. (금액: %s원)",
        $type_text,
        number_format($amount)
    );
    $response['data'] = [
        'transaction_id' => $transaction_id,
        'type' => $type,
        'amount' => $amount,
        'balance_before' => $current_balance,
        'balance_after' => $new_balance,
        'member_name' => $member['user_name'],
        'created_at' => date('Y-m-d H:i:s')
    ];
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    if (isset($con)) {
        rollbackTransaction($con);
    }
    
    $response['success'] = false;
    $response['message'] = $e->getMessage();
    
    // 에러 로그 기록 - admin_member_id 전달
    if (isset($con) && isset($admin_member_id)) {
        $error_detail = "[수동거래 실패] " . $e->getMessage();
        writeLog($con, 'MANUAL_TRANSACTION_ERROR', $inputData ?? null, null, 0, $error_detail, $admin_member_id);
    }
    
    // 서버 에러 로그
    error_log("process_transaction.php Error: " . $e->getMessage());
}

// JSON 응답 출력
echo json_encode($response, JSON_UNESCAPED_UNICODE);

// 데이터베이스 연결 종료
if (isset($con)) {
    mysqli_close($con);
}
?>