<?php
// 에러 표시 설정
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결 (writeLog 함수 포함)
require_once '../inc/db_connect.php';

// 트랜잭션 ID 생성 함수
function generateTransactionId() {
    return 'TXN' . date('YmdHis') . sprintf('%06d', mt_rand(0, 999999));
}

// 관리자 권한 체크
if (!isset($_SESSION['id'])) {
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => '관리자 권한이 필요합니다.'
    ]);
    exit;
}

// 관리자 정보 조회
$adminCheckQuery = "SELECT id, grade, user_id FROM member WHERE user_id = ?";
$adminCheckStmt = mysqli_prepare($con, $adminCheckQuery);
mysqli_stmt_bind_param($adminCheckStmt, "s", $_SESSION['id']);
mysqli_stmt_execute($adminCheckStmt);
$adminResult = mysqli_stmt_get_result($adminCheckStmt);
$adminData = mysqli_fetch_assoc($adminResult);
mysqli_stmt_close($adminCheckStmt);

if (!$adminData || $adminData['grade'] !== 'admin') {
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => '관리자 권한이 필요합니다.'
    ]);
    exit;
}

$adminMemberId = $adminData['id'];
$adminUserId = $_SESSION['id'];

// POST 데이터 받기
$input = json_decode(file_get_contents('php://input'), true);
$withdrawal_id = isset($input['id']) ? intval($input['id']) : 0;
$action = isset($input['action']) ? $input['action'] : '';

// 유효성 검사
if ($withdrawal_id <= 0) {
    echo json_encode([
        'success' => false,
        'message' => '잘못된 출금 ID입니다.'
    ]);
    exit;
}

// 허용된 액션 확인
$allowed_actions = ['pending', 'completed', 'cancelled'];
if (!in_array($action, $allowed_actions)) {
    echo json_encode([
        'success' => false,
        'message' => '잘못된 액션입니다.'
    ]);
    exit;
}

try {
    // 트랜잭션 시작
    mysqli_begin_transaction($con);
    
    // 1. 현재 출금 신청 정보 조회 (잠금)
    $checkQuery = "
        SELECT 
            t.id,
            t.user_id,
            t.amount,
            t.status,
            t.created_at,
            t.trans_id,
            m.amount as user_balance,
            m.user_id as member_user_id,
            m.user_name,
            m.rate
        FROM transfer t
        INNER JOIN member m ON t.user_id = m.id
        WHERE t.id = ? 
        AND t.type = 'withdraw'
        FOR UPDATE
    ";
    
    $checkStmt = mysqli_prepare($con, $checkQuery);
    if (!$checkStmt) {
        throw new Exception("조회 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($checkStmt, "i", $withdrawal_id);
    mysqli_stmt_execute($checkStmt);
    $result = mysqli_stmt_get_result($checkStmt);
    
    if (!$withdrawal = mysqli_fetch_assoc($result)) {
        throw new Exception("출금 신청을 찾을 수 없습니다.");
    }
    
    mysqli_stmt_close($checkStmt);
    
    // 2. 상태 변경 가능 여부 확인
    $current_status = $withdrawal['status'];
    $memberId = $withdrawal['user_id'];
    $memberUserId = $withdrawal['member_user_id'];
    $userName = $withdrawal['user_name'];
    $withdrawAmount = $withdrawal['amount'];
    $currentBalance = $withdrawal['user_balance'];
    $memberRate = $withdrawal['rate'] ?? 0;
    $existingTransId = $withdrawal['trans_id'] ?? null;
    
    // 상태별 허용 액션 확인
    if ($current_status === 'requested') {
        // 신청 상태에서는 대기, 완료, 취소 모두 가능
    } elseif ($current_status === 'pending') {
        // 대기 상태에서는 완료, 취소만 가능
        if ($action === 'pending') {
            throw new Exception("이미 대기 상태입니다.");
        }
    } elseif ($current_status === 'completed') {
        throw new Exception("이미 완료된 출금 신청입니다.");
    } elseif ($current_status === 'cancelled') {
        throw new Exception("이미 취소된 출금 신청입니다.");
    }
    
    // 이미 승인된 건을 다시 승인하려는 경우 체크
    if ($action === 'completed' && $existingTransId !== null) {
        throw new Exception('이미 처리된 거래입니다.');
    }
    
    // 완료 처리 시 잔액 확인
    if ($action === 'completed' && $currentBalance < $withdrawAmount) {
        throw new Exception("출금 불가: 사용자({$userName})의 현재 잔액(" . number_format($currentBalance) . "원)이 출금액(" . number_format($withdrawAmount) . "원)보다 부족합니다.");
    }
    
    // 3. 상태 업데이트
    $updateQuery = "
        UPDATE transfer 
        SET status = ?,
            updated_at = NOW()
        WHERE id = ?
    ";
    
    $updateStmt = mysqli_prepare($con, $updateQuery);
    if (!$updateStmt) {
        throw new Exception("업데이트 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($updateStmt, "si", $action, $withdrawal_id);
    
    if (!mysqli_stmt_execute($updateStmt)) {
        throw new Exception("상태 업데이트 실패: " . mysqli_stmt_error($updateStmt));
    }
    
    mysqli_stmt_close($updateStmt);
    
    // 4. 상태 변경 all_log 기록
    $oldData = [
        'transfer_id' => $withdrawal_id,
        'status' => $current_status,
        'amount' => $withdrawAmount,
        'user_name' => $userName,
        'member_user_id' => $memberUserId,
        'balance' => $currentBalance
    ];
    
    $newData = [
        'transfer_id' => $withdrawal_id,
        'status' => $action,
        'amount' => $withdrawAmount,
        'user_name' => $userName,
        'member_user_id' => $memberUserId,
        'processed_at' => date('Y-m-d H:i:s')
    ];
    
    // 상태별 상세 메시지
    $statusMessages = [
        'pending' => '대기',
        'completed' => '완료',
        'cancelled' => '취소'
    ];
    
    $statusMessage = $statusMessages[$action] ?? $action;
    $detail = "출금 신청 {$statusMessage} 처리 - ID: {$withdrawal_id}, 사용자: {$userName}({$memberUserId}), 금액: " . number_format($withdrawAmount) . "원, 변경: {$current_status} → {$action}";
    
    writeLog($con, 'WITHDRAWAL_STATUS_CHANGE', $oldData, $newData, 1, $detail, $adminMemberId);
    
    // 변수 초기화
    $newBalance = null;
    $actualAmount = null;
    $feeAmount = null;
    
    // 5. 완료 처리인 경우 추가 처리
    if ($action === 'completed' && $current_status !== 'completed') {
        // 고정 수수료 1000원
        $feeAmount = 1000;
        $actualAmount = $withdrawAmount; // 실제 차감액 = 출금요청액
        $actualPaidAmount = $withdrawAmount - $feeAmount; // 실제 지급액 = 출금요청액 - 1000원
        
        // 잔액이 출금액보다 부족한 경우 체크
        if ($currentBalance < $actualAmount) {
            throw new Exception("출금 불가: 잔액 부족 (필요액: " . number_format($actualAmount) . "원, 현재잔액: " . number_format($currentBalance) . "원)");
        }
        
        // 최소 출금액 체크 (수수료 때문에)
        if ($withdrawAmount <= $feeAmount) {
            throw new Exception("출금 불가: 출금액이 수수료(1,000원)보다 커야 합니다.");
        }
        
        // 트랜잭션 ID 생성
        $transId = generateTransactionId();
        
        // transfer 테이블에 trans_id 업데이트 및 완료 시간 업데이트
        $updateTransQuery = "
            UPDATE transfer 
            SET trans_id = ?,
                completed_at = NOW()
            WHERE id = ?
        ";
        $updateTransStmt = mysqli_prepare($con, $updateTransQuery);
        mysqli_stmt_bind_param($updateTransStmt, "si", $transId, $withdrawal_id);
        mysqli_stmt_execute($updateTransStmt);
        mysqli_stmt_close($updateTransStmt);
        
        // member 테이블의 잔액 차감 (출금요청액 전체)
        $balanceQuery = "
            UPDATE member 
            SET amount = amount - ?
            WHERE id = ?
        ";
        
        $balanceStmt = mysqli_prepare($con, $balanceQuery);
        mysqli_stmt_bind_param($balanceStmt, "di", $actualAmount, $memberId);
        
        if (!mysqli_stmt_execute($balanceStmt)) {
            throw new Exception("잔액 차감 실패");
        }
        
        mysqli_stmt_close($balanceStmt);
        
        // 업데이트된 잔액 조회
        $getBalanceQuery = "SELECT amount FROM member WHERE id = ?";
        $getBalanceStmt = mysqli_prepare($con, $getBalanceQuery);
        mysqli_stmt_bind_param($getBalanceStmt, "i", $memberId);
        mysqli_stmt_execute($getBalanceStmt);
        $balanceResult = mysqli_stmt_get_result($getBalanceStmt);
        $newBalance = mysqli_fetch_assoc($balanceResult)['amount'];
        mysqli_stmt_close($getBalanceStmt);
        
        $historyQuery = "INSERT INTO transaction_history 
        (trans_id, transfer_id, user_id, type, requested_amount, 
        fee_rate, fee_amount, actual_amount, balance_before, 
        balance_after, status, processed_by, created_at) 
        VALUES (?, ?, ?, 'withdraw', ?, ?, ?, ?, ?, ?, 'completed', ?, NOW())";
        $historyStmt = mysqli_prepare($con, $historyQuery);

        mysqli_stmt_bind_param($historyStmt, "siiddddddi", 
            $transId,           // s - trans_id
            $withdrawal_id,     // i - transfer_id  
            $memberId,          // i - user_id
            $withdrawAmount,    // d - requested_amount
            $memberRate,        // d - fee_rate (기록용)
            $feeAmount,         // d - fee_amount (1000)
            $actualPaidAmount,  // d - actual_amount (출금액 - 1000)
            $currentBalance,    // d - balance_before
            $newBalance,        // d - balance_after
            $adminMemberId      // i - processed_by
        );

        if (!mysqli_stmt_execute($historyStmt)) {
            throw new Exception('트랜잭션 내역 저장 실패: ' . mysqli_error($con));
        }
        mysqli_stmt_close($historyStmt);
        
        // 잔액 차감 all_log 기록
        $balanceOldData = [
            'member_id' => $memberId,
            'member_user_id' => $memberUserId,
            'balance' => $currentBalance,
            'transfer_id' => $withdrawal_id
        ];
        
        $balanceNewData = [
            'trans_id' => $transId,
            'member_id' => $memberId,
            'member_user_id' => $memberUserId,
            'balance' => $newBalance,
            'requested_amount' => $withdrawAmount,
            'fee_amount' => $feeAmount,
            'actual_paid_amount' => $actualPaidAmount,
            'transfer_id' => $withdrawal_id
        ];
        
        $balanceDetail = "출금 승인 - Trans ID: {$transId}, 사용자: {$userName}({$memberUserId}), " .
                        "출금요청액: " . number_format($withdrawAmount) . "원, " .
                        "수수료: " . number_format($feeAmount) . "원, " .
                        "실지급액: " . number_format($actualPaidAmount) . "원, " .
                        "잔액: " . number_format($currentBalance) . "원 → " . number_format($newBalance) . "원";
        writeLog($con, 'WITHDRAWAL_BALANCE_SUBTRACT', $balanceOldData, $balanceNewData, 1, $balanceDetail, $adminMemberId);
    }

    // 6. 출금 취소 시 처리 (completed → cancelled)
    $actualAmountToRevert = null;

    if ($action === 'cancelled' && $current_status === 'completed') {
        // trans_id 조회
        $getTransQuery = "SELECT trans_id FROM transfer WHERE id = ?";
        $getTransStmt = mysqli_prepare($con, $getTransQuery);
        mysqli_stmt_bind_param($getTransStmt, "i", $withdrawal_id);
        mysqli_stmt_execute($getTransStmt);
        $transResult = mysqli_stmt_get_result($getTransStmt);
        $transData = mysqli_fetch_assoc($transResult);
        $transId = $transData['trans_id'] ?? null;
        mysqli_stmt_close($getTransStmt);
        
        // 실제 차감액 = 출금요청액 (전체 복구)
        $actualAmountToRevert = $withdrawAmount;
        $feeAmount = 1000;
        
        // member 테이블의 잔액 복구 (출금요청액 전체)
        $balanceQuery = "UPDATE member SET amount = amount + ? WHERE id = ?";
        $balanceStmt = mysqli_prepare($con, $balanceQuery);
        mysqli_stmt_bind_param($balanceStmt, "di", $actualAmountToRevert, $memberId);
        
        if (!mysqli_stmt_execute($balanceStmt)) {
            throw new Exception("잔액 복구 실패: " . mysqli_error($con));
        }
        mysqli_stmt_close($balanceStmt);
        
        // 업데이트된 잔액 조회
        $getBalanceQuery = "SELECT amount FROM member WHERE id = ?";
        $getBalanceStmt = mysqli_prepare($con, $getBalanceQuery);
        mysqli_stmt_bind_param($getBalanceStmt, "i", $memberId);
        mysqli_stmt_execute($getBalanceStmt);
        $balanceResult = mysqli_stmt_get_result($getBalanceStmt);
        $newBalance = mysqli_fetch_assoc($balanceResult)['amount'];
        mysqli_stmt_close($getBalanceStmt);
        
        // transaction_history에서 내역 삭제
        if ($transId) {
            $deleteHistoryQuery = "DELETE FROM transaction_history WHERE trans_id = ?";
            $deleteHistoryStmt = mysqli_prepare($con, $deleteHistoryQuery);
            mysqli_stmt_bind_param($deleteHistoryStmt, "s", $transId);
            mysqli_stmt_execute($deleteHistoryStmt);
            mysqli_stmt_close($deleteHistoryStmt);
            
            // transfer 테이블의 trans_id를 NULL로 초기화
            $clearTransIdQuery = "UPDATE transfer SET trans_id = NULL WHERE id = ?";
            $clearTransIdStmt = mysqli_prepare($con, $clearTransIdQuery);
            mysqli_stmt_bind_param($clearTransIdStmt, "i", $withdrawal_id);
            mysqli_stmt_execute($clearTransIdStmt);
            mysqli_stmt_close($clearTransIdStmt);
        }
        
        // 로그 기록
        if (function_exists('writeLog')) {
            $balanceOldData = [
                'trans_id' => $transId,
                'member_id' => $memberId,
                'member_user_id' => $memberUserId,
                'balance' => $currentBalance,
                'transfer_id' => $withdrawal_id
            ];
            
            $balanceNewData = [
                'trans_id' => $transId,
                'member_id' => $memberId,
                'member_user_id' => $memberUserId,
                'balance' => $newBalance,
                'amount_reverted' => $actualAmountToRevert,
                'fee_amount' => $feeAmount,
                'transfer_id' => $withdrawal_id
            ];
            
            $balanceDetail = "출금 취소 - " . ($transId ? "Trans ID: {$transId}, " : "") . 
                            "사용자: {$userName}({$memberUserId}), " .
                            "출금요청액: " . number_format($withdrawAmount) . "원, " .
                            "복구액: " . number_format($actualAmountToRevert) . "원, " .
                            "잔액: " . number_format($currentBalance) . "원 → " . number_format($newBalance) . "원";
            writeLog($con, 'WITHDRAWAL_BALANCE_ADD', $balanceOldData, $balanceNewData, 1, $balanceDetail, $adminMemberId);
        }
    }
    
    // 트랜잭션 커밋
    mysqli_commit($con);
    
    // 성공 응답
    $responseData = [
        'success' => true,
        'message' => '출금 신청이 성공적으로 처리되었습니다.',
        'data' => [
            'id' => $withdrawal_id,
            'old_status' => $current_status,
            'new_status' => $action,
            'processed_at' => date('Y-m-d H:i:s')
        ]
    ];
    
    // 잔액 정보 추가
    if ($newBalance !== null) {
        $responseData['data']['new_balance'] = $newBalance;
        
        if ($action === 'completed' && $actualAmount !== null) {
            $responseData['data']['requested_amount'] = $withdrawAmount;
            $responseData['data']['fee_rate'] = $memberRate;
            $responseData['data']['fee_amount'] = $feeAmount;
            $responseData['data']['actual_amount'] = $actualAmount;
        } else if ($action === 'cancelled' && $actualAmountToRevert !== null) {
            $responseData['data']['reverted_amount'] = $actualAmountToRevert;
            $responseData['data']['fee_amount'] = $feeAmount ?? 0;
        }
    }
    
    echo json_encode($responseData, JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    mysqli_rollback($con);
    
    error_log("Process Withdrawal Error: " . $e->getMessage());
    
    // 에러 all_log 기록
    $errorData = [
        'withdrawal_id' => $withdrawal_id ?? null,
        'action' => $action ?? null,
        'error' => $e->getMessage()
    ];
    
    $errorDetail = "출금 처리 오류 - Withdrawal ID: " . ($withdrawal_id ?? 'N/A') . ", 시도한 액션: " . ($action ?? 'N/A') . ", 오류: " . $e->getMessage();
    writeLog($con, 'WITHDRAWAL_PROCESS_ERROR', null, $errorData, 0, $errorDetail, $adminMemberId ?? 0);
    
    // 에러 응답
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>