<?php
// 디버깅을 위한 로그
error_log("save_withdraw_settings.php 실행 시작");
error_log("REQUEST_METHOD: " . $_SERVER['REQUEST_METHOD']);
error_log("Content-Type: " . ($_SERVER['CONTENT_TYPE'] ?? 'not set'));

header('Content-Type: application/json; charset=utf-8');
session_start();

error_log("세션 정보: " . print_r($_SESSION, true));

// 로그인 확인
if (!isset($_SESSION['id'])) {
    writeLog($con, 'WITHDRAW_SETTINGS_DENIED', null, null, 0, 
            '출금 설정 변경 시도 실패 - 미인증 사용자');
    
    echo json_encode(['success' => false, 'message' => '로그인이 필요합니다.']);
    exit;
}

// POST 요청만 허용
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => '잘못된 요청 방법입니다.']);
    exit;
}

// 데이터베이스 연결
require_once '../inc/db_connect.php';

try {
    // 현재 로그인한 사용자 정보 조회
    $currentUserId = $_SESSION['id'];
    $currentUserQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $currentUserQuery);
    mysqli_stmt_bind_param($stmt, "s", $currentUserId);
    mysqli_stmt_execute($stmt);
    $currentUserResult = mysqli_stmt_get_result($stmt);
    $currentUserData = mysqli_fetch_assoc($currentUserResult);
    mysqli_stmt_close($stmt);

    if (!$currentUserData) {
        throw new Exception('사용자 정보를 찾을 수 없습니다.');
    }

    $currentMemberId = $currentUserData['id'];
    $currentUserName = $currentUserData['user_name'];

    // 관리자 권한 확인 (출금 설정은 관리자만)
    if ($currentUserData['grade'] !== 'admin') {
        writeLog($con, 'WITHDRAW_SETTINGS_DENIED', null, null, 0, 
                "출금 설정 변경 권한 없음 - 시도한 사용자: {$currentUserName}({$currentUserId}), 등급: {$currentUserData['grade']}", 
                $currentMemberId);
        
        echo json_encode(['success' => false, 'message' => '관리자 권한이 필요합니다.']);
        exit;
    }

    // JSON 데이터 받기
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (!$data) {
        writeLog($con, 'WITHDRAW_SETTINGS_FAILED', null, null, 0, 
                '출금 설정 변경 실패 - 잘못된 데이터 형식', $currentMemberId);
        throw new Exception('잘못된 데이터 형식입니다.');
    }
    
    // 데이터 검증 및 정제
    $partner_enabled = isset($data['partner_withdraw_block_enabled']) ? (int)$data['partner_withdraw_block_enabled'] : 0;
    $partner_start = $data['partner_withdraw_block_start'] ?? null;
    $partner_end = $data['partner_withdraw_block_end'] ?? null;
    $basic_enabled = isset($data['basic_withdraw_block_enabled']) ? (int)$data['basic_withdraw_block_enabled'] : 0;
    $basic_start = $data['basic_withdraw_block_start'] ?? null;
    $basic_end = $data['basic_withdraw_block_end'] ?? null;
    $hourly_enabled = isset($data['hourly_withdraw_block_enabled']) ? (int)$data['hourly_withdraw_block_enabled'] : 0;

    // 입력 데이터 로그용
    $inputData = [
        'partner_withdraw_block_enabled' => $partner_enabled,
        'partner_withdraw_block_start' => $partner_start,
        'partner_withdraw_block_end' => $partner_end,
        'basic_withdraw_block_enabled' => $basic_enabled,
        'basic_withdraw_block_start' => $basic_start,
        'basic_withdraw_block_end' => $basic_end,
        'hourly_withdraw_block_enabled' => $hourly_enabled,
        'modified_by' => $currentUserName
    ];
    
    // 시간 형식 정제 함수
    function formatTime($time) {
        if (empty($time)) return null;
        
        // HH:MM:SS 형식을 HH:MM으로 변환
        if (preg_match('/^(\d{2}):(\d{2}):(\d{2})$/', $time, $matches)) {
            return $matches[1] . ':' . $matches[2];
        }
        
        // 이미 HH:MM 형식이면 그대로 반환
        if (preg_match('/^(\d{2}):(\d{2})$/', $time)) {
            return $time;
        }
        
        return null;
    }
    
    // 시간 형식 정제
    $partner_start = formatTime($partner_start);
    $partner_end = formatTime($partner_end);
    $basic_start = formatTime($basic_start);
    $basic_end = formatTime($basic_end);
    
    // 활성화된 경우 시간 유효성 검사
    if ($partner_enabled && (empty($partner_start) || empty($partner_end))) {
        writeLog($con, 'WITHDRAW_SETTINGS_FAILED', $inputData, null, 0, 
                '출금 설정 변경 실패 - 총판 출금 차단 시간 누락', $currentMemberId);
        throw new Exception('총판 출금 차단이 활성화된 경우 시작 시간과 종료 시간을 모두 입력해야 합니다.');
    }
    
    if ($basic_enabled && (empty($basic_start) || empty($basic_end))) {
        writeLog($con, 'WITHDRAW_SETTINGS_FAILED', $inputData, null, 0, 
                '출금 설정 변경 실패 - 가맹점 출금 차단 시간 누락', $currentMemberId);
        throw new Exception('가맹점 출금 차단이 활성화된 경우 시작 시간과 종료 시간을 모두 입력해야 합니다.');
    }
    
    // 자정 넘나드는 시간 처리 함수 추가
    function isValidTimeRange($start, $end) {
        if (empty($start) || empty($end)) {
            return false;
        }
        
        $startTime = strtotime($start);
        $endTime = strtotime($end);
        
        // 자정을 넘나드는 경우 (종료시간이 시작시간보다 작은 경우)
        if ($endTime <= $startTime) {
            // 종료시간에 24시간(하루)을 더해서 다음날로 처리
            $endTime += 24 * 60 * 60;
        }
        
        return true; // 자정 넘나드는 시간도 유효함
    }

    // 기존 시간 형식 검증 후에 추가
    if ($partner_enabled) {
        if (!isValidTimeRange($partner_start, $partner_end)) {
            writeLog($con, 'WITHDRAW_SETTINGS_FAILED', $inputData, null, 0, 
                    '출금 설정 변경 실패 - 총판 시간 설정 오류', $currentMemberId);
            throw new Exception('총판 시간 설정이 올바르지 않습니다.');
        }
    }

    if ($basic_enabled) {
        if (!isValidTimeRange($basic_start, $basic_end)) {
            writeLog($con, 'WITHDRAW_SETTINGS_FAILED', $inputData, null, 0, 
                    '출금 설정 변경 실패 - 가맹점 시간 설정 오류', $currentMemberId);
            throw new Exception('가맹점 시간 설정이 올바르지 않습니다.');
        }
    }
    
    // 비활성화된 경우 시간을 NULL로 설정
    if (!$partner_enabled) {
        $partner_start = null;
        $partner_end = null;
    }
    
    if (!$basic_enabled) {
        $basic_start = null;
        $basic_end = null;
    }
    
    // 트랜잭션 시작
    mysqli_begin_transaction($con);
    
    // 기존 설정 조회 (로그용)
    $oldSettingsQuery = "SELECT * FROM settings LIMIT 1";
    $oldSettingsResult = mysqli_query($con, $oldSettingsQuery);
    $oldSettings = mysqli_fetch_assoc($oldSettingsResult);
    
    // 기존 설정이 있는지 확인
    $checkQuery = "SELECT id FROM settings LIMIT 1";
    $checkResult = mysqli_query($con, $checkQuery);
    
    if (mysqli_num_rows($checkResult) > 0) {
        // 업데이트 - 매시간 차단 필드 추가
        $updateQuery = "UPDATE settings SET 
                          partner_withdraw_block_enabled = ?,
                          partner_withdraw_block_start = ?,
                          partner_withdraw_block_end = ?,
                          basic_withdraw_block_enabled = ?,
                          basic_withdraw_block_start = ?,
                          basic_withdraw_block_end = ?,
                          hourly_withdraw_block_enabled = ?,
                          updated_at = NOW()
                        WHERE id = (SELECT id FROM (SELECT id FROM settings LIMIT 1) as temp)";
        
        $stmt = mysqli_prepare($con, $updateQuery);
        
        // 타입 수정: 7개 파라미터에 맞게 "isissi" -> "issisii"
        mysqli_stmt_bind_param($stmt, "issisii", 
            $partner_enabled, $partner_start, $partner_end,
            $basic_enabled, $basic_start, $basic_end,
            $hourly_enabled
        );
    } else {
        // 새로 삽입 - 매시간 차단 필드 추가
        $insertQuery = "INSERT INTO settings (
                          partner_withdraw_block_enabled,
                          partner_withdraw_block_start,
                          partner_withdraw_block_end,
                          basic_withdraw_block_enabled,
                          basic_withdraw_block_start,
                          basic_withdraw_block_end,
                          hourly_withdraw_block_enabled,
                          created_at,
                          updated_at
                        ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), NOW())";
        
        $stmt = mysqli_prepare($con, $insertQuery);
        
        // 타입 수정: 7개 파라미터에 맞게 "issisii"
        mysqli_stmt_bind_param($stmt, "issisii", 
            $partner_enabled, $partner_start, $partner_end,
            $basic_enabled, $basic_start, $basic_end,
            $hourly_enabled
        );
    }
    
    if (!mysqli_stmt_execute($stmt)) {
        writeLog($con, 'WITHDRAW_SETTINGS_FAILED', $inputData, $oldSettings, 0, 
                '출금 설정 변경 실패 - DB 오류: ' . mysqli_error($con), $currentMemberId);
        throw new Exception('설정 저장 실패: ' . mysqli_error($con));
    }
    
    // 트랜잭션 커밋
    mysqli_commit($con);
    
    // 새로운 설정 데이터
    $newSettings = [
        'partner_withdraw_block_enabled' => $partner_enabled,
        'partner_withdraw_block_start' => $partner_start,
        'partner_withdraw_block_end' => $partner_end,
        'basic_withdraw_block_enabled' => $basic_enabled,
        'basic_withdraw_block_start' => $basic_start,
        'basic_withdraw_block_end' => $basic_end,
        'hourly_withdraw_block_enabled' => $hourly_enabled,
        'modified_by' => $currentUserName,
        'updated_at' => date('Y-m-d H:i:s')
    ];
    
    // 성공 로그 기록 - currentMemberId 전달
    $detail = "출금 설정 변경 완료 - 총판차단: " . ($partner_enabled ? "활성({$partner_start}~{$partner_end})" : "비활성") . 
             ", 가맹점차단: " . ($basic_enabled ? "활성({$basic_start}~{$basic_end})" : "비활성") . 
             ", 매시간차단: " . ($hourly_enabled ? "활성" : "비활성") . 
             ", 변경자: {$currentUserName}({$currentUserId})";
    
    writeLog($con, 'WITHDRAW_SETTINGS_SUCCESS', $oldSettings, $newSettings, 1, $detail, $currentMemberId);
    
    // 로그 기록 - 매시간 차단 정보 추가
    error_log("출금 설정 변경 - 사용자: {$_SESSION['id']}, 총판차단: {$partner_enabled}, 가맹점차단: {$basic_enabled}, 매시간차단: {$hourly_enabled}");
    
    echo json_encode([
        'success' => true,
        'message' => '설정이 성공적으로 저장되었습니다.',
        'data' => [
            'partner_withdraw_block_enabled' => $partner_enabled,
            'partner_withdraw_block_start' => $partner_start,
            'partner_withdraw_block_end' => $partner_end,
            'basic_withdraw_block_enabled' => $basic_enabled,
            'basic_withdraw_block_start' => $basic_start,
            'basic_withdraw_block_end' => $basic_end,
            'hourly_withdraw_block_enabled' => $hourly_enabled
        ]
    ]);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    if (isset($con)) {
        mysqli_rollback($con);
    }
    
    // 에러 로그 기록
    if (isset($currentMemberId) && isset($inputData)) {
        writeLog($con, 'WITHDRAW_SETTINGS_ERROR', $inputData, $oldSettings ?? null, 0, 
                '출금 설정 변경 오류: ' . $e->getMessage(), $currentMemberId);
    }
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
    
} finally {
    if (isset($stmt)) {
        mysqli_stmt_close($stmt);
    }
    if (isset($con)) {
        mysqli_close($con);
    }
}
?>