<?php
// 에러 표시 설정
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크 - user_id로 체크 (문자열)
if (!isset($_SESSION['id'])) {
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => '접근 권한이 없습니다.'
    ]);
    exit;
}

try {
    // 현재 로그인한 사용자의 user_id (문자열)
    $sessionUserId = $_SESSION['id'];
    
    // 사용자 정보 가져오기 (grade 확인 및 member.id 가져오기)
    $userQuery = "SELECT id, grade, user_name FROM member WHERE user_id = ?";
    $userStmt = mysqli_prepare($con, $userQuery);
    mysqli_stmt_bind_param($userStmt, "s", $sessionUserId);
    mysqli_stmt_execute($userStmt);
    $userResult = mysqli_stmt_get_result($userStmt);
    $userData = mysqli_fetch_assoc($userResult);
    mysqli_stmt_close($userStmt);
    
    if (!$userData) {
        throw new Exception("사용자 정보를 찾을 수 없습니다.");
    }
    
    // member 테이블의 id (정수) - transfer 테이블과 연결되는 키
    $memberId = $userData['id'];
    $isAdmin = ($userData['grade'] === 'admin');
    
    // 오늘 날짜
    $today = date('Y-m-d');
    
    // WHERE 조건 설정 (관리자면 전체, 아니면 본인 것만)
    $userCondition = $isAdmin ? "" : " AND t.user_id = ?";
    
    // 1. 오늘 완료된 입금 건수와 총액 (status = 'completed'만 집계)
    $depositStatsQuery = "
        SELECT 
            COUNT(*) as count, 
            COALESCE(SUM(amount), 0) as total 
        FROM transfer t
        WHERE t.type = 'deposit' 
        AND t.status = 'completed'
        AND DATE(t.created_at) = ?" . $userCondition;
    
    $depositStmt = mysqli_prepare($con, $depositStatsQuery);
    if ($isAdmin) {
        mysqli_stmt_bind_param($depositStmt, "s", $today);
    } else {
        mysqli_stmt_bind_param($depositStmt, "si", $today, $memberId);
    }
    mysqli_stmt_execute($depositStmt);
    $depositResult = mysqli_stmt_get_result($depositStmt);
    $depositStats = mysqli_fetch_assoc($depositResult);
    mysqli_stmt_close($depositStmt);
    
    // 2. 오늘 완료된 출금 건수와 총액 (status = 'completed'만 집계)
    $withdrawStatsQuery = "
        SELECT 
            COUNT(*) as count, 
            COALESCE(SUM(amount), 0) as total 
        FROM transfer t
        WHERE t.type = 'withdraw' 
        AND t.status = 'completed'
        AND DATE(t.created_at) = ?" . $userCondition;
    
    $withdrawStmt = mysqli_prepare($con, $withdrawStatsQuery);
    if ($isAdmin) {
        mysqli_stmt_bind_param($withdrawStmt, "s", $today);
    } else {
        mysqli_stmt_bind_param($withdrawStmt, "si", $today, $memberId);
    }
    mysqli_stmt_execute($withdrawStmt);
    $withdrawResult = mysqli_stmt_get_result($withdrawStmt);
    $withdrawStats = mysqli_fetch_assoc($withdrawResult);
    mysqli_stmt_close($withdrawStmt);
    
    // 3. 최근 입금 신청 내역 (오늘 기준, 최근 7건) - 모든 상태 표시
    $recentDepositsQuery = "
        SELECT 
            t.*,
            m.user_name,
            m.user_id as member_user_id,
            m.code as store_code,
            t.depositor_name
        FROM transfer t
        LEFT JOIN member m ON t.user_id = m.id
        WHERE t.type = 'deposit' 
        AND DATE(t.created_at) = ?" . $userCondition . "
        ORDER BY t.created_at DESC 
        LIMIT 7";
    
    $depositsStmt = mysqli_prepare($con, $recentDepositsQuery);
    if ($isAdmin) {
        mysqli_stmt_bind_param($depositsStmt, "s", $today);
    } else {
        mysqli_stmt_bind_param($depositsStmt, "si", $today, $memberId);
    }
    mysqli_stmt_execute($depositsStmt);
    $depositsResult = mysqli_stmt_get_result($depositsStmt);
    
    $recentDeposits = [];
    while ($row = mysqli_fetch_assoc($depositsResult)) {
        // transfer의 user_id는 member의 id를 참조하므로
        // 화면에 표시할 때는 member_user_id (로그인 ID)를 사용
        $displayRow = [
            'id' => $row['id'],
            'depositor_name' => $row['depositor_name'] ?: '-', // 입금자명
            'user_id' => $row['member_user_id'] ?: 'Unknown', // 로그인 ID
            'user_name' => $row['user_name'] ?: '-', // 가맹점명
            'store_code' => $row['store_code'] ?: '-',
            'amount' => $row['amount'],
            'kakao_tid' => $row['kakao_tid'],
            'status' => $row['status'],
            'payment_status' => $row['payment_status'],
            'created_at' => date('Y-m-d H:i:s', strtotime($row['created_at']))
        ];
        
        $recentDeposits[] = $displayRow;
    }
    mysqli_stmt_close($depositsStmt);
    
    // 4. 최근 출금 신청 내역 (오늘 기준, 최근 7건) - 모든 상태 표시
    $recentWithdrawalsQuery = "
        SELECT 
            t.*,
            m.user_name,
            m.user_id as member_user_id,
            m.code as store_code,
            t.depositor_name
        FROM transfer t
        LEFT JOIN member m ON t.user_id = m.id
        WHERE t.type = 'withdraw' 
        AND DATE(t.created_at) = ?" . $userCondition . "
        ORDER BY t.created_at DESC 
        LIMIT 7";
    
    $withdrawalsStmt = mysqli_prepare($con, $recentWithdrawalsQuery);
    if ($isAdmin) {
        mysqli_stmt_bind_param($withdrawalsStmt, "s", $today);
    } else {
        mysqli_stmt_bind_param($withdrawalsStmt, "si", $today, $memberId);
    }
    mysqli_stmt_execute($withdrawalsStmt);
    $withdrawalsResult = mysqli_stmt_get_result($withdrawalsStmt);
    
    $recentWithdrawals = [];
    while ($row = mysqli_fetch_assoc($withdrawalsResult)) {
        // transfer의 user_id는 member의 id를 참조하므로
        // 화면에 표시할 때는 member_user_id (로그인 ID)를 사용
        $displayRow = [
            'id' => $row['id'],
            'depositor_name' => $row['depositor_name'] ?: '-', // 입금자명 (출금시에도 동일 필드 사용)
            'user_id' => $row['member_user_id'] ?: 'Unknown', // 로그인 ID
            'user_name' => $row['user_name'] ?: '-', // 가맹점명
            'store_code' => $row['store_code'] ?: '-',
            'amount' => $row['amount'],
            'kakao_tid' => $row['kakao_tid'],
            'status' => $row['status'],
            'payment_status' => $row['payment_status'],
            'created_at' => date('Y-m-d H:i:s', strtotime($row['created_at']))
        ];
        
        $recentWithdrawals[] = $displayRow;
    }
    mysqli_stmt_close($withdrawalsStmt);
    
    // 응답 데이터
    $response = [
        'success' => true,
        'isAdmin' => $isAdmin,
        'currentUser' => $userData['user_name'],
        'userGrade' => $userData['grade'],
        'todayDeposits' => (int)$depositStats['count'],
        'todayDepositAmount' => (float)$depositStats['total'],
        'todayWithdrawals' => (int)$withdrawStats['count'],
        'todayWithdrawalAmount' => (float)$withdrawStats['total'],
        'recentDeposits' => $recentDeposits,
        'recentWithdrawals' => $recentWithdrawals
    ];
    
    // 디버깅용 (필요시 활성화)
    if (isset($_GET['debug'])) {
        $response['debug'] = [
            'session_user_id' => $sessionUserId,
            'member_id' => $memberId,
            'is_admin' => $isAdmin,
            'grade' => $userData['grade']
        ];
    }
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 에러 로깅
    error_log("Transfer Stats Error: " . $e->getMessage());
    
    // 에러 응답
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => '데이터 조회 중 오류가 발생했습니다.',
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>