<?php
// 에러 표시 (디버깅용 - 운영환경에서는 주석처리)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크 - 로그인된 사용자만
if (!isset($_SESSION['id'])) {
    writeLog($con, 'BANK_ACCOUNT_UPDATE_DENIED', null, null, 0, 
            '은행 계좌 수정 시도 실패 - 미인증 사용자');
    
    http_response_code(403);
    echo json_encode([
        'success' => false, 
        'message' => '접근 권한이 없습니다.',
        'error_code' => 'AUTH_FAILED'
    ]);
    exit;
}

try {
    // 현재 로그인한 사용자 정보 조회
    $currentUserId = $_SESSION['id'];
    $currentUserQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $currentUserQuery);
    mysqli_stmt_bind_param($stmt, "s", $currentUserId);
    mysqli_stmt_execute($stmt);
    $currentUserResult = mysqli_stmt_get_result($stmt);
    $currentUserData = mysqli_fetch_assoc($currentUserResult);
    mysqli_stmt_close($stmt);

    if (!$currentUserData) {
        throw new Exception('사용자 정보를 찾을 수 없습니다.');
    }

    $currentMemberId = $currentUserData['id'];
    $currentUserName = $currentUserData['user_name'];

    // JSON 데이터 받기
    $inputData = json_decode(file_get_contents('php://input'), true);

    // 필수 파라미터 확인
    if (!isset($inputData['id']) || empty($inputData['id'])) {
        writeLog($con, 'BANK_ACCOUNT_UPDATE_FAILED', $inputData, null, 0, 
                '은행 계좌 수정 실패 - 계좌 ID 누락', $currentMemberId);
        
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => '계좌 ID가 필요합니다.',
            'error_code' => 'MISSING_ID'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $id = intval($inputData['id']);
    $bank = isset($inputData['bank_name']) ? trim($inputData['bank_name']) : null;
    $account_number = isset($inputData['account_number']) ? trim($inputData['account_number']) : null;
    $account_holder = isset($inputData['account_holder']) ? trim($inputData['account_holder']) : null;

    // 입력 데이터 로그용
    $requestData = [
        'id' => $id,
        'bank_name' => $bank,
        'account_number' => $account_number,
        'account_holder' => $account_holder,
        'modified_by' => $currentUserName
    ];

    // 유효성 검사
    $errors = [];

    // 계좌번호가 있을 경우 숫자만 입력 검증
    if (!empty($account_number) && !preg_match('/^\d+$/', $account_number)) {
        $errors[] = '계좌번호는 숫자만 입력 가능합니다.';
    }

    // 입력값 길이 체크
    if (!empty($bank) && strlen($bank) > 100) {
        $errors[] = '은행명은 100자 이내로 입력해주세요.';
    }
    if (!empty($account_number) && strlen($account_number) > 100) {
        $errors[] = '계좌번호는 100자 이내로 입력해주세요.';
    }
    if (!empty($account_holder) && strlen($account_holder) > 100) {
        $errors[] = '예금주명은 100자 이내로 입력해주세요.';
    }

    // 에러가 있으면 반환
    if (!empty($errors)) {
        $errorDetail = implode(' ', $errors);
        writeLog($con, 'BANK_ACCOUNT_UPDATE_FAILED', $requestData, null, 0, 
                "은행 계좌 수정 실패 - 유효성 검사: {$errorDetail}", $currentMemberId);
        
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => $errorDetail,
            'errors' => $errors
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // 트랜잭션 시작
    beginTransaction($con);
    
    // 1. 기존 데이터 조회 (로그용)
    $selectQuery = "SELECT * FROM bank_account WHERE id = ?";
    $selectStmt = mysqli_prepare($con, $selectQuery);
    
    if (!$selectStmt) {
        throw new Exception("SELECT 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($selectStmt, "i", $id);
    mysqli_stmt_execute($selectStmt);
    $result = mysqli_stmt_get_result($selectStmt);
    $oldData = mysqli_fetch_assoc($result);
    
    if (!$oldData) {
        mysqli_stmt_close($selectStmt);
        writeLog($con, 'BANK_ACCOUNT_UPDATE_FAILED', $requestData, null, 0, 
                "은행 계좌 수정 실패 - 존재하지 않는 계좌 ID: {$id}", $currentMemberId);
        throw new Exception("존재하지 않는 은행 계좌입니다.");
    }
    
    mysqli_stmt_close($selectStmt);
    
    // 2. 중복 체크 - 다른 계좌가 같은 은행의 같은 계좌번호를 사용하는지
    if (!empty($bank) && !empty($account_number)) {
        $checkQuery = "SELECT id FROM bank_account WHERE bank = ? AND account_number = ? AND id != ?";
        $checkStmt = mysqli_prepare($con, $checkQuery);
        
        if (!$checkStmt) {
            throw new Exception("중복 체크 쿼리 준비 실패: " . mysqli_error($con));
        }
        
        mysqli_stmt_bind_param($checkStmt, "ssi", $bank, $account_number, $id);
        mysqli_stmt_execute($checkStmt);
        mysqli_stmt_store_result($checkStmt);
        
        if (mysqli_stmt_num_rows($checkStmt) > 0) {
            mysqli_stmt_close($checkStmt);
            writeLog($con, 'BANK_ACCOUNT_UPDATE_FAILED', $requestData, $oldData, 0, 
                    "은행 계좌 수정 실패 - 중복된 계좌번호: {$bank} {$account_number}", $currentMemberId);
            throw new Exception("이미 등록된 계좌번호입니다.");
        }
        
        mysqli_stmt_close($checkStmt);
    }
    
    // 3. 업데이트 쿼리 구성 (변경된 필드만 업데이트)
    $updateFields = [];
    $updateValues = [];
    $updateTypes = "";
    
    if (!empty($bank)) {
        $updateFields[] = "bank = ?";
        $updateValues[] = $bank;
        $updateTypes .= "s";
    }
    
    if (!empty($account_number)) {
        $updateFields[] = "account_number = ?";
        $updateValues[] = $account_number;
        $updateTypes .= "s";
    }
    
    if (!empty($account_holder)) {
        $updateFields[] = "account_holder = ?";
        $updateValues[] = $account_holder;
        $updateTypes .= "s";
    }
    
    // 변경사항이 없으면 에러
    if (empty($updateFields)) {
        writeLog($con, 'BANK_ACCOUNT_UPDATE_FAILED', $requestData, $oldData, 0, 
                "은행 계좌 수정 실패 - 변경할 내용 없음", $currentMemberId);
        throw new Exception("변경할 내용이 없습니다.");
    }
    
    // updated_at 추가
    $updateFields[] = "updated_at = NOW()";
    
    // 4. 업데이트 실행
    $updateQuery = "UPDATE bank_account SET " . implode(", ", $updateFields) . " WHERE id = ?";
    $updateValues[] = $id;
    $updateTypes .= "i";
    
    $updateStmt = mysqli_prepare($con, $updateQuery);
    
    if (!$updateStmt) {
        throw new Exception("UPDATE 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($updateStmt, $updateTypes, ...$updateValues);
    
    if (!mysqli_stmt_execute($updateStmt)) {
        throw new Exception("데이터 수정 실패: " . mysqli_stmt_error($updateStmt));
    }
    
    mysqli_stmt_close($updateStmt);
    
    // 5. 수정된 데이터 조회
    $newSelectQuery = "SELECT * FROM bank_account WHERE id = ?";
    $newSelectStmt = mysqli_prepare($con, $newSelectQuery);
    mysqli_stmt_bind_param($newSelectStmt, "i", $id);
    mysqli_stmt_execute($newSelectStmt);
    $newResult = mysqli_stmt_get_result($newSelectStmt);
    $newData = mysqli_fetch_assoc($newResult);
    mysqli_stmt_close($newSelectStmt);
    
    // 트랜잭션 커밋
    commitTransaction($con);
    
    // 성공 로그 기록 - currentMemberId 전달
    $detail = "은행 계좌 수정 성공 - ID: {$id}, 은행: {$newData['bank']}, " .
             "계좌: {$newData['account_number']}, 예금주: {$newData['account_holder']}, " .
             "수정자: {$currentUserName}({$currentUserId})";
    
    writeLog($con, 'BANK_ACCOUNT_UPDATE_SUCCESS', $oldData, $newData, 1, $detail, $currentMemberId);
    
    // 성공 응답
    echo json_encode([
        'success' => true,
        'message' => '은행 계좌가 성공적으로 수정되었습니다.',
        'id' => $id,
        'data' => $newData
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    rollbackTransaction($con);
    
    $errorMessage = $e->getMessage();
    
    // 실패 로그 기록 - currentMemberId 전달
    if (isset($currentMemberId) && isset($requestData)) {
        writeLog($con, 'BANK_ACCOUNT_UPDATE_FAILED', $requestData, $oldData ?? null, 0, 
                "은행 계좌 수정 실패 - " . $errorMessage, $currentMemberId);
    }
    
    // 에러 응답
    http_response_code(500);
    
    // 중복 에러인 경우 409 Conflict
    if (strpos($errorMessage, '이미 등록된') !== false) {
        http_response_code(409);
    }
    
    // 존재하지 않는 경우 404 Not Found
    if (strpos($errorMessage, '존재하지 않는') !== false) {
        http_response_code(404);
    }
    
    echo json_encode([
        'success' => false,
        'message' => $errorMessage,
        'error_code' => 'UPDATE_FAILED'
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>