<?php
session_start();
header('Content-Type: application/json; charset=utf-8');

// 트랜잭션 ID 생성 함수
function generateTransactionId() {
    return 'TXN' . date('YmdHis') . sprintf('%06d', mt_rand(0, 999999));
}

try {
    // DB 연결
    include "../inc/db_connect.php";
    
    if (!isset($con)) {
        throw new Exception('DB 연결 객체 없음');
    }

    // 권한 확인 - 어드민만 가능
    if (!isset($_SESSION['grade']) || $_SESSION['grade'] !== 'admin') {
        throw new Exception('권한이 없습니다.');
    }

    // 관리자 정보
    $adminCheckQuery = "SELECT id FROM member WHERE user_id = ?";
    $adminCheckStmt = mysqli_prepare($con, $adminCheckQuery);
    mysqli_stmt_bind_param($adminCheckStmt, "s", $_SESSION['id']);
    mysqli_stmt_execute($adminCheckStmt);
    $adminResult = mysqli_stmt_get_result($adminCheckStmt);
    $adminData = mysqli_fetch_assoc($adminResult);
    $adminMemberId = $adminData['id'];
    mysqli_stmt_close($adminCheckStmt);

    // POST 데이터 받기
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['id']) || !isset($input['amount'])) {
        throw new Exception('필수 파라미터가 누락되었습니다.');
    }

    $id = intval($input['id']);
    $newAmount = floatval($input['amount']);

    // 금액 검증
    if ($newAmount <= 0) {
        throw new Exception('올바른 금액을 입력해주세요.');
    }

    // 트랜잭션 시작
    mysqli_begin_transaction($con);

    // 기존 데이터 확인 (회원 정보와 parents 정보 포함)
    $checkQuery = "SELECT t.id, t.user_id, t.amount, t.status, t.trans_id,
                          m.amount as member_balance, m.rate as fee_rate, m.user_name, 
                          m.user_id as member_user_id, m.parents
                   FROM transfer t
                   JOIN member m ON t.user_id = m.id
                   WHERE t.id = ? AND t.type = 'deposit'";
    $checkStmt = mysqli_prepare($con, $checkQuery);
    mysqli_stmt_bind_param($checkStmt, "i", $id);
    mysqli_stmt_execute($checkStmt);
    $checkResult = mysqli_stmt_get_result($checkStmt);
    
    if (mysqli_num_rows($checkResult) == 0) {
        throw new Exception('해당 입금 내역을 찾을 수 없습니다.');
    }

    $oldData = mysqli_fetch_assoc($checkResult);
    $oldAmount = $oldData['amount'];
    $memberId = $oldData['user_id'];  // member 테이블의 id
    $memberUserId = $oldData['member_user_id'];
    $userName = $oldData['user_name'];
    $status = $oldData['status'];
    $transId = $oldData['trans_id'];
    $currentBalance = $oldData['member_balance'];
    $feeRate = $oldData['fee_rate'] ?? 0;
    $parentsJson = $oldData['parents'] ?? '[]';
    
    mysqli_stmt_close($checkStmt);

    // 1. transfer 테이블 금액 업데이트 (모든 상태에서 수행)
    $updateQuery = "UPDATE transfer SET amount = ?, updated_at = NOW() WHERE id = ?";
    $updateStmt = mysqli_prepare($con, $updateQuery);
    mysqli_stmt_bind_param($updateStmt, "di", $newAmount, $id);
    
    if (!mysqli_stmt_execute($updateStmt)) {
        throw new Exception('금액 수정에 실패했습니다.');
    }
    mysqli_stmt_close($updateStmt);

    // 로그용 데이터
    $logOldData = [
        'transfer_id' => $id,
        'amount' => $oldAmount,
        'status' => $status,
        'trans_id' => $transId,
        'member_balance' => $currentBalance
    ];

    $logNewData = [
        'transfer_id' => $id,
        'amount' => $newAmount,
        'status' => $status,
        'trans_id' => $transId
    ];

    // 2. 상태가 completed인 경우 다단계 롤링 처리
    $memberBalanceUpdated = false;
    $newBalance = null;
    $actualDifference = 0;
    $rollingDifferences = [];
    
    if ($status === 'completed') {
        // 기존 수수료 및 롤링 계산
        $oldTotalFeeAmount = round($oldAmount * ($feeRate / 100), 2);
        $oldActualAmount = $oldAmount - $oldTotalFeeAmount;
        
        // 새로운 수수료 및 롤링 계산
        $newTotalFeeAmount = round($newAmount * ($feeRate / 100), 2);
        $newActualAmount = $newAmount - $newTotalFeeAmount;
        
        // 실제 차액 (수수료 차감 후)
        $actualDifference = $newActualAmount - $oldActualAmount;
        
        // ===== 다단계 롤링 차액 계산 =====
        $parentsArray = json_decode($parentsJson, true);
        if (!is_array($parentsArray)) {
            $parentsArray = [];
        }
        
        $oldTotalRollingAmount = 0;
        $newTotalRollingAmount = 0;
        
        if (count($parentsArray) > 0) {
            $currentMemberId = $memberId;
            $currentRate = $feeRate;
            
            // parents 배열을 역순으로 처리 (직속상위부터 최상위까지)
            $reversedParents = array_reverse($parentsArray);
            
            foreach ($reversedParents as $parentId) {
                // 상위 회원 정보 조회
                $parentInfoQuery = "SELECT id, rate, user_name, user_id FROM member WHERE id = ?";
                $parentInfoStmt = mysqli_prepare($con, $parentInfoQuery);
                mysqli_stmt_bind_param($parentInfoStmt, "i", $parentId);
                mysqli_stmt_execute($parentInfoStmt);
                $parentInfoResult = mysqli_stmt_get_result($parentInfoStmt);
                
                if ($parentInfo = mysqli_fetch_assoc($parentInfoResult)) {
                    $parentRate = floatval($parentInfo['rate']);
                    $parentName = $parentInfo['user_name'];
                    $parentUserId = $parentInfo['user_id'];
                    
                    if ($currentRate > $parentRate) {
                        $rollingRate = $currentRate - $parentRate;
                        
                        // 기존 롤링액과 새 롤링액 계산
                        $oldRollingAmount = round($oldAmount * ($rollingRate / 100), 2);
                        $newRollingAmount = round($newAmount * ($rollingRate / 100), 2);
                        $rollingDifference = $newRollingAmount - $oldRollingAmount;
                        
                        if ($rollingDifference != 0) {
                            // 차감 시 잔액 체크
                            if ($rollingDifference < 0) {
                                $parentBalanceQuery = "SELECT amount FROM member WHERE id = ?";
                                $parentBalanceStmt = mysqli_prepare($con, $parentBalanceQuery);
                                mysqli_stmt_bind_param($parentBalanceStmt, "i", $parentId);
                                mysqli_stmt_execute($parentBalanceStmt);
                                $parentBalanceResult = mysqli_stmt_get_result($parentBalanceStmt);
                                $parentBalance = mysqli_fetch_assoc($parentBalanceResult)['amount'];
                                mysqli_stmt_close($parentBalanceStmt);
                                
                                if ($parentBalance < abs($rollingDifference)) {
                                    throw new Exception(
                                        sprintf("상위 회원(%s) 잔액 부족: 현재 잔액(%s원)에서 %s원을 차감할 수 없습니다.",
                                            $parentName,
                                            number_format($parentBalance),
                                            number_format(abs($rollingDifference))
                                        )
                                    );
                                }
                            }
                            
                            // 상위 회원 잔액 조정
                            $updateParentBalanceQuery = "UPDATE member SET amount = amount + ? WHERE id = ?";
                            $updateParentBalanceStmt = mysqli_prepare($con, $updateParentBalanceQuery);
                            mysqli_stmt_bind_param($updateParentBalanceStmt, "di", $rollingDifference, $parentId);
                            
                            if (!mysqli_stmt_execute($updateParentBalanceStmt)) {
                                throw new Exception("상위 회원({$parentName}) 잔액 업데이트 실패");
                            }
                            mysqli_stmt_close($updateParentBalanceStmt);
                            
                            $rollingDifferences[] = [
                                'parent_member_id' => $parentId,
                                'parent_name' => $parentName,
                                'parent_user_id' => $parentUserId,
                                'child_member_id' => $currentMemberId,  // 현재 처리중인 멤버 ID
                                'rolling_rate' => $rollingRate,
                                'old_rolling_amount' => $oldRollingAmount,
                                'new_rolling_amount' => $newRollingAmount,
                                'rolling_difference' => $rollingDifference
                            ];
                        }
                        
                        $oldTotalRollingAmount += $oldRollingAmount;
                        $newTotalRollingAmount += $newRollingAmount;
                    }
                    
                    // 다음 단계를 위해 현재 정보를 상위 정보로 업데이트
                    $currentMemberId = $parentId;
                    $currentRate = $parentRate;
                }
                mysqli_stmt_close($parentInfoStmt);
            }
        }
        
        // 회사 커미션 계산
        $oldCompanyFeeAmount = $oldTotalFeeAmount - $oldTotalRollingAmount;
        $newCompanyFeeAmount = $newTotalFeeAmount - $newTotalRollingAmount;
        $companyFeeRate = ($newAmount > 0) ? ($newCompanyFeeAmount / $newAmount) * 100 : 0;
        
        // transaction_history 테이블 업데이트 (trans_id가 있는 경우)
        if ($transId) {
            // 기존 transaction_history 확인
            $historyCheckQuery = "SELECT id FROM transaction_history WHERE trans_id = ?";
            $historyCheckStmt = mysqli_prepare($con, $historyCheckQuery);
            mysqli_stmt_bind_param($historyCheckStmt, "s", $transId);
            mysqli_stmt_execute($historyCheckStmt);
            $historyResult = mysqli_stmt_get_result($historyCheckStmt);
            
            if (mysqli_num_rows($historyResult) > 0) {
                $historyData = mysqli_fetch_assoc($historyResult);
                $transactionHistoryId = $historyData['id'];
                
                // transaction_history 업데이트
                $updateHistoryQuery = "UPDATE transaction_history 
                                        SET requested_amount = ?,
                                            total_fee_rate = ?,
                                            total_fee_amount = ?,
                                            fee_rate = ?,
                                            fee_amount = ?,
                                            actual_amount = ?,
                                            balance_after = balance_before + ?  -- 새로운 actual_amount
                                        WHERE trans_id = ?";
                $updateHistoryStmt = mysqli_prepare($con, $updateHistoryQuery);
                mysqli_stmt_bind_param($updateHistoryStmt, "ddddddds", 
                $newAmount, 
                $feeRate, 
                $newTotalFeeAmount, 
                $companyFeeRate, 
                $newCompanyFeeAmount,
                $newActualAmount, 
                $newActualAmount,  // actualDifference 대신 newActualAmount 사용
                $transId
            );
                mysqli_stmt_execute($updateHistoryStmt);
                mysqli_stmt_close($updateHistoryStmt);
                
                // rolling_history 업데이트 - 모든 관련 레코드 업데이트
                if (!empty($rollingDifferences)) {
                    foreach ($rollingDifferences as $rollingDiff) {
                        // 기존 롤링 내역 확인 및 업데이트
                        $checkRollingQuery = "SELECT id FROM rolling_history 
                                            WHERE original_transaction_id = ? 
                                            AND parent_member_id = ? 
                                            AND child_member_id = ?";
                        $checkRollingStmt = mysqli_prepare($con, $checkRollingQuery);
                        mysqli_stmt_bind_param($checkRollingStmt, "iii", 
                            $transactionHistoryId,
                            $rollingDiff['parent_member_id'],
                            $rollingDiff['child_member_id']
                        );
                        mysqli_stmt_execute($checkRollingStmt);
                        $checkRollingResult = mysqli_stmt_get_result($checkRollingStmt);
                        
                        if (mysqli_num_rows($checkRollingResult) > 0) {
                            // 기존 레코드가 있으면 UPDATE
                            $updateRollingQuery = "UPDATE rolling_history 
                                                 SET rolling_rate = ?,
                                                     rolling_amount = ?,
                                                     updated_at = NOW()
                                                 WHERE original_transaction_id = ? 
                                                 AND parent_member_id = ? 
                                                 AND child_member_id = ?";
                            $updateRollingStmt = mysqli_prepare($con, $updateRollingQuery);
                            mysqli_stmt_bind_param($updateRollingStmt, "ddiii", 
                                $rollingDiff['rolling_rate'],
                                $rollingDiff['new_rolling_amount'],
                                $transactionHistoryId,
                                $rollingDiff['parent_member_id'],
                                $rollingDiff['child_member_id']
                            );
                            mysqli_stmt_execute($updateRollingStmt);
                            mysqli_stmt_close($updateRollingStmt);
                        }
                        // INSERT 로직 제거 - UPDATE만 수행
                        
                        mysqli_stmt_close($checkRollingStmt);
                    }
                }
            }
            mysqli_stmt_close($historyCheckStmt);
        }
        
        // member 테이블의 잔액 조정 (실제 차액만큼)
        if ($actualDifference != 0) {
            // 차감 시 잔액 체크
            if ($actualDifference < 0 && $currentBalance < abs($actualDifference)) {
                throw new Exception(
                    sprintf("잔액 부족: 현재 잔액(%s원)에서 %s원을 차감할 수 없습니다.",
                        number_format($currentBalance),
                        number_format(abs($actualDifference))
                    )
                );
            }
            
            $balanceQuery = "UPDATE member SET amount = amount + ? WHERE id = ?";
            $balanceStmt = mysqli_prepare($con, $balanceQuery);
            mysqli_stmt_bind_param($balanceStmt, "di", $actualDifference, $memberId);
            
            if (!mysqli_stmt_execute($balanceStmt)) {
                throw new Exception('회원 잔액 업데이트에 실패했습니다.');
            }
            mysqli_stmt_close($balanceStmt);
            
            // 업데이트된 잔액 조회
            $getBalanceQuery = "SELECT amount FROM member WHERE id = ?";
            $getBalanceStmt = mysqli_prepare($con, $getBalanceQuery);
            mysqli_stmt_bind_param($getBalanceStmt, "i", $memberId);
            mysqli_stmt_execute($getBalanceStmt);
            $balanceResult = mysqli_stmt_get_result($getBalanceStmt);
            $newBalance = mysqli_fetch_assoc($balanceResult)['amount'];
            mysqli_stmt_close($getBalanceStmt);
            
            $memberBalanceUpdated = true;
            
            // 로그 데이터 업데이트
            $logNewData['member_balance'] = $newBalance;
            $logNewData['actual_difference'] = $actualDifference;
            $logNewData['old_actual_amount'] = $oldActualAmount;
            $logNewData['new_actual_amount'] = $newActualAmount;
            $logNewData['rolling_differences'] = $rollingDifferences;
            $logNewData['old_total_rolling'] = $oldTotalRollingAmount;
            $logNewData['new_total_rolling'] = $newTotalRollingAmount;
        }
    }

    // all_log 기록
    if (function_exists('writeLog')) {
        $detail = sprintf("입금액 수정 - ID: %d, 사용자: %s(%s), 변경: %s원 → %s원",
            $id, $userName, $memberUserId,
            number_format($oldAmount), number_format($newAmount)
        );
        
        if ($memberBalanceUpdated) {
            $detail .= sprintf(", 실지급액 차이: %s원, 잔액: %s원 → %s원",
                number_format($actualDifference),
                number_format($currentBalance),
                number_format($newBalance)
            );
            
            if (!empty($rollingDifferences)) {
                $detail .= ", 롤링조정: ";
                $rollingDetails = [];
                foreach ($rollingDifferences as $rd) {
                    $rollingDetails[] = sprintf("%s %s원",
                        $rd['parent_name'],
                        number_format($rd['rolling_difference'])
                    );
                }
                $detail .= implode(', ', $rollingDetails);
            }
        }
        
        writeLog($con, 'DEPOSIT_AMOUNT_UPDATE', $logOldData, $logNewData, 1, $detail, $adminMemberId);
    }

    // 트랜잭션 커밋
    mysqli_commit($con);

    // 응답 데이터 구성
    $response = [
        'success' => true,
        'message' => '금액이 성공적으로 수정되었습니다.',
        'data' => [
            'transfer_id' => $id,
            'old_amount' => $oldAmount,
            'new_amount' => $newAmount,
            'status' => $status
        ]
    ];
    
    // 상태별 추가 정보
    if ($status === 'completed') {
        $response['data']['fee_rate'] = $feeRate;
        $response['data']['old_fee'] = $oldTotalFeeAmount ?? 0;
        $response['data']['new_fee'] = $newTotalFeeAmount ?? 0;
        $response['data']['old_actual'] = $oldActualAmount ?? $oldAmount;
        $response['data']['new_actual'] = $newActualAmount ?? $newAmount;
        $response['data']['actual_difference'] = $actualDifference;
        $response['data']['old_total_rolling'] = $oldTotalRollingAmount ?? 0;
        $response['data']['new_total_rolling'] = $newTotalRollingAmount ?? 0;
        $response['data']['rolling_adjustments'] = count($rollingDifferences);
        
        if ($memberBalanceUpdated) {
            $response['data']['member_balance_updated'] = true;
            $response['data']['new_balance'] = $newBalance;
            
            $message = sprintf('실지급액 %s원 %s, 회원 잔액: %s원', 
                number_format(abs($actualDifference)), 
                $actualDifference > 0 ? '증가' : '감소',
                number_format($newBalance)
            );
            
            if (!empty($rollingDifferences)) {
                $message .= sprintf(', %d단계 롤링 조정됨', count($rollingDifferences));
            }
            
            $response['message'] .= ' (' . $message . ')';
        }
    } else {
        $response['data']['note'] = sprintf('%s 상태이므로 transfer 테이블의 금액만 수정되었습니다.', $status);
    }

    echo json_encode($response, JSON_UNESCAPED_UNICODE);

} catch (Exception $e) {
    // 트랜잭션 롤백
    if (isset($con)) {
        mysqli_rollback($con);
    }
    
    // 에러 로그
    if (function_exists('writeLog') && isset($adminMemberId)) {
        $errorData = [
            'transfer_id' => $id ?? null,
            'error' => $e->getMessage()
        ];
        writeLog($con, 'DEPOSIT_AMOUNT_UPDATE_ERROR', null, $errorData, 0, 
                "입금액 수정 오류: " . $e->getMessage(), $adminMemberId);
    }
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // DB 연결 종료
    if (isset($con)) {
        mysqli_close($con);
    }
}
?>