<?php
// 에러 표시 (디버깅용 - 운영환경에서는 주석처리)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// 세션 시작
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// CORS 헤더 설정
header('Content-Type: application/json; charset=UTF-8');

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// 권한 체크 - 로그인된 사용자만
if (!isset($_SESSION['id'])) {
    writeLog($con, 'KAKAO_ACCOUNT_UPDATE_DENIED', null, null, 0, 
            '카카오 계정 수정 시도 실패 - 미인증 사용자');
    
    http_response_code(403);
    echo json_encode([
        'success' => false, 
        'message' => '접근 권한이 없습니다.',
        'error_code' => 'AUTH_FAILED'
    ]);
    exit;
}

try {
    // 현재 로그인한 사용자 정보 조회
    $currentUserId = $_SESSION['id'];
    $currentUserQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $currentUserQuery);
    mysqli_stmt_bind_param($stmt, "s", $currentUserId);
    mysqli_stmt_execute($stmt);
    $currentUserResult = mysqli_stmt_get_result($stmt);
    $currentUserData = mysqli_fetch_assoc($currentUserResult);
    mysqli_stmt_close($stmt);

    if (!$currentUserData) {
        throw new Exception('사용자 정보를 찾을 수 없습니다.');
    }

    $currentMemberId = $currentUserData['id'];
    $currentUserName = $currentUserData['user_name'];

    // JSON 데이터 받기
    $inputData = json_decode(file_get_contents('php://input'), true);

    // 데이터 추출
    $id = isset($inputData['id']) ? intval($inputData['id']) : 0;
    $kakao_tid = isset($inputData['kakao_tid']) ? trim($inputData['kakao_tid']) : '';
    $name = isset($inputData['name']) ? trim($inputData['name']) : '';
    $qr_link = isset($inputData['qr_link']) ? trim($inputData['qr_link']) : '';
    $friend_link = isset($inputData['friend_link']) ? trim($inputData['friend_link']) : '';
    $default_link = isset($inputData['default_link']) ? trim($inputData['default_link']) : '';

    // 입력 데이터 로그용
    $requestData = [
        'id' => $id,
        'kakao_tid' => $kakao_tid,
        'name' => $name,
        'qr_link' => $qr_link,
        'friend_link' => $friend_link,
        'default_link' => $default_link,
        'modified_by' => $currentUserName
    ];

    // 유효성 검사
    $errors = [];

    if (empty($id) || $id <= 0) {
        $errors[] = '유효하지 않은 계정 ID입니다.';
    }
    if (empty($kakao_tid)) {
        $errors[] = '카카오 TID를 입력해주세요.';
    }
    if (empty($name)) {
        $errors[] = '계정명을 입력해주세요.';
    }

    // 입력값 길이 체크
    if (strlen($kakao_tid) > 100) {
        $errors[] = '카카오 TID는 100자 이내로 입력해주세요.';
    }
    if (strlen($name) > 45) {
        $errors[] = '계정명은 45자 이내로 입력해주세요.';
    }

    // URL 유효성 검사 (값이 있을 때만)
    if (empty($qr_link)) {
        $errors[] = 'QR 링크가 필수입니다.';
    }
    if (empty($friend_link)) {
        $errors[] = '친구 링크가 필수입니다.';
    }

    // 에러가 있으면 반환
    if (!empty($errors)) {
        $errorDetail = implode(' ', $errors);
        writeLog($con, 'KAKAO_ACCOUNT_UPDATE_FAILED', $requestData, null, 0, 
                "카카오 계정 수정 실패 - 유효성 검사: {$errorDetail}", $currentMemberId);
        
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => $errorDetail,
            'errors' => $errors
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    if (!empty($default_link) && !empty($qr_link)) {
        try {
            $qr_data = json_decode($qr_link, true);
            if (is_array($qr_data)) {
                $qr_data['default'] = $default_link;
                $qr_link = json_encode($qr_data);
            }
        } catch (Exception $e) {
            // 오류 시 무시하고 계속 진행
        }
    }

    // 트랜잭션 시작
    beginTransaction($con);

    // 1. 기존 데이터 조회 (로그용)
    $selectQuery = "SELECT * FROM kakao_account WHERE id = ?";
    $selectStmt = mysqli_prepare($con, $selectQuery);
    
    if (!$selectStmt) {
        throw new Exception("기존 데이터 조회 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($selectStmt, "i", $id);
    mysqli_stmt_execute($selectStmt);
    
    $result = mysqli_stmt_get_result($selectStmt);
    $oldData = mysqli_fetch_assoc($result);
    
    if (!$oldData) {
        mysqli_stmt_close($selectStmt);
        writeLog($con, 'KAKAO_ACCOUNT_UPDATE_FAILED', $requestData, null, 0, 
                "카카오 계정 수정 실패 - 존재하지 않는 계정 ID: {$id}", $currentMemberId);
        throw new Exception("존재하지 않는 카카오 계정입니다.");
    }
    
    mysqli_stmt_close($selectStmt);

    // 2. 중복 체크
    // 2-1. kakao_tid 중복 체크
    $checkTidQuery = "SELECT id, name FROM kakao_account WHERE kakao_tid = ? AND id != ?";
    $checkTidStmt = mysqli_prepare($con, $checkTidQuery);

    if (!$checkTidStmt) {
        throw new Exception("카카오 TID 중복 체크 쿼리 준비 실패: " . mysqli_error($con));
    }

    mysqli_stmt_bind_param($checkTidStmt, "si", $kakao_tid, $id);
    mysqli_stmt_execute($checkTidStmt);
    $tidResult = mysqli_stmt_get_result($checkTidStmt);

    if ($duplicateTid = mysqli_fetch_assoc($tidResult)) {
        mysqli_stmt_close($checkTidStmt);
        writeLog($con, 'KAKAO_ACCOUNT_UPDATE_FAILED', $requestData, $oldData, 0, 
                "카카오 계정 수정 실패 - 중복된 TID: {$kakao_tid} (사용중인 계정: {$duplicateTid['name']})", $currentMemberId);
        throw new Exception("카카오 TID '{$kakao_tid}'는 이미 '{$duplicateTid['name']}' 계정에서 사용중입니다.");
    }
    mysqli_stmt_close($checkTidStmt);

    // 2-2. name 중복 체크
    $checkNameQuery = "SELECT id FROM kakao_account WHERE name = ? AND id != ?";
    $checkNameStmt = mysqli_prepare($con, $checkNameQuery);

    if (!$checkNameStmt) {
        throw new Exception("계정명 중복 체크 쿼리 준비 실패: " . mysqli_error($con));
    }

    mysqli_stmt_bind_param($checkNameStmt, "si", $name, $id);
    mysqli_stmt_execute($checkNameStmt);
    mysqli_stmt_store_result($checkNameStmt);

    if (mysqli_stmt_num_rows($checkNameStmt) > 0) {
        mysqli_stmt_close($checkNameStmt);
        writeLog($con, 'KAKAO_ACCOUNT_UPDATE_FAILED', $requestData, $oldData, 0, 
                "카카오 계정 수정 실패 - 중복된 계정명: {$name}", $currentMemberId);
        throw new Exception("계정명 '{$name}'은 이미 사용중입니다.");
    }
    mysqli_stmt_close($checkNameStmt);
    
    // 3. 업데이트 실행
    $updateQuery = "
        UPDATE kakao_account 
        SET kakao_tid = ?, 
            name = ?, 
            qr_link = ?, 
            friend_link = ?,
            updated_at = NOW()
        WHERE id = ?
    ";
    
    $updateStmt = mysqli_prepare($con, $updateQuery);
    
    if (!$updateStmt) {
        throw new Exception("업데이트 쿼리 준비 실패: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($updateStmt, "ssssi", 
        $kakao_tid, 
        $name, 
        $qr_link, 
        $friend_link, 
        $id
    );
    
    if (!mysqli_stmt_execute($updateStmt)) {
        throw new Exception("카카오 계정 수정 실패: " . mysqli_stmt_error($updateStmt));
    }
    
    mysqli_stmt_close($updateStmt);
    
    // 트랜잭션 커밋
    commitTransaction($con);
    
    // 새 데이터 준비 (로그용)
    $newData = [
        'id' => $id,
        'kakao_tid' => $kakao_tid,
        'name' => $name,
        'qr_link' => $qr_link,
        'friend_link' => $friend_link,
        'modified_by' => $currentUserName,
        'updated_at' => date('Y-m-d H:i:s')
    ];
    
    // 성공 로그 기록 - currentMemberId 전달
    $detail = "카카오 계정 수정 성공 - ID: {$id}, 계정명: {$name}, TID: {$kakao_tid}, 수정자: {$currentUserName}({$currentUserId})";
    writeLog($con, 'KAKAO_ACCOUNT_UPDATE_SUCCESS', $oldData, $newData, 1, $detail, $currentMemberId);
    
    // 성공 응답
    echo json_encode([
        'success' => true,
        'message' => '카카오 계정이 성공적으로 수정되었습니다.',
        'id' => $id,
        'data' => $newData
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    rollbackTransaction($con);
    
    $errorMessage = $e->getMessage();
    
    // 실패 로그 기록 - currentMemberId 전달
    if (isset($currentMemberId) && isset($requestData)) {
        writeLog($con, 'KAKAO_ACCOUNT_UPDATE_FAILED', $requestData, $oldData ?? null, 0, 
                "카카오 계정 수정 실패 - " . $errorMessage, $currentMemberId);
    }
    
    // 에러 응답
    http_response_code(500);
    
    // 존재하지 않는 계정인 경우 404
    if (strpos($errorMessage, '존재하지 않는') !== false) {
        http_response_code(404);
    }
    // 중복 에러인 경우 409
    elseif (strpos($errorMessage, '이미 사용중') !== false) {
        http_response_code(409);
    }
    
    echo json_encode([
        'success' => false,
        'message' => $errorMessage,
        'error_code' => 'UPDATE_FAILED'
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
}
?>