<?php
header('Content-Type: application/json; charset=utf-8');

// POST 요청만 허용
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => '잘못된 요청 방법입니다.']);
    exit;
}

// 세션 시작 및 로그인 확인
session_start();
if (!isset($_SESSION["id"]) || empty($_SESSION["id"])) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => '로그인이 필요합니다.']);
    exit;
}

// 데이터베이스 연결
require_once '../inc/db_connect.php';

// MySQLi 연결 확인
if (!isset($con) || !$con) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => '데이터베이스 연결 오류가 발생했습니다.']);
    exit;
}

try {
    // 현재 로그인한 사용자 정보 조회
    $currentUserId = $_SESSION["id"];
    $currentUserQuery = "SELECT id, user_id, user_name, grade FROM member WHERE user_id = ?";
    $stmt = mysqli_prepare($con, $currentUserQuery);
    mysqli_stmt_bind_param($stmt, "s", $currentUserId);
    mysqli_stmt_execute($stmt);
    $currentUserResult = mysqli_stmt_get_result($stmt);
    $currentUserData = mysqli_fetch_assoc($currentUserResult);
    mysqli_stmt_close($stmt);

    if (!$currentUserData) {
        throw new Exception('사용자 정보를 찾을 수 없습니다.');
    }

    $currentMemberId = $currentUserData['id'];
    $currentUserName = $currentUserData['user_name'];

    // 입력 데이터 검증 및 정제
    $user_id = trim($_POST['user_id'] ?? '');
    $user_name = trim($_POST['user_name'] ?? '');
    $password = trim($_POST['password'] ?? ''); // 비밀번호 추가
    $code = trim($_POST['code'] ?? '');
    $grade = trim($_POST['grade'] ?? '');
    $kakao_account_id = intval($_POST['kakao_account_id'] ?? 0);
    $kakao_link_type = trim($_POST['kakao_link_type'] ?? '');
    $bank_account_id = intval($_POST['bank_account_id'] ?? 0);
    $bank = trim($_POST['bank'] ?? '');
    $account_number = trim($_POST['account_number'] ?? '');
    $account_holder = trim($_POST['account_holder'] ?? '');
    $rate = floatval($_POST['rate'] ?? 0);

    // 입력 데이터 로그용 (비밀번호는 로그에서 제외)
    $inputData = [
        'user_id' => $user_id,
        'user_name' => $user_name,
        'password_changed' => !empty($password) ? 'YES' : 'NO',
        'code' => $code,
        'grade' => $grade,
        'kakao_account_id' => $kakao_account_id,
        'kakao_link_type' => $kakao_link_type,
        'bank_account_id' => $bank_account_id,
        'bank' => $bank,
        'account_number' => $account_number,
        'account_holder' => $account_holder,
        'rate' => $rate,
        'modified_by' => $currentUserName
    ];

    // 필수 필드 검증
    if (empty($user_id)) {
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, null, 0, 
                "가맹점 수정 실패 - 가맹점 아이디 누락", $currentMemberId);
        throw new Exception('가맹점 아이디가 필요합니다.');
    }
    if (empty($user_name)) {
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, null, 0, 
                "가맹점 수정 실패 - 가맹점명 누락", $currentMemberId);
        throw new Exception('가맹점명을 입력해주세요.');
    }
    if (empty($code)) {
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, null, 0, 
                "가맹점 수정 실패 - 가맹점 코드 누락", $currentMemberId);
        throw new Exception('가맹점 코드를 입력해주세요.');
    }
    if (empty($grade)) {
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, null, 0, 
                "가맹점 수정 실패 - 등급 누락", $currentMemberId);
        throw new Exception('등급을 선택해주세요.');
    }

    // 비밀번호 유효성 검사 (입력된 경우만)
    if (!empty($password)) {
        if (strlen($password) < 4) {
            writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, null, 0, 
                    "가맹점 수정 실패 - 비밀번호가 너무 짧음", $currentMemberId);
            throw new Exception('비밀번호는 최소 4자 이상이어야 합니다.');
        }
        if (strlen($password) > 50) {
            writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, null, 0, 
                    "가맹점 수정 실패 - 비밀번호가 너무 김", $currentMemberId);
            throw new Exception('비밀번호는 50자 이하여야 합니다.');
        }
    }

    // 등급 유효성 검사
    $validGrades = ['basic', 'partner', 'admin'];
    if (!in_array($grade, $validGrades)) {
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, null, 0, 
                "가맹점 수정 실패 - 유효하지 않은 등급: {$grade}", $currentMemberId);
        throw new Exception('유효하지 않은 등급입니다.');
    }

    // 기존 가맹점 정보 조회 (수정 전 데이터)
    $user_id_escaped = mysqli_real_escape_string($con, $user_id);
    $originalQuery = "SELECT * FROM member WHERE user_id = '$user_id_escaped'";
    $originalResult = mysqli_query($con, $originalQuery);
    
    if (!$originalResult || mysqli_num_rows($originalResult) === 0) {
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, null, 0, 
                "가맹점 수정 실패 - 존재하지 않는 가맹점: {$user_id}", $currentMemberId);
        throw new Exception('존재하지 않는 가맹점입니다.');
    }
    
    $originalData = mysqli_fetch_assoc($originalResult);
    $memberId = $originalData['id'];
    $parents = json_decode($originalData['parents'] ?? '[]', true);

    // 권한 체크 - 자신의 정보이거나 관리자만 수정 가능
    if ($currentMemberId != $memberId && $currentUserData['grade'] !== 'admin') {
        writeLog($con, 'MEMBER_UPDATE_DENIED', $inputData, $originalData, 0, 
                "가맹점 수정 권한 없음 - 시도한 사용자: {$currentUserName}({$currentUserId}), 대상: {$user_id}", 
                $currentMemberId);
        throw new Exception('해당 가맹점을 수정할 권한이 없습니다.');
    }

    // 카카오 계정과 은행 연결계정 검증
    if ($kakao_account_id <= 0 && $bank_account_id <= 0) {
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, $originalData, 0, 
                "가맹점 수정 실패 - 연결계정 미선택", $currentMemberId);
        throw new Exception('카카오 연결계정 또는 은행 연결계정 중 하나를 선택해주세요.');
    }

    if ($kakao_account_id > 0 && $bank_account_id > 0) {
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, $originalData, 0, 
                "가맹점 수정 실패 - 카카오/은행 계정 중복 선택", $currentMemberId);
        throw new Exception('카카오 연결계정과 은행 연결계정 중 하나만 선택할 수 있습니다.');
    }

    // 카카오 계정 선택 시 링크 타입 필수 확인
    if ($kakao_account_id > 0 && empty($kakao_link_type)) {
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, $originalData, 0, 
                "가맹점 수정 실패 - 카카오 링크 타입 누락", $currentMemberId);
        throw new Exception('카카오 링크 타입을 선택해주세요.');
    }

    // 은행 계정 선택 시 카카오 링크 타입 초기화
    if ($bank_account_id > 0) {
        $kakao_link_type = '';
    }

    // 개별 출금정보 필수 체크
    if (empty($bank) || empty($account_number) || empty($account_holder)) {
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, $originalData, 0, 
                "가맹점 수정 실패 - 출금정보 누락", $currentMemberId);
        throw new Exception('개별 출금정보(은행명, 계좌번호, 예금주)를 모두 입력해주세요.');
    }

    // 입력값 유효성 검사
    if (strlen($user_name) > 100) {
        throw new Exception('가맹점명은 100자 이하여야 합니다.');
    }
    if (strlen($code) > 20) {
        throw new Exception('가맹점 코드는 20자 이하여야 합니다.');
    }
    if ($rate < 0 || $rate > 100) {
        throw new Exception('수수료율은 0~100% 사이여야 합니다.');
    }

    // 계좌번호 형식 검증
    if (!preg_match('/^[0-9-]+$/', $account_number)) {
        throw new Exception('계좌번호는 숫자와 하이픈(-)만 사용할 수 있습니다.');
    }

    // 상위 가맹점의 수수료율 체크
    if (!empty($parents) && is_array($parents)) {
        $parentId = $parents[0];
        $parentQuery = "SELECT rate, user_name FROM member WHERE id = " . intval($parentId);
        $parentResult = mysqli_query($con, $parentQuery);
        
        if ($parentResult && $parentRow = mysqli_fetch_assoc($parentResult)) {
            $parentRate = floatval($parentRow['rate']);
            
            if ($rate < $parentRate) {
                writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, $originalData, 0, 
                        "가맹점 수정 실패 - 수수료율 오류: 수정값({$rate}%) < 상위({$parentRate}%)", $currentMemberId);
                throw new Exception("수수료율은 상위 가맹점의 수수료율({$parentRate}%)보다 높거나 같아야 합니다.");
            }
        }
    }
    
    // 하위 가맹점들의 수수료율 체크
    $childrenQuery = "SELECT user_id, user_name, rate FROM member WHERE JSON_CONTAINS(parents, '$memberId', '$')";
    $childrenResult = mysqli_query($con, $childrenQuery);
    
    if ($childrenResult && mysqli_num_rows($childrenResult) > 0) {
        $invalidChildren = [];
        while ($child = mysqli_fetch_assoc($childrenResult)) {
            if (floatval($child['rate']) < $rate) {
                $invalidChildren[] = "{$child['user_name']}({$child['rate']}%)";
            }
        }
        
        if (!empty($invalidChildren)) {
            $childrenList = implode(', ', $invalidChildren);
            writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, $originalData, 0, 
                    "가맹점 수정 실패 - 하위 가맹점 수수료율 충돌: {$childrenList}", $currentMemberId);
            throw new Exception("다음 하위 가맹점의 수수료율이 설정하려는 수수료율({$rate}%)보다 낮습니다: {$childrenList}. 먼저 하위 가맹점의 수수료율을 조정해주세요.");
        }
    }

    // 다른 가맹점에서 같은 코드를 사용하는지 확인
    $code_escaped = mysqli_real_escape_string($con, $code);
    $duplicateCheckSql = "SELECT user_id FROM member WHERE code = '$code_escaped' AND user_id != '$user_id_escaped'";
    $duplicateResult = mysqli_query($con, $duplicateCheckSql);
    
    if ($duplicateResult && mysqli_num_rows($duplicateResult) > 0) {
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, $originalData, 0, 
                "가맹점 수정 실패 - 중복된 가맹점 코드: {$code}", $currentMemberId);
        throw new Exception('이미 사용 중인 가맹점 코드입니다.');
    }

    // 트랜잭션 시작
    mysqli_autocommit($con, false);
    mysqli_begin_transaction($con);

    // 데이터 이스케이프
    $user_name_escaped = mysqli_real_escape_string($con, $user_name);
    $bank_escaped = mysqli_real_escape_string($con, $bank);
    $account_number_escaped = mysqli_real_escape_string($con, $account_number);
    $account_holder_escaped = mysqli_real_escape_string($con, $account_holder);
    $kakao_link_type_escaped = mysqli_real_escape_string($con, $kakao_link_type);
    $grade_escaped = mysqli_real_escape_string($con, $grade);

    // 등급에 따른 레벨 설정
    $level = 1; // 기본값
    switch ($grade) {
        case 'basic':
            $level = 1;
            break;
        case 'partner':
            $level = 2;
            break;
        case 'admin':
            $level = 3;
            break;
    }

    // 비밀번호 처리
    $passwordUpdatePart = '';
    if (!empty($password)) {
        // $password_hashed = password_hash($password, PASSWORD_DEFAULT);
        $password_escaped = mysqli_real_escape_string($con, $password);
        $passwordUpdatePart = "password = '$password',";
    }

    // 가맹점 정보 업데이트
    $updateSql = "UPDATE member SET 
                    user_name = '$user_name_escaped',
                    $passwordUpdatePart
                    code = '$code_escaped',
                    grade = '$grade_escaped',
                    level = $level,
                    kakao_account_id = " . ($kakao_account_id > 0 ? $kakao_account_id : "0") . ",
                    kakao_link_type = " . (!empty($kakao_link_type) ? "'$kakao_link_type_escaped'" : "NULL") . ",
                    bank_account_id = " . ($bank_account_id > 0 ? $bank_account_id : "0") . ",
                    bank = '$bank_escaped',
                    account_number = '$account_number_escaped',
                    account_holder = '$account_holder_escaped',
                    rate = $rate,
                    updated_at = NOW()
                  WHERE user_id = '$user_id_escaped'";

    $result = mysqli_query($con, $updateSql);

    if ($result) {
        $affected_rows = mysqli_affected_rows($con);
        
        // 수정된 데이터 구성 (비밀번호는 로그에서 제외)
        $updatedData = [
            'user_id' => $user_id,
            'user_name' => $user_name,
            'password_changed' => !empty($password) ? 'YES' : 'NO',
            'code' => $code,
            'grade' => $grade,
            'level' => $level,
            'kakao_account_id' => $kakao_account_id,
            'kakao_link_type' => $kakao_link_type,
            'bank_account_id' => $bank_account_id,
            'bank' => $bank,
            'account_number' => $account_number,
            'account_holder' => $account_holder,
            'rate' => $rate,
            'modified_by' => $currentUserName,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        if ($affected_rows > 0) {
            // 성공 로그 기록
            $passwordChangeText = !empty($password) ? ', 비밀번호 변경됨' : '';
            $detail = "가맹점 정보 수정 완료 - ID: {$user_id}, 가맹점명: {$user_name}, 코드: {$code}, " .
                     "등급: {$grade}, 수수료율: {$rate}%{$passwordChangeText}, 수정자: {$currentUserName}({$currentUserId})";
            
            writeLog($con, 'MEMBER_UPDATE_SUCCESS', $originalData, $updatedData, 1, $detail, $currentMemberId);

            // 트랜잭션 커밋
            mysqli_commit($con);

            $responseMessage = '가맹점 정보가 성공적으로 수정되었습니다.';
            if (!empty($password)) {
                $responseMessage .= ' (비밀번호도 변경되었습니다)';
            }

            echo json_encode([
                'success' => true,
                'message' => $responseMessage,
                'data' => $updatedData
            ]);
        } else {
            // 변경사항 없음 로그
            writeLog($con, 'MEMBER_UPDATE_NO_CHANGE', $originalData, $inputData, 1, 
                    "가맹점 정보 수정 시도 - 변경사항 없음: {$user_id}", $currentMemberId);

            mysqli_commit($con);

            echo json_encode([
                'success' => true,
                'message' => '변경된 내용이 없습니다.',
                'data' => ['user_id' => $user_id]
            ]);
        }
    } else {
        $errorMsg = '가맹점 수정 중 오류가 발생했습니다: ' . mysqli_error($con);
        writeLog($con, 'MEMBER_UPDATE_FAILED', $inputData, $originalData, 0, $errorMsg, $currentMemberId);
        
        mysqli_rollback($con);
        throw new Exception($errorMsg);
    }

} catch (Exception $e) {
    // 트랜잭션 롤백
    if (isset($con)) {
        mysqli_rollback($con);
    }
    
    // 에러 로그 기록
    if (isset($currentMemberId) && isset($inputData)) {
        writeLog($con, 'MEMBER_UPDATE_ERROR', $inputData ?? null, $originalData ?? null, 0, 
                $e->getMessage(), $currentMemberId ?? 0);
    }
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
    
} finally {
    // autocommit 복원
    if (isset($con)) {
        mysqli_autocommit($con, true);
        mysqli_close($con);
    }
}
?>