<?php
include 'layout/header.php';
?>

<style>
.setting-card {
  background: rgba(255, 255, 255, 0.05);
  border: 1px solid rgba(255, 255, 255, 0.1);
  border-radius: 12px;
  padding: 24px;
  margin-bottom: 20px;
  transition: all 0.3s ease;
}

.setting-card:hover {
  background: rgba(255, 255, 255, 0.07);
  border-color: rgba(255, 255, 255, 0.15);
}

.time-input-group {
  display: flex;
  gap: 1rem;
  align-items: center;
  flex-wrap: wrap;
}

.time-input-wrapper {
  position: relative;
  flex: 1;
  min-width: 200px;
}

.time-input {
  width: 100%;
  background: rgba(0, 0, 0, 0.3);
  border: 2px solid rgba(255, 255, 255, 0.1);
  border-radius: 8px;
  padding: 12px 16px;
  color: white;
  font-size: 1.125rem;
  font-weight: 600;
  text-align: center;
  transition: all 0.3s ease;
}

.time-input:focus {
  outline: none;
  border-color: #10b981;
  background: rgba(0, 0, 0, 0.4);
}

.time-input::-webkit-inner-spin-button,
.time-input::-webkit-outer-spin-button {
  opacity: 1;
  height: 40px;
}

.time-label {
  position: absolute;
  top: -8px;
  left: 12px;
  background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
  padding: 2px 12px;
  border-radius: 12px;
  font-size: 0.75rem;
  font-weight: 600;
  color: white;
  letter-spacing: 0.05em;
}

.time-unit {
  color: #9ca3af;
  font-size: 1.25rem;
  font-weight: 600;
  min-width: 40px;
}

.current-time-display {
  background: linear-gradient(135deg, rgba(16, 185, 129, 0.1), rgba(5, 150, 105, 0.1));
  border: 2px solid rgba(16, 185, 129, 0.3);
  border-radius: 16px;
  padding: 24px;
  text-align: center;
}

.current-time-value {
  font-size: 3rem;
  font-weight: 700;
  background: linear-gradient(135deg, #10b981, #059669);
  -webkit-background-clip: text;
  -webkit-text-fill-color: transparent;
  background-clip: text;
  margin: 12px 0;
}

.time-range-info {
  display: flex;
  justify-content: space-between;
  align-items: center;
  background: rgba(0, 0, 0, 0.3);
  border: 1px solid rgba(255, 255, 255, 0.1);
  border-radius: 8px;
  padding: 16px;
  margin-top: 16px;
}

.range-item {
  text-align: center;
}

.range-label {
  color: #9ca3af;
  font-size: 0.875rem;
  margin-bottom: 4px;
}

.range-value {
  color: white;
  font-size: 1.25rem;
  font-weight: 700;
}

.preset-buttons {
  display: flex;
  gap: 0.75rem;
  flex-wrap: wrap;
  margin-top: 20px;
}

.preset-btn {
  padding: 10px 20px;
  background: rgba(99, 102, 241, 0.1);
  border: 1px solid rgba(99, 102, 241, 0.3);
  border-radius: 8px;
  color: #818cf8;
  font-weight: 600;
  cursor: pointer;
  transition: all 0.3s ease;
}

.preset-btn:hover {
  background: rgba(99, 102, 241, 0.2);
  border-color: rgba(99, 102, 241, 0.5);
  transform: translateY(-2px);
}

.preset-btn:active {
  transform: translateY(0);
}

.info-box {
  background: linear-gradient(135deg, rgba(59, 130, 246, 0.1), rgba(37, 99, 235, 0.1));
  border: 1px solid rgba(59, 130, 246, 0.3);
  border-radius: 12px;
  padding: 20px;
}

.info-icon {
  font-size: 2rem;
  margin-bottom: 12px;
}

.warning-box {
  background: rgba(245, 158, 11, 0.1);
  border: 1px solid rgba(245, 158, 11, 0.3);
  border-radius: 12px;
  padding: 20px;
}

.save-button {
  background: linear-gradient(135deg, #10b981, #059669);
  color: white;
  padding: 14px 32px;
  border-radius: 10px;
  font-weight: 700;
  font-size: 1.125rem;
  border: none;
  cursor: pointer;
  transition: all 0.3s ease;
  box-shadow: 0 4px 6px rgba(16, 185, 129, 0.3);
}

.save-button:hover {
  transform: translateY(-2px);
  box-shadow: 0 6px 12px rgba(16, 185, 129, 0.4);
}

.save-button:active {
  transform: translateY(0);
}

.error-message {
  color: #ef4444;
  font-size: 0.875rem;
  margin-top: 8px;
  display: none;
}

.error-message.show {
  display: block;
}
</style>

<main class="flex-1 p-8 space-y-8">
  <!-- 헤더 -->
  <div class="flex justify-between items-center mb-6">
    <div>
      <h2 class="text-3xl font-bold text-primary">자동 승인 설정</h2>
      <p class="text-gray-400 mt-2">입출금 요청의 자동 승인 대기 시간을 설정합니다</p>
    </div>
    <button onclick="saveApprovalSettings()" class="save-button">
      <span class="mr-2">💾</span> 설정 저장
    </button>
  </div>

  <!-- 현재 설정 표시 -->
  <div class="setting-card">
    <div class="current-time-display">
      <div class="text-gray-400 text-sm font-semibold uppercase tracking-wider">현재 자동 승인 대기 시간</div>
      <div id="current-display" class="current-time-value">1분 0초</div>
      <div class="text-gray-400 text-sm">설정된 시간이 지나면 자동으로 승인 처리됩니다</div>
    </div>

    <div class="time-range-info">
      <div class="range-item">
        <div class="range-label">최소 시간</div>
        <div class="range-value text-blue-400">1분 0초</div>
      </div>
      <div class="range-item">
        <div class="text-gray-600 text-2xl">━━━</div>
      </div>
      <div class="range-item">
        <div class="range-label">최대 시간</div>
        <div class="range-value text-purple-400">5분 0초</div>
      </div>
    </div>
  </div>

  <!-- 시간 설정 -->
  <div class="setting-card">
    <h3 class="text-xl font-semibold text-white mb-6 flex items-center gap-3">
      <span class="text-2xl">⏱️</span>
      대기 시간 설정
    </h3>

    <div class="time-input-group">
      <div class="time-input-wrapper">
        <span class="time-label">분</span>
        <input 
          type="number" 
          id="minutes-input" 
          class="time-input" 
          min="1" 
          max="5" 
          value="1"
          onchange="updateTimeDisplay()"
          oninput="validateInput()"
        >
      </div>
      <span class="time-unit">분</span>

      <div class="time-input-wrapper">
        <span class="time-label">초</span>
        <input 
          type="number" 
          id="seconds-input" 
          class="time-input" 
          min="0" 
          max="59" 
          value="0"
          onchange="updateTimeDisplay()"
          oninput="validateInput()"
        >
      </div>
      <span class="time-unit">초</span>
    </div>

    <div id="error-message" class="error-message">
      ⚠️ 설정 가능한 시간은 1분 0초부터 5분 0초까지입니다.
    </div>

    <!-- 빠른 설정 버튼 -->
    <div class="mt-6">
      <div class="text-sm text-gray-400 mb-3 font-semibold">⚡ 빠른 설정</div>
      <div class="preset-buttons">
        <button class="preset-btn" onclick="setPresetTime(1, 0)">1분</button>
        <button class="preset-btn" onclick="setPresetTime(1, 20)">1분 20초</button>
        <button class="preset-btn" onclick="setPresetTime(1, 30)">1분 30초</button>
        <button class="preset-btn" onclick="setPresetTime(2, 0)">2분</button>
        <button class="preset-btn" onclick="setPresetTime(2, 30)">2분 30초</button>
        <button class="preset-btn" onclick="setPresetTime(3, 0)">3분</button>
        <button class="preset-btn" onclick="setPresetTime(4, 0)">4분</button>
        <button class="preset-btn" onclick="setPresetTime(4, 20)">4분 20초</button>
        <button class="preset-btn" onclick="setPresetTime(5, 0)">5분</button>
      </div>
    </div>
  </div>

  <!-- 안내 정보 -->
  <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
    <div class="info-box">
      <div class="info-icon">ℹ️</div>
      <h4 class="text-white font-semibold text-lg mb-2">자동 승인이란?</h4>
      <p class="text-gray-300 text-sm leading-relaxed">
        회원의 입출금 요청이 접수된 후 설정된 시간이 지나면 관리자의 별도 확인 없이 자동으로 승인 처리되는 기능입니다. 
        빠른 처리가 필요한 경우 시간을 짧게, 검토가 필요한 경우 시간을 길게 설정하세요.
      </p>
    </div>

    <div class="warning-box">
      <div class="info-icon">⚠️</div>
      <h4 class="text-amber-400 font-semibold text-lg mb-2">주의사항</h4>
      <ul class="text-gray-300 text-sm space-y-2">
        <li class="flex items-start">
          <span class="mr-2">•</span>
          <span>대기 시간이 너무 짧으면 부정 거래 확인이 어려울 수 있습니다.</span>
        </li>
        <li class="flex items-start">
          <span class="mr-2">•</span>
          <span>대기 시간이 너무 길면 회원 경험이 저하될 수 있습니다.</span>
        </li>
        <li class="flex items-start">
          <span class="mr-2">•</span>
          <span>권장 시간은 1분 30초 ~ 3분입니다.</span>
        </li>
      </ul>
    </div>
  </div>

  <!-- 적용 예시 -->
  <div class="setting-card">
    <h3 class="text-xl font-semibold text-white mb-4 flex items-center gap-3">
      <span class="text-2xl">📋</span>
      적용 예시
    </h3>
    <div class="bg-black/30 rounded-lg p-6 border border-gray-700">
      <div class="space-y-3 text-gray-300">
        <div class="flex items-center gap-3">
          <span class="text-green-400 text-xl">✓</span>
          <span><strong class="text-white">14:00:00</strong> - 회원이 출금 요청</span>
        </div>
        <div class="flex items-center gap-3 ml-8">
          <span class="text-yellow-400">↓</span>
          <span class="text-gray-400">대기 시간: <span id="example-time" class="text-yellow-400 font-semibold">1분 0초</span></span>
        </div>
        <div class="flex items-center gap-3">
          <span class="text-blue-400 text-xl">✓</span>
          <span><strong class="text-white" id="example-approval-time">14:01:00</strong> - 자동 승인 처리</span>
        </div>
      </div>
    </div>
  </div>
</main>

<script>
// 페이지 로드 시 현재 설정 가져오기
document.addEventListener('DOMContentLoaded', function() {
  loadApprovalSettings();
});

// 현재 설정 불러오기
function loadApprovalSettings() {
  fetch('ajax/get_approval_settings.php')
    .then(response => response.json())
    .then(data => {
      if (data.success) {
        const totalSeconds = parseInt(data.settings.auto_approval_wait_time);
        const minutes = Math.floor(totalSeconds / 60);
        const seconds = totalSeconds % 60;
        
        document.getElementById('minutes-input').value = minutes;
        document.getElementById('seconds-input').value = seconds;
        
        updateTimeDisplay();
        console.log('자동 승인 설정 로드 완료:', data.settings);
      }
    })
    .catch(error => {
      console.error('설정 로드 실패:', error);
    });
}

// 프리셋 시간 설정
function setPresetTime(minutes, seconds) {
  document.getElementById('minutes-input').value = minutes;
  document.getElementById('seconds-input').value = seconds;
  updateTimeDisplay();
  validateInput();
}

// 시간 표시 업데이트
function updateTimeDisplay() {
  const minutes = parseInt(document.getElementById('minutes-input').value) || 0;
  const seconds = parseInt(document.getElementById('seconds-input').value) || 0;
  
  // 현재 설정 표시
  document.getElementById('current-display').textContent = `${minutes}분 ${seconds}초`;
  
  // 예시 시간 업데이트
  document.getElementById('example-time').textContent = `${minutes}분 ${seconds}초`;
  
  // 승인 예정 시간 계산
  const now = new Date();
  const approvalTime = new Date(now.getTime() + (minutes * 60 + seconds) * 1000);
  const timeString = approvalTime.toTimeString().split(' ')[0];
  document.getElementById('example-approval-time').textContent = timeString;
}

// 입력 검증
function validateInput() {
  const minutes = parseInt(document.getElementById('minutes-input').value) || 0;
  const seconds = parseInt(document.getElementById('seconds-input').value) || 0;
  const totalSeconds = minutes * 60 + seconds;
  
  const errorMessage = document.getElementById('error-message');
  const minutesInput = document.getElementById('minutes-input');
  const secondsInput = document.getElementById('seconds-input');
  
  // 최소 시간: 60초 (1분), 최대 시간: 300초 (5분)
  if (totalSeconds < 60 || totalSeconds > 300) {
    errorMessage.classList.add('show');
    minutesInput.style.borderColor = '#ef4444';
    secondsInput.style.borderColor = '#ef4444';
    return false;
  } else {
    errorMessage.classList.remove('show');
    minutesInput.style.borderColor = 'rgba(255, 255, 255, 0.1)';
    secondsInput.style.borderColor = 'rgba(255, 255, 255, 0.1)';
    return true;
  }
}

// 설정 저장
function saveApprovalSettings() {
  if (!validateInput()) {
    showNotification('❌ 올바른 시간 범위를 입력해주세요. (1분 0초 ~ 5분 0초)', 'error');
    return;
  }
  
  const minutes = parseInt(document.getElementById('minutes-input').value) || 0;
  const seconds = parseInt(document.getElementById('seconds-input').value) || 0;
  const totalSeconds = minutes * 60 + seconds;
  
  const settings = {
    auto_approval_wait_time: totalSeconds
  };
  
  // 로딩 표시
  const saveBtn = event.target;
  const originalText = saveBtn.innerHTML;
  saveBtn.innerHTML = '<span class="mr-2">⏳</span> 저장 중...';
  saveBtn.disabled = true;
  
  fetch('ajax/save_approval_settings.php', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify(settings)
  })
  .then(response => response.json())
  .then(data => {
    if (data.success) {
      showNotification('✅ 자동 승인 설정이 저장되었습니다.', 'success');
      loadApprovalSettings();
    } else {
      showNotification('❌ ' + (data.message || '설정 저장에 실패했습니다.'), 'error');
    }
  })
  .catch(error => {
    console.error('저장 실패:', error);
    showNotification('❌ 네트워크 오류가 발생했습니다.', 'error');
  })
  .finally(() => {
    saveBtn.innerHTML = originalText;
    saveBtn.disabled = false;
  });
}

// 알림 표시 함수
function showNotification(message, type = 'info') {
  const existingNotif = document.querySelector('.notification-toast');
  if (existingNotif) {
    existingNotif.remove();
  }
  
  const notif = document.createElement('div');
  notif.className = 'notification-toast fixed top-4 right-4 p-4 rounded-lg shadow-lg transition-all z-50';
  
  const styles = {
    success: 'bg-green-500 text-white',
    error: 'bg-red-500 text-white',
    info: 'bg-blue-500 text-white'
  };
  
  notif.className += ' ' + styles[type];
  notif.textContent = message;
  
  document.body.appendChild(notif);
  
  setTimeout(() => {
    notif.style.opacity = '0';
    setTimeout(() => notif.remove(), 300);
  }, 3000);
}
</script>

</body>
</html>