<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>은행 송금</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-900 text-white min-h-screen">
    <!-- 로딩 화면 -->
    <div id="loadingScreen" class="fixed inset-0 bg-gray-900 flex items-center justify-center z-50">
        <div class="text-center">
            <div class="w-20 h-20 bg-blue-400 rounded-full flex items-center justify-center mx-auto mb-4">
                <span class="text-3xl">🏦</span>
            </div>
            <div class="w-16 h-16 border-4 border-blue-400 border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
            <p class="text-white text-lg">은행 송금 확인 중...</p>
        </div>
    </div>

    <!-- 승인 완료 화면 -->
    <div id="approvedScreen" class="fixed inset-0 bg-gray-900 flex items-center justify-center z-50 hidden">
        <div class="text-center max-w-md mx-4">
            <div class="w-20 h-20 bg-green-400 rounded-full flex items-center justify-center mx-auto mb-4">
                <span class="text-3xl">✅</span>
            </div>
            <h1 class="text-2xl font-bold text-green-400 mb-4">거래 승인 완료</h1>
            <p class="text-white mb-6">이미 승인된 거래입니다.<br>결제가 완료되었습니다.</p>
            <div class="space-y-3">
                <button onclick="window.close()" 
                        class="w-full bg-gray-600 text-white font-semibold py-3 px-6 rounded-lg hover:bg-gray-700 transition">
                    창 닫기
                </button>
            </div>
        </div>
    </div>

    <!-- 오류 화면 -->
    <div id="errorScreen" class="fixed inset-0 bg-gray-900 flex items-center justify-center z-50 hidden">
        <div class="text-center max-w-md mx-4">
            <div class="w-20 h-20 bg-red-400 rounded-full flex items-center justify-center mx-auto mb-4">
                <span class="text-3xl">❌</span>
            </div>
            <h1 class="text-2xl font-bold text-red-400 mb-4">오류 발생</h1>
            <p id="errorMessage" class="text-white mb-6">오류가 발생했습니다.</p>
            <div class="space-y-3">
                <button onclick="retryPayment()" 
                        class="w-full bg-blue-400 text-gray-900 font-bold py-3 px-6 rounded-lg hover:bg-blue-500 transition">
                    다시 시도
                </button>
                <button onclick="goToPaymentSelect()" 
                        class="w-full bg-green-400 text-gray-900 font-bold py-3 px-6 rounded-lg hover:bg-green-500 transition">
                    결제 페이지로 돌아가기
                </button>
                <button onclick="window.close()" 
                        class="w-full bg-gray-600 text-white font-semibold py-3 px-6 rounded-lg hover:bg-gray-700 transition">
                    닫기
                </button>
            </div>
        </div>
    </div>

    <script>
        // ========== 전역 변수 ==========
        let paymentData = null;
        let storeData = null;
        let countdownTimer = null;
        let isTimerExpired = false;
        let sessionId = null;
        let statusCheckInterval = null;
        let alertModalActive = false;
        let redirectAfterAlert = false;
        let modalBlockingRefresh = false; // 모달 상태에서 새로고침 차단용
        let isRefetchingStoreData = false; // 무한루프 방지용

        // 중복 방지를 위한 변수들
        let isInitializing = false;
        let isTransactionCreating = false;
        let currentTransaction = null;
        let isStatusUpdating = false;
        let transactionFinalized = false;

        //모바일 중복 방지
        let isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
        let mobilePreventDuplicate = false;

        // ========== HTML onclick 함수들 ==========
        function retryPayment() {
            if (transactionFinalized) {
                alert('이미 완료된 거래입니다. 새로운 거래를 원하시면 처음부터 다시 진행해주세요.');
                goToPaymentSelect();
                return;
            }
            
            isInitializing = false;
            isTransactionCreating = false;
            isStatusUpdating = false;
            currentTransaction = null;
            
            document.getElementById('errorScreen').classList.add('hidden');
            document.getElementById('loadingScreen').classList.remove('hidden');
            
            setTimeout(() => {
                initializeBankTransfer();
            }, 1000);
        }

        function goToPaymentSelect() {
            if (statusCheckInterval) clearInterval(statusCheckInterval);
            if (countdownTimer) clearInterval(countdownTimer);
            
            // 정상적으로 세션 정리
            localStorage.removeItem('transactionSessionId');
            
            window.location.href = 'payment_select.php';
        }

        function copyAccountNumber() {
            copyText(window.currentAccountNumber);
        }

        function confirmCurrentTransfer() {
            confirmTransfer(window.currentTransactionId);
        }

        // ========== 페이지 로드 및 초기화 ==========
        document.addEventListener('DOMContentLoaded', function() {
            console.log('페이지 로드 시작');
            
            // 1. 모달이 활성화된 상태에서 새로고침된 경우 체크
            const modalActiveTransaction = localStorage.getItem('modalActiveTransaction');
            const modalActiveTime = localStorage.getItem('modalActiveTime');
            
            if (modalActiveTransaction && modalActiveTime) {
                const timeDiff = Date.now() - parseInt(modalActiveTime);
                
                if (timeDiff < 10000) { // 10초 이내면 모달 상태로 간주
                    console.log('모달 활성화 상태에서 새로고침 감지, 즉시 리다이렉트');
                    localStorage.removeItem('modalActiveTransaction');
                    localStorage.removeItem('modalActiveTime');
                    localStorage.removeItem('transactionSessionId'); // 세션도 정리
                    window.location.replace('payment_select.php');
                    return;
                } else {
                    // 10초 이상 지났으면 정리
                    localStorage.removeItem('modalActiveTransaction');
                    localStorage.removeItem('modalActiveTime');
                }
            }
           // 모바일 전용 중복 방지 강화 (기존 모바일 체크 부분 교체)
            if (isMobile) {
                // 1. 모바일 모달 상태 체크
                const mobileModalBlocking = localStorage.getItem('mobileModalBlocking');
                const mobileModalTime = localStorage.getItem('mobileModalTime');
                const mobileModalActive = sessionStorage.getItem('mobileModalActive');
                
                if (mobileModalBlocking || mobileModalActive) {
                    console.log('모바일 모달 활성화 상태에서 새로고침 감지');
                    localStorage.removeItem('mobileModalBlocking');
                    localStorage.removeItem('mobileModalTime');
                    sessionStorage.removeItem('mobileModalActive');
                    localStorage.removeItem('transactionSessionId');
                    window.location.replace('payment_select.php');
                    return;
                }
                
                // 2. 모바일 완료 거래 체크
                const mobileCompleted = localStorage.getItem('mobileTransactionCompleted');
                const mobileCompletedTime = localStorage.getItem('mobileCompletedTime');
                
                if (mobileCompleted && mobileCompletedTime) {
                    const timeDiff = Date.now() - parseInt(mobileCompletedTime);
                    if (timeDiff < 8000) { // 모바일은 8초로 더 길게
                        console.log('모바일 완료 거래 중복 접근 감지');
                        window.location.replace('payment_select.php');
                        return;
                    } else {
                        localStorage.removeItem('mobileTransactionCompleted');
                        localStorage.removeItem('mobileCompletedTime');
                    }
                }
                
                // 3. 모바일 일반 중복 접근 체크
                const mobileLastAccess = localStorage.getItem('mobileLastAccess');
                const currentTime = Date.now();
                
                if (mobileLastAccess) {
                    const timeDiff = currentTime - parseInt(mobileLastAccess);
                    if (timeDiff < 2000) { // 2초로 짧게
                        console.log('모바일 일반 중복 접근 감지');
                        window.location.replace('payment_select.php');
                        return;
                    }
                }
                
                localStorage.setItem('mobileLastAccess', currentTime.toString());
                setTimeout(() => {
                    localStorage.removeItem('mobileLastAccess');
                }, 2000);
            }
            
            // 2. 기존 중복 방지 체크
            const justCompleted = sessionStorage.getItem('transactionJustCompleted');
            const completedTime = sessionStorage.getItem('completedTime');
            
            if (justCompleted && completedTime) {
                const timeDiff = Date.now() - parseInt(completedTime);
                
                if (timeDiff < 5000) {
                    console.log('5초 내 중복 접근 감지, 상태:', justCompleted);
                    
                    if (justCompleted === 'approved' || justCompleted === 'rejected' || justCompleted === 'cancelled') {
                        console.log('완료 상태 감지, 새 거래를 위해 플래그 클리어');
                        sessionStorage.removeItem('transactionJustCompleted');
                        sessionStorage.removeItem('completedTime');
                        localStorage.removeItem('transactionSessionId');
                    } else {
                        window.location.replace('payment_select.php');
                        return;
                    }
                } else {
                    sessionStorage.removeItem('transactionJustCompleted');
                    sessionStorage.removeItem('completedTime');
                }
            }
            
            // 3. 세션 ID 생성
            sessionId = localStorage.getItem('transactionSessionId');
            if (!sessionId) {
                sessionId = 'SESSION_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
                localStorage.setItem('transactionSessionId', sessionId);
            }
            console.log('세션 ID:', sessionId);
            
            initializeBankTransfer();
        });

        // 은행 송금 초기화
        function initializeBankTransfer() {
            const justCompleted = sessionStorage.getItem('transactionJustCompleted');
            if (justCompleted) {
                console.log('초기화 중 완료된 거래 감지, 중단');
                return;
            }
            
            if (isInitializing) {
                console.log('이미 초기화가 진행 중입니다.');
                return;
            }
            
            isInitializing = true;
            console.log('initializeBankTransfer 시작');
            
            const storedData = localStorage.getItem('paymentData');
            
            if (!storedData) {
                isInitializing = false;
                showError('결제 정보를 찾을 수 없습니다.');
                return;
            }

            try {
                paymentData = JSON.parse(storedData);
                
                if (paymentData.type !== 'bank') {
                    isInitializing = false;
                    showError('은행 송금 정보가 아닙니다.');
                    return;
                }

                if (!paymentData.amount || !paymentData.depositorName || !paymentData.storeCode) {
                    isInitializing = false;
                    showError('필수 결제 정보가 누락되었습니다.');
                    return;
                }

                // storeData 복구 로직 강화
                if (paymentData.storeData && paymentData.storeData.bank && paymentData.storeData.account_number) {
                    storeData = paymentData.storeData;
                    console.log('paymentData에서 storeData 사용');
                    proceedToBankTransfer();
                } else {
                    // localStorage에서 storeData 복구 시도
                    const savedStoreData = localStorage.getItem('currentStoreData');
                    if (savedStoreData) {
                        try {
                            const parsedStoreData = JSON.parse(savedStoreData);
                            if (parsedStoreData.bank && parsedStoreData.account_number) {
                                storeData = parsedStoreData;
                                console.log('localStorage에서 storeData 복구됨:', storeData);
                                proceedToBankTransfer();
                            } else {
                                console.log('저장된 storeData가 불완전, 서버에서 다시 가져오기');
                                checkStoreInfo();
                            }
                        } catch (e) {
                            console.error('저장된 storeData 파싱 오류:', e);
                            checkStoreInfo();
                        }
                    } else {
                        console.log('저장된 storeData가 없음, 서버에서 가져오기');
                        checkStoreInfo();
                    }
                }
                
            } catch (e) {
                console.error('Payment data parsing error:', e);
                isInitializing = false;
                showError('결제 정보 형식이 올바르지 않습니다.');
            }
        }

        // ========== storeData 필드명 확인용 임시 코드 ==========
        function checkStoreInfo() {
            fetch('check_store_code.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ store_code: paymentData.storeCode })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    storeData = data.store;
                    // storeData를 localStorage에 저장 (새로고침 대비)
                    localStorage.setItem('currentStoreData', JSON.stringify(storeData));

                    proceedToBankTransfer();
                } else {
                    isInitializing = false;
                    showError('가맹점 정보를 찾을 수 없습니다: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Store check error:', error);
                isInitializing = false;
                showError('가맹점 정보 확인 중 오류가 발생했습니다.');
            });
        }

        function proceedToBankTransfer() {
            checkExistingBankTransaction();
        }

        // ========== 거래 관리 ==========
        function checkExistingBankTransaction() {
            console.log('기존 은행 거래 확인 중...');
            
            if (transactionFinalized) {
                isInitializing = false;
                return;
            }
            
            // storeData가 없으면 먼저 복구 시도
            if (!storeData) {
                console.log('storeData가 없음, 복구 시도');
                const savedStoreData = localStorage.getItem('currentStoreData');
                if (savedStoreData) {
                    try {
                        storeData = JSON.parse(savedStoreData);
                        console.log('localStorage에서 storeData 복구 성공:', storeData);
                    } catch (e) {
                        console.error('storeData 복구 실패:', e);
                    }
                }
            }
            
            if (currentTransaction) {
                isInitializing = false;
                const mockData = {
                    transaction_id: currentTransaction,
                    amount: paymentData.amount,
                    depositor_name: paymentData.depositorName,
                    bank_info: {
                        bank: storeData?.bank_info?.bank || '은행 정보 없음',
                        account_number: storeData?.bank_info?.account_number || '계좌번호 없음',
                        account_holder: storeData?.bank_info?.account_holder || '예금주 없음'
                    }
                };
                showBankAccountInfo(mockData);
                return;
            }
            
            // 나머지 fetch 로직은 동일...
            fetch('get_remaining_time.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    session_id: sessionId,
                    check_existing: true
                })
            })
            .then(response => response.json())
            .then(data => {
                console.log('기존 거래 확인 결과:', data);
                
                if (data.finalized_transaction) {
                    console.log('서버에서 완료된 거래 감지');
                    isInitializing = false;
                    
                    const status = data.finalized_transaction.status;
                    if (status === 'approved' || status === 'completed') {
                        alert('이미 승인된 거래입니다. 결제가 완료되었습니다.');
                    } else if (status === 'rejected') {
                        alert('이미 거절된 거래입니다. 다시 시도해주세요.');
                    } else {
                        alert('이미 처리된 거래입니다.');
                    }
                    
                    goToPaymentSelect();
                    return;
                }
                
                if (data.existing_transaction) {
                    const transaction = data.existing_transaction;
                    
                    if (transaction.status === 'approved' || transaction.status === 'completed') {
                        isInitializing = false;
                        alert('이미 승인된 거래입니다. 결제가 완료되었습니다.');
                        goToPaymentSelect();
                        return;
                    }
                    
                    if (transaction.status === 'rejected') {
                        isInitializing = false;
                        alert('거래가 거절되었습니다. 다시 시도해주세요.');
                        goToPaymentSelect();
                        return;
                    }
                    
                    if (data.is_expired || transaction.status === 'cancelled') {
                        createNewBankTransaction();
                        return;
                    }
                    
                    currentTransaction = transaction.kakao_tid;
                    isInitializing = false;
                    
                    // 기존 거래 표시 시에도 올바른 구조 사용
                    const mockData = {
                        transaction_id: transaction.kakao_tid,
                        amount: paymentData.amount,
                        depositor_name: paymentData.depositorName,
                        bank_info: {
                            bank: storeData?.bank_info?.bank || '은행 정보 없음',
                            account_number: storeData?.bank_info?.account_number || '계좌번호 없음',
                            account_holder: storeData?.bank_info?.account_holder || '예금주 없음'
                        }
                    };
                    
                    showBankAccountInfo(mockData);
                } else {
                    createNewBankTransaction();
                }
            })
            .catch(error => {
                console.error('기존 거래 확인 오류:', error);
                isInitializing = false;
                showError('거래 확인 중 오류가 발생했습니다. 다시 시도해주세요.');
            });
        }


        function createNewBankTransaction() {
            if (transactionFinalized) {
                console.log('이미 완료된 거래가 있어 새 거래를 생성하지 않습니다.');
                isInitializing = false;
                return;
            }
            
            if (isTransactionCreating) {
                console.log('이미 거래 생성이 진행 중입니다.');
                return;
            }
            
            if (currentTransaction) {
                console.log('이미 활성 거래가 있어 새 거래를 생성하지 않습니다.');
                isInitializing = false;
                return;
            }
            
            isTransactionCreating = true;
            console.log('새 은행 거래 생성 시작');
            
            const requestData = {
                store_id: paymentData.storeCode,
                amount: paymentData.amount,
                depositor_name: paymentData.depositorName,
                transfer_type: 'bank',
                session_id: sessionId
            };
            
            fetch('create_bank_transfer_request.php', {
                method: 'POST',
                headers: { 
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify(requestData)
            })
            .then(response => response.text())
            .then(text => {
                try {
                    const data = JSON.parse(text);
                    if (data.success) {
                        currentTransaction = data.data.transaction_id;
                        showBankAccountInfo(data.data);
                    } else {
                        if (data.message && (data.message.includes('이미 진행 중인') || data.message.includes('너무 빠른'))) {
                            isTransactionCreating = false;
                            isInitializing = false;
                            checkExistingBankTransaction();
                        } else {
                            showError('은행 송금 신청 실패: ' + data.message);
                        }
                    }
                } catch (e) {
                    console.error('JSON 파싱 에러:', e);
                    showError('서버 응답 형식 오류: ' + text.substring(0, 100));
                } finally {
                    isTransactionCreating = false;
                    isInitializing = false;
                }
            })
            .catch(error => {
                console.error('Fetch error:', error);
                isTransactionCreating = false;
                isInitializing = false;
                showError('연결 중 오류가 발생했습니다: ' + error.message);
            });
        }

        // ========== 화면 표시 ==========
        function showBankAccountInfo(data) {
            console.log('showBankAccountInfo 시작:', data);
            console.log('현재 storeData:', storeData);
            
            const depositorName = String(data.depositor_name || '').replace(/'/g, "\\'").replace(/"/g, '\\"');
            const amount = Number(data.amount) || 0;
            const transactionId = String(data.transaction_id || '').replace(/'/g, "\\'").replace(/"/g, '\\"');

            // 은행 정보 추출 로직 수정
            let bankName, accountNumber, accountHolder;
            
            // 1차: data.bank_info 확인
            if (data.bank_info && 
                data.bank_info.bank && 
                data.bank_info.bank !== '은행 정보 없음' &&
                data.bank_info.bank.trim() !== '' &&
                data.bank_info.account_number && 
                data.bank_info.account_number !== '계좌번호 없음' &&
                data.bank_info.account_number.trim() !== '') {
                
                console.log('data.bank_info에서 유효한 은행 정보 사용');
                bankName = String(data.bank_info.bank).replace(/'/g, "\\'").replace(/"/g, '\\"');
                accountNumber = String(data.bank_info.account_number).replace(/'/g, "\\'").replace(/"/g, '\\"');
                accountHolder = String(data.bank_info.account_holder || '').replace(/'/g, "\\'").replace(/"/g, '\\"') || '예금주 없음';
                
            } else if (storeData && 
                    storeData.bank_info && 
                    storeData.bank_info.bank && 
                    storeData.bank_info.bank.trim() !== '' && 
                    storeData.bank_info.account_number && 
                    storeData.bank_info.account_number.trim() !== '') {
                // 2차: 현재 storeData.bank_info 사용 (올바른 구조)
                console.log('현재 storeData.bank_info에서 유효한 은행 정보 사용');
                bankName = String(storeData.bank_info.bank).replace(/'/g, "\\'").replace(/"/g, '\\"');
                accountNumber = String(storeData.bank_info.account_number).replace(/'/g, "\\'").replace(/"/g, '\\"');
                accountHolder = String(storeData.bank_info.account_holder || '').replace(/'/g, "\\'").replace(/"/g, '\\"') || '예금주 없음';
                
            } else {
                // 3차: localStorage에서 storeData 복구 시도
                console.log('localStorage에서 storeData 복구 시도');
                const savedStoreData = localStorage.getItem('currentStoreData');
                if (savedStoreData) {
                    try {
                        const recoveredStoreData = JSON.parse(savedStoreData);
                        if (recoveredStoreData.bank_info && 
                            recoveredStoreData.bank_info.bank && 
                            recoveredStoreData.bank_info.bank.trim() !== '' &&
                            recoveredStoreData.bank_info.account_number && 
                            recoveredStoreData.bank_info.account_number.trim() !== '') {
                            
                            console.log('localStorage에서 유효한 storeData.bank_info 복구 성공');
                            bankName = String(recoveredStoreData.bank_info.bank).replace(/'/g, "\\'").replace(/"/g, '\\"');
                            accountNumber = String(recoveredStoreData.bank_info.account_number).replace(/'/g, "\\'").replace(/"/g, '\\"');
                            accountHolder = String(recoveredStoreData.bank_info.account_holder || '').replace(/'/g, "\\'").replace(/"/g, '\\"') || '예금주 없음';
                            
                            // storeData도 복구
                            storeData = recoveredStoreData;
                        } else {
                            console.log('복구된 storeData.bank_info가 불완전함 - 기본값 사용');
                            bankName = '은행 정보 없음';
                            accountNumber = '계좌번호 없음';
                            accountHolder = '예금주 없음';
                        }
                    } catch (e) {
                        console.error('storeData 복구 실패:', e);
                        bankName = '은행 정보 없음';
                        accountNumber = '계좌번호 없음';
                        accountHolder = '예금주 없음';
                    }
                } else {
                    console.log('localStorage에 저장된 storeData가 없음');
                    bankName = '은행 정보 없음';
                    accountNumber = '계좌번호 없음';
                    accountHolder = '예금주 없음';
                }
            }
            
            console.log('최종 은행 정보:', { bankName, accountNumber, accountHolder });
            
            // 은행 정보가 여전히 없고, 아직 서버에서 가져오기를 시도하지 않았다면 시도
            if ((bankName === '은행 정보 없음' || accountNumber === '계좌번호 없음') && 
                paymentData && 
                paymentData.storeCode && 
                !isRefetchingStoreData) {
                
                console.log('은행 정보가 없어서 서버에서 다시 가져오기 시도 (한 번만)');
                isRefetchingStoreData = true;
                
                fetch('check_store_code.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ store_code: paymentData.storeCode })
                })
                .then(response => response.json())
                .then(storeResponse => {
                    if (storeResponse.success && storeResponse.store) {
                        const newStoreData = storeResponse.store;
                        console.log('서버에서 storeData 새로 가져옴:', newStoreData);
                        
                        // 올바른 구조로 확인 (bank_info 객체 안에 있음)
                        if (newStoreData.bank_info && 
                            newStoreData.bank_info.bank && 
                            newStoreData.bank_info.bank.trim() !== '' &&
                            newStoreData.bank_info.account_number && 
                            newStoreData.bank_info.account_number.trim() !== '') {
                            
                            storeData = newStoreData;
                            localStorage.setItem('currentStoreData', JSON.stringify(storeData));
                            
                            console.log('서버에서 가져온 유효한 bank_info 데이터로 업데이트');
                            const updatedBankName = String(newStoreData.bank_info.bank).replace(/'/g, "\\'").replace(/"/g, '\\"');
                            const updatedAccountNumber = String(newStoreData.bank_info.account_number).replace(/'/g, "\\'").replace(/"/g, '\\"');
                            const updatedAccountHolder = String(newStoreData.bank_info.account_holder || '').replace(/'/g, "\\'").replace(/"/g, '\\"') || '예금주 없음';
                            
                            renderBankAccountHTML(data, updatedBankName, updatedAccountNumber, updatedAccountHolder);
                        } else {
                            console.log('서버에서 가져온 storeData.bank_info도 불완전함 - 기본값 유지');
                            renderBankAccountHTML(data, bankName, accountNumber, accountHolder);
                        }
                    } else {
                        console.log('서버에서 storeData 가져오기 실패 - 기본값 유지');
                        renderBankAccountHTML(data, bankName, accountNumber, accountHolder);
                    }
                })
                .catch(error => {
                    console.error('스토어 정보 재조회 실패:', error);
                    renderBankAccountHTML(data, bankName, accountNumber, accountHolder);
                })
                .finally(() => {
                    isRefetchingStoreData = false;
                });
                
                return;
            }
            
            // 일반적인 경우 HTML 렌더링
            renderBankAccountHTML(data, bankName, accountNumber, accountHolder);
        }

        // ========== HTML 렌더링 분리 함수 ==========
        function renderBankAccountHTML(data, bankName, accountNumber, accountHolder) {
            const depositorName = String(data.depositor_name || '').replace(/'/g, "\\'").replace(/"/g, '\\"');
            const amount = Number(data.amount) || 0;
            const transactionId = String(data.transaction_id || '').replace(/'/g, "\\'").replace(/"/g, '\\"');
            
            window.currentTransactionId = data.transaction_id;
            window.currentAccountNumber = accountNumber !== '계좌번호 없음' ? accountNumber : '';
            
            document.body.innerHTML = `
                <div class="bg-gray-900 text-white min-h-screen py-8">
                    <div class="container mx-auto px-4 max-w-md">
                        <div class="text-center mb-8">
                            <div class="w-20 h-20 bg-green-400 rounded-full flex items-center justify-center mx-auto mb-4">
                                <span class="text-3xl">✅</span>
                            </div>
                            <h1 class="text-2xl font-bold text-green-400 mb-2">송금 신청 완료</h1>
                            <p class="text-gray-300">아래 계좌로 송금해주세요</p>
                            
                            <div class="bg-red-600/20 border border-red-500/50 rounded-lg p-3 mt-4 mb-4">
                                <div class="flex items-center justify-center space-x-2">
                                    <span class="text-red-400 text-lg">⏰</span>
                                    <span class="text-red-300 text-sm">송금 제한시간:</span>
                                    <span id="countdown" class="text-red-200 font-mono text-lg font-bold">10:00</span>
                                </div>
                                <p class="text-red-300 text-xs mt-1">시간 초과 시 자동으로 신청이 취소됩니다</p>
                            </div>
                        </div>

                        <div class="bg-gray-800 rounded-xl p-6 mb-6">
                            <h3 class="text-lg font-semibold mb-4 text-blue-400">송금 정보</h3>
                            <div class="space-y-3">
                                <div class="flex justify-between">
                                    <span class="text-gray-300">입금자</span>
                                    <span class="text-white font-semibold">${depositorName}</span>
                                </div>
                                <div class="flex justify-between">
                                    <span class="text-gray-300">금액</span>
                                    <span class="text-white font-semibold text-xl">${amount.toLocaleString()}원</span>
                                </div>
                                <div class="flex justify-between">
                                    <span class="text-gray-300">신청번호</span>
                                    <span class="text-white font-mono text-sm">${transactionId}</span>
                                </div>
                            </div>
                        </div>

                        <div class="bg-gradient-to-r from-blue-600 to-blue-700 rounded-xl p-6 mb-6">
                            <h3 class="text-lg font-semibold mb-4 text-white">송금할 계좌</h3>
                            <div class="space-y-3">
                                <div class="flex justify-between items-center">
                                    <span class="text-blue-100">은행</span>
                                    <span class="text-white font-bold text-lg">${bankName}</span>
                                </div>
                                <div class="flex justify-between items-center">
                                    <span class="text-blue-100">계좌번호</span>
                                    <div class="text-right">
                                        <span class="text-white font-mono text-lg block">${accountNumber}</span>
                                        ${accountNumber !== '계좌번호 없음' ? 
                                            '<button onclick="copyAccountNumber()" class="text-blue-200 text-sm hover:text-white">복사</button>' : ''}
                                    </div>
                                </div>
                                <div class="flex justify-between items-center">
                                    <span class="text-blue-100">예금주</span>
                                    <span class="text-white font-semibold">${accountHolder}</span>
                                </div>
                            </div>
                        </div>

                        <div class="space-y-3">
                            <button id="confirmButton" onclick="confirmCurrentTransfer()" 
                                    class="w-full bg-green-500 text-white font-bold py-4 px-6 rounded-xl hover:bg-green-600 transition text-lg">
                                송금 완료
                            </button>
                            <button onclick="goToPaymentSelect()" 
                                    class="w-full bg-blue-400 text-gray-900 font-bold py-3 px-6 rounded-xl hover:bg-blue-500 transition">
                                결제 페이지로 돌아가기
                            </button>
                            <button onclick="window.close()" 
                                    class="w-full bg-gray-600 text-white font-semibold py-3 px-6 rounded-xl hover:bg-gray-700 transition">
                                창 닫기
                            </button>
                        </div>
                    </div>
                </div>
            `;
            
            getRemainingTimeFromDB(data.transaction_id);
            startStatusCheck(data.transaction_id);
        }


        // ========== 유틸리티 함수 ==========
        function showError(message) {
            document.getElementById('loadingScreen').classList.add('hidden');
            document.getElementById('errorScreen').classList.remove('hidden');
            document.getElementById('errorMessage').textContent = message;
            
            isInitializing = false;
            isTransactionCreating = false;
            isStatusUpdating = false;
        }

        function copyText(text) {
            navigator.clipboard.writeText(text).then(function() {
                alert('복사되었습니다: ' + text);
            }).catch(function() {
                prompt('복사하세요:', text);
            });
        }

        // ========== 타이머 관리 ==========
        function getRemainingTimeFromDB(transactionId) {
            fetch('get_remaining_time.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ transaction_id: transactionId })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (data.is_expired) {
                        handleTimeExpired(transactionId);
                    } else {
                        startCountdown(data.remaining_seconds, transactionId);
                    }
                } else {
                    console.error('남은 시간 조회 실패:', data.message);
                    startCountdown(600, transactionId);
                }
            })
            .catch(error => {
                console.error('남은 시간 조회 오류:', error);
                startCountdown(600, transactionId);
            });
        }

        function startCountdown(seconds, transactionId) {
            if (countdownTimer) {
                clearInterval(countdownTimer);
            }
            
            let timeLeft = seconds;
            
            countdownTimer = setInterval(() => {
                const countdownElement = document.getElementById('countdown');
                
                if (!countdownElement) {
                    console.error('countdown 엘리먼트를 찾을 수 없습니다.');
                    return;
                }
                
                const minutes = Math.floor(timeLeft / 60);
                const secs = timeLeft % 60;
                
                const display = minutes.toString().padStart(2, '0') + ':' + secs.toString().padStart(2, '0');
                countdownElement.textContent = display;
                
                if (timeLeft <= 60) {
                    countdownElement.className = 'text-red-400 font-mono text-lg font-bold animate-pulse';
                } else if (timeLeft <= 180) {
                    countdownElement.className = 'text-orange-400 font-mono text-lg font-bold';
                } else {
                    countdownElement.className = 'text-red-200 font-mono text-lg font-bold';
                }
                
                timeLeft--;
                
                if (timeLeft < 0) {
                    clearInterval(countdownTimer);
                    handleTimeExpired(transactionId);
                }
            }, 1000);
        }

        function handleTimeExpired(transactionId) {
            isTimerExpired = true;
            
            if (statusCheckInterval) {
                clearInterval(statusCheckInterval);
            }
            
            const confirmButton = document.getElementById('confirmButton');
            if (confirmButton) {
                confirmButton.disabled = true;
                confirmButton.className = 'w-full bg-gray-500 text-gray-300 font-bold py-4 px-6 rounded-xl cursor-not-allowed text-lg';
                confirmButton.textContent = '시간 만료';
            }
            
            const countdownElement = document.getElementById('countdown');
            if (countdownElement) {
                countdownElement.textContent = '만료';
                countdownElement.className = 'text-red-500 font-mono text-lg font-bold';
            }
            
            currentTransaction = null;
            alert('10분 제한시간이 초과되어 송금 신청이 만료되었습니다.');
            
            setTimeout(() => {
                goToPaymentSelect();
            }, 3000);
        }

        // ========== 거래 상태 관리 ==========
        function confirmTransfer(transactionId) {
            if (isTimerExpired) {
                alert('송금 시간이 만료되었습니다. 다시 신청해주세요.');
                goToPaymentSelect();
                return;
            }
            
            if (confirm('송금을 완료하셨습니까?\n완료하지 않으셨다면 취소를 눌러주세요.')) {
                updateTransferStatus(transactionId);
            }
        }

        function updateTransferStatus(transactionId) {
            if (isStatusUpdating) {
                return;
            }
            
            isStatusUpdating = true;
            
            fetch('update_transfer_status.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    transaction_id: transactionId,
                    status: 'requested'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (countdownTimer) clearInterval(countdownTimer);
                    if (statusCheckInterval) clearInterval(statusCheckInterval);
                    
                    alert('송금 완료 신청이 접수되었습니다.\n입금 확인 후 처리됩니다.');
                    goToPaymentSelect();
                } else {
                    if (data.code === 'DUPLICATE' || data.code === 'SAME_STATUS') {
                        alert('이미 처리된 요청입니다.');
                        goToPaymentSelect();
                    } else {
                        alert('상태 업데이트에 실패했습니다: ' + data.message);
                    }
                }
            })
            .catch(error => {
                console.error('상태 업데이트 오류:', error);
                alert('상태 업데이트 중 오류가 발생했습니다.');
            })
            .finally(() => {
                isStatusUpdating = false;
            });
        }

        function checkTransactionStatus(transactionId) {
            if (transactionFinalized) {
                return;
            }
            
            fetch('check_transaction_status.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ transaction_id: transactionId })
            })
            .then(response => response.text())
            .then(text => {
                try {
                    const data = JSON.parse(text);
                    console.log('상태 확인 응답:', data);
                    
                    if (data.success) {
                        const status = data.status;
                        
                        if (status === 'approved' || status === 'completed') {
                            console.log('승인 처리 시작');
                            handleTransactionComplete('approved', '거래가 승인되었습니다!');
                            
                        } else if (status === 'rejected') {
                            console.log('거절 처리 시작');
                            handleTransactionComplete('rejected', '거래가 거절되었습니다.\n다시 시도해주세요.');
                            
                        } else if (status === 'cancelled') {
                            console.log('취소 처리 시작');
                            handleTransactionComplete('cancelled', '거래가 취소되었습니다.');
                        }
                    }
                } catch (parseError) {
                    console.error('JSON 파싱 오류:', parseError);
                }
            })
            .catch(error => {
                console.error('상태 확인 오류:', error);
            });
        }

        function startStatusCheck(transactionId) {
            if (statusCheckInterval) {
                clearInterval(statusCheckInterval);
            }
            
            statusCheckInterval = setInterval(() => {
                checkTransactionStatus(transactionId);
            }, 3000);
        }

        // ========== 이벤트 리스너 ==========
        // ========== 새로고침 방지 이벤트 리스너 강화 ==========
        window.addEventListener('beforeunload', function(e) {
            if (transactionFinalized) {
                return;
            }

            // 정상적으로 페이지를 떠날 때만 정리 (새로고침이 아닌 경우)
            if (!modalBlockingRefresh && !alertModalActive) {
                // localStorage.removeItem('currentStoreData'); // 필요시 주석 해제
            }
            
            // 모달이 활성화된 상태에서 새로고침 시도 시 강력한 경고
            if (modalBlockingRefresh || alertModalActive) {
                const message = '처리 완료 알림이 표시 중입니다. 새로고침하지 마세요!';
                e.returnValue = message;
                return message;
            }
            
            if (currentTransaction && !isTimerExpired) {
                const message = '송금이 진행 중입니다. 페이지를 떠나시겠습니까?';
                e.returnValue = message;
                return message;
            }
        });

        // ========== 추가: 페이지 포커스 감지로 모달 상태 복구 ==========
        window.addEventListener('focus', function() {
            // 페이지로 돌아왔을 때 모달 상태 확인
            const modalActiveTransaction = localStorage.getItem('modalActiveTransaction');
            const modalActiveTime = localStorage.getItem('modalActiveTime');
            
            if (modalActiveTransaction && modalActiveTime) {
                const timeDiff = Date.now() - parseInt(modalActiveTime);
                
                if (timeDiff < 10000) {
                    console.log('페이지 포커스 시 미완료 모달 감지, 리다이렉트');
                    localStorage.removeItem('modalActiveTransaction');
                    localStorage.removeItem('modalActiveTime');
                    localStorage.removeItem('transactionSessionId');
                    window.location.replace('payment_select.php');
                }
            }
        });

        // ========== 추가: 키보드 이벤트로 새로고침 차단 ==========
        document.addEventListener('keydown', function(e) {
            if (modalBlockingRefresh || alertModalActive) {
                // F5, Ctrl+R, Cmd+R 차단
                if (e.key === 'F5' || 
                    (e.ctrlKey && e.key === 'r') || 
                    (e.metaKey && e.key === 'r')) {
                    e.preventDefault();
                    console.log('모달 활성화 중 새로고침 차단됨');
                    
                    // 사용자에게 알림 (기존 모달 위에 추가 알림)
                    const warningDiv = document.createElement('div');
                    warningDiv.className = 'fixed top-4 left-1/2 transform -translate-x-1/2 bg-red-600 text-white px-4 py-2 rounded-lg z-[10000]';
                    warningDiv.textContent = '처리 중입니다. 새로고침하지 마세요!';
                    document.body.appendChild(warningDiv);
                    
                    setTimeout(() => {
                        warningDiv.remove();
                    }, 3000);
                    
                    return false;
                }
            }
        });

        document.addEventListener('visibilitychange', function() {
            if (document.hidden && transactionFinalized) {
                console.log('완료된 거래 상태에서 탭 숨김 감지');
                window.location.replace('payment_select.php');
            }
        });

        window.addEventListener('beforeunload', function() {
            if (countdownTimer) {
                clearInterval(countdownTimer);
            }
            if (statusCheckInterval) {
                clearInterval(statusCheckInterval);
            }
        });

        // ========== 거래 완료 처리 함수 ==========
        // ========== 수정된 거래 완료 처리 함수 ==========
        function handleTransactionComplete(status, message) {
            transactionFinalized = true;
            
            if (statusCheckInterval) clearInterval(statusCheckInterval);
            if (countdownTimer) clearInterval(countdownTimer);
            currentTransaction = null;
            
            // 완료 상태를 즉시 여러 곳에 저장 (중복 방지 강화)
            sessionStorage.setItem('transactionJustCompleted', status);
            sessionStorage.setItem('completedTime', Date.now().toString());
            localStorage.setItem('lastCompletedTransaction', status);
            localStorage.setItem('lastCompletedTime', Date.now().toString());

            // 모바일 전용 추가
            if (isMobile) {
                localStorage.setItem('mobileTransactionCompleted', status);
                localStorage.setItem('mobileCompletedTime', Date.now().toString());
                mobilePreventDuplicate = true;
            }
            
            // messageType 한 번만 선언
            let alertType = status === 'completed' ? 'success' : 'warning';
            showCustomModal(message, alertType);
            redirectAfterAlert = true;
            
            setTimeout(() => {
                if (alertModalActive) closeCustomModal();
                else if (redirectAfterAlert) {
                    redirectAfterAlert = false;
                    window.location.replace('payment_select.php');
                }
            }, 6000);

            console.log(`${status} 완료 플래그 설정됨`);
            
            // 세션 ID 제거 (새 거래를 위해)
            localStorage.removeItem('transactionSessionId');
            
            // 커스텀 모달로 메시지 표시
            const messageType = status === 'approved' ? 'success' : 
                            status === 'rejected' ? 'error' : 'warning';
            
            showCustomModal(message, messageType);
            
            // 모달이 닫힌 후 리다이렉트하도록 설정
            redirectAfterAlert = true;
            
            // 백업용 타이머 (모달이 제대로 안 닫힐 경우를 대비)
            setTimeout(() => {
                if (alertModalActive) {
                    closeCustomModal();
                } else if (redirectAfterAlert) {
                    redirectAfterAlert = false;
                    window.location.replace('payment_select.php');
                }
            }, 6000);
        }

        // ========== 추가: 페이지 언로드 시 정리 작업 ==========
        window.addEventListener('unload', function() {
            // 페이지를 떠날 때 모달 관련 정보 정리
            if (modalBlockingRefresh) {
                localStorage.removeItem('modalActiveTransaction');
                localStorage.removeItem('modalActiveTime');
            }
        });



        // ========== 커스텀 모달 함수들 ==========
        // 수정된 커스텀 모달 함수
        function showCustomModal(message, type = 'info') {
            if (alertModalActive) return;
            
            alertModalActive = true;
            modalBlockingRefresh = true; // 모달 활성화 시 새로고침 차단 활성화
            
             // 모바일 전용 강화
            if (isMobile) {
                localStorage.setItem('mobileModalBlocking', 'true');
                localStorage.setItem('mobileModalTime', Date.now().toString());
                // 여러 곳에 저장해서 확실히 감지
                sessionStorage.setItem('mobileModalActive', 'true');
            }

            // 즉시 완료 상태를 localStorage에 저장 (새로고침 대비)
            localStorage.setItem('modalActiveTransaction', 'true');
            localStorage.setItem('modalActiveTime', Date.now().toString());
            
            const modal = document.createElement('div');
            modal.id = 'customAlertModal';
            modal.className = 'fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-[9999]';
            
            const iconMap = {
                'success': '✅',
                'error': '❌',
                'warning': '⚠️',
                'info': 'ℹ️'
            };
            
            const colorMap = {
                'success': 'text-green-400',
                'error': 'text-red-400', 
                'warning': 'text-yellow-400',
                'info': 'text-blue-400'
            };
            
            modal.innerHTML = `
                <div class="bg-gray-800 rounded-xl p-6 max-w-sm mx-4 text-center">
                    <div class="w-16 h-16 bg-gray-700 rounded-full flex items-center justify-center mx-auto mb-4">
                        <span class="text-2xl">${iconMap[type] || iconMap.info}</span>
                    </div>
                    <h3 class="text-lg font-bold ${colorMap[type] || colorMap.info} mb-3">알림</h3>
                    <p class="text-white mb-6">${message}</p>
                    <button onclick="closeCustomModal()" 
                            class="w-full bg-blue-500 text-white font-bold py-3 px-6 rounded-lg hover:bg-blue-600 transition">
                        확인
                    </button>
                </div>
            `;
            
            document.body.appendChild(modal);
            
            // 5초 후 자동 닫기
            setTimeout(() => {
                if (document.getElementById('customAlertModal')) {
                    closeCustomModal();
                }
            }, 5000);
        }

        function closeCustomModal() {
            const modal = document.getElementById('customAlertModal');
            if (modal) {
                modal.remove();
                alertModalActive = false;
                modalBlockingRefresh = false; // 모달 닫힐 때 차단 해제

                // 모바일 전용 정리
                if (isMobile) {
                    localStorage.removeItem('mobileModalBlocking');
                    localStorage.removeItem('mobileModalTime');
                    sessionStorage.removeItem('mobileModalActive');
                }
                
                // 모달 관련 localStorage 정리
                localStorage.removeItem('modalActiveTransaction');
                localStorage.removeItem('modalActiveTime');
                
                if (redirectAfterAlert) {
                    redirectAfterAlert = false;
                    window.location.replace('payment_select.php');
                }
            }
        }

        // 모바일 전용 이벤트들
        if (isMobile) {
            // 페이지 숨김/표시 감지 (모바일 앱 전환)
            document.addEventListener('visibilitychange', function() {
                if (document.hidden) {
                    if (transactionFinalized || alertModalActive) {
                        localStorage.setItem('mobileForceRedirect', 'true');
                    }
                } else {
                    // 페이지가 다시 보일 때
                    if (localStorage.getItem('mobileForceRedirect')) {
                        localStorage.removeItem('mobileForceRedirect');
                        window.location.replace('payment_select.php');
                    }
                }
            });
            
            // 강화된 터치 새로고침 차단 (여기서부터 새로운 코드)
            let preventRefresh = false;
            let touchStartY = 0;
            
            document.addEventListener('touchstart', function(e) {
                if (alertModalActive || modalBlockingRefresh || localStorage.getItem('mobileModalBlocking')) {
                    preventRefresh = true;
                }
                touchStartY = e.touches[0].clientY;
            }, {passive: false});
            
            document.addEventListener('touchmove', function(e) {
                if (preventRefresh && window.scrollY === 0) {
                    const touchY = e.touches[0].clientY;
                    const pullDistance = touchY - touchStartY;
                    
                    if (pullDistance > 30) {
                        e.preventDefault();
                        e.stopPropagation();
                        
                        const warningDiv = document.createElement('div');
                        warningDiv.className = 'fixed top-4 left-1/2 transform -translate-x-1/2 bg-red-600 text-white px-4 py-2 rounded-lg z-[10000]';
                        warningDiv.textContent = '처리 중입니다. 새로고침하지 마세요!';
                        document.body.appendChild(warningDiv);
                        setTimeout(() => warningDiv.remove(), 2000);
                        
                        return false;
                    }
                }
            }, {passive: false});
            
            document.addEventListener('touchend', function() {
                setTimeout(() => {
                    preventRefresh = false;
                }, 1000);
            });
        }
    </script>
</body>
</html>