<?php
// check_store_code.php - 디버깅 버전
header('Content-Type: application/json; charset=utf-8');

try {
    include 'inc/db_connect.php';
    
    error_log("=== check_store_code.php 시작 ===");
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('POST 요청만 허용됩니다.');
    }
    
    $json_input = file_get_contents('php://input');
    error_log("Raw input: " . $json_input);
    
    $input = json_decode($json_input, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('JSON 파싱 에러: ' . json_last_error_msg());
    }
    
    $storeCode = $input['store_code'] ?? '';
    error_log("Store code: '{$storeCode}'");
    
    if (!$storeCode) {
        throw new Exception('가맹점 코드가 누락되었습니다.');
    }
    
    // 1단계: 기본 member 테이블만 확인
    $basicQuery = "SELECT * FROM member WHERE code = ? AND grade != 'admin'";
    $stmt = mysqli_prepare($con, $basicQuery);
    mysqli_stmt_bind_param($stmt, 's', $storeCode);
    mysqli_stmt_execute($stmt);
    $basicResult = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt));
    
    error_log("Basic member query result: " . print_r($basicResult, true));
    
    if (!$basicResult) {
        // 모든 코드 출력해서 비교
        $allQuery = "SELECT code, user_name FROM member WHERE grade != 'admin'";
        $allResult = mysqli_query($con, $allQuery);
        $allCodes = [];
        while ($row = mysqli_fetch_assoc($allResult)) {
            $allCodes[] = $row['code'];
        }
        
        echo json_encode([
            'success' => false,
            'message' => "코드 '{$storeCode}'에 해당하는 가맹점을 찾을 수 없습니다.",
            'debug' => [
                'searched_code' => $storeCode,
                'searched_length' => strlen($storeCode),
                'available_codes' => $allCodes,
                'input_raw' => $json_input
            ]
        ]);
        exit;
    }
    
    // 2단계: kakao_link_type 컬럼 존재 확인
    $columnQuery = "SHOW COLUMNS FROM member LIKE 'kakao_link_type'";
    $columnResult = mysqli_query($con, $columnQuery);
    $hasKakaoLinkType = mysqli_num_rows($columnResult) > 0;
    
    error_log("Has kakao_link_type column: " . ($hasKakaoLinkType ? 'YES' : 'NO'));
    
    // 3단계: 조건부 쿼리 실행
    if ($hasKakaoLinkType) {
        // kakao_link_type 컬럼이 있는 경우
        $query = "
            SELECT m.*, 
                   ka.kakao_tid, ka.name as kakao_name, ka.qr_link, ka.friend_link,
                   ba.bank as bank_name, ba.account_holder as bank_holder, ba.account_number as bank_number
            FROM member m 
            LEFT JOIN kakao_account ka ON m.kakao_account_id = ka.id 
            LEFT JOIN bank_account ba ON m.bank_account_id = ba.id
            WHERE m.code = ? AND m.grade != 'admin'
        ";
    } else {
        // kakao_link_type 컬럼이 없는 경우 (기존 방식)
        $query = "
            SELECT m.*, 
                   ka.kakao_tid, ka.name as kakao_name, ka.qr_link,
                   ba.bank as bank_name, ba.account_holder as bank_holder, ba.account_number as bank_number
            FROM member m 
            LEFT JOIN kakao_account ka ON m.kakao_account_id = ka.id 
            LEFT JOIN bank_account ba ON m.bank_account_id = ba.id
            WHERE m.code = ? AND m.grade != 'admin'
        ";
    }
    
    $stmt = mysqli_prepare($con, $query);
    mysqli_stmt_bind_param($stmt, 's', $storeCode);
    mysqli_stmt_execute($stmt);
    $store = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt));
    
    error_log("Final query result: " . print_r($store, true));
    
    if (!$store) {
        echo json_encode([
            'success' => false,
            'message' => 'JOIN 쿼리에서 결과를 찾을 수 없습니다.',
            'debug' => [
                'has_kakao_link_type' => $hasKakaoLinkType,
                'basic_member_found' => !empty($basicResult)
            ]
        ]);
        exit;
    }
    
    // 링크 선택 로직
    $kakao_link = null;
    if ($hasKakaoLinkType) {
        // kakao_link_type이 있는 경우
        if ($store['kakao_link_type'] === 'qr' && !empty($store['qr_link'])) {
            $kakao_link = $store['qr_link'];
        } elseif ($store['kakao_link_type'] === 'friend' && !empty($store['friend_link'])) {
            $kakao_link = $store['friend_link'];
        }
    } else {
        // kakao_link_type이 없는 경우 기존 방식
        $kakao_link = $store['qr_link'] ?? null;
    }
    
    // 송금 방법 확인
    $has_kakao = false;
    $has_bank = false;

    if ($store['kakao_account_id'] && $store['kakao_account_id'] > 0 && !empty($kakao_link)) {
        $has_kakao = true;
    }

    if ($store['bank_account_id'] && $store['bank_account_id'] > 0 && !empty($store['bank_name']) && !empty($store['bank_number'])) {
        $has_bank = true;
    }

    if (!$has_kakao && !$has_bank) {
        echo json_encode([
            'success' => false,
            'message' => '해당 가맹점의 송금 방법이 설정되지 않았습니다.'
        ]);
        exit;
    }

    // 성공 응답
    echo json_encode([
        'success' => true,
        'message' => '가맹점 확인 완료',
        'store' => [
            'id' => $store['id'],
            'user_name' => $store['user_name'],
            'code' => $store['code'],
            'kakao_name' => $store['kakao_name'],
            'kakao_tid' => $store['kakao_tid'],
            'kakao_account_id' => (int)$store['kakao_account_id'],
            'bank_account_id' => (int)$store['bank_account_id'],
            'kakao_link_type' => $store['kakao_link_type'] ?? null,
            'qr_link' => $kakao_link,
            'has_kakao' => $has_kakao,
            'has_bank' => $has_bank,
            'bank_info' => [
                'bank' => $store['bank_name'],
                'account_holder' => $store['bank_holder'], 
                'account_number' => $store['bank_number']
            ]
        ],
        'debug' => [
            'has_kakao_link_type_column' => $hasKakaoLinkType,
            'selected_link' => $kakao_link,
            'link_type' => $store['kakao_link_type'] ?? 'not_set'
        ]
    ]);
    
} catch (Exception $e) {
    error_log("Error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'debug' => [
            'error_line' => $e->getLine(),
            'error_file' => $e->getFile()
        ]
    ]);
}
?>