<?php
// check_transaction_status.php
header('Content-Type: application/json; charset=utf-8');

try {
    include 'inc/db_connect.php';
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('POST 요청만 허용됩니다.');
    }
    
    $json_input = file_get_contents('php://input');
    if (empty($json_input)) {
        throw new Exception('요청 데이터가 비어있습니다.');
    }
    
    $input = json_decode($json_input, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('JSON 파싱 에러: ' . json_last_error_msg());
    }
    
    $transactionId = $input['transaction_id'] ?? '';
    
    if (!$transactionId) {
        throw new Exception('거래 ID가 필요합니다.');
    }
    
    // 거래 정보 조회
    $query = "
        SELECT id, user_id, status, payment_status, created_at, updated_at,
               TIMESTAMPDIFF(SECOND, created_at, NOW()) as elapsed_seconds,
               amount, depositor_name
        FROM transfer 
        WHERE kakao_tid = ?
        AND type = 'deposit'
    ";
    
    $stmt = mysqli_prepare($con, $query);
    if (!$stmt) {
        throw new Exception('쿼리 준비 실패: ' . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($stmt, 's', $transactionId);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $transaction = mysqli_fetch_assoc($result);
    
    if (!$transaction) {
        echo json_encode([
            'success' => false,
            'message' => '거래를 찾을 수 없습니다.'
        ]);
        exit;
    }
    
    $elapsedSeconds = $transaction['elapsed_seconds'];
    $remainingSeconds = 600 - $elapsedSeconds; // 10분 = 600초
    
    // 시간 만료된 거래 자동 취소
    if ($remainingSeconds <= 0 && in_array($transaction['status'], ['pending', 'requested'])) {
        $cancelQuery = "
            UPDATE transfer 
            SET status = 'cancelled', 
                cancelled_at = NOW(), 
                updated_at = NOW() 
            WHERE kakao_tid = ? 
            AND status IN ('pending', 'requested')
        ";
        $cancelStmt = mysqli_prepare($con, $cancelQuery);
        if ($cancelStmt) {
            mysqli_stmt_bind_param($cancelStmt, 's', $transactionId);
            mysqli_stmt_execute($cancelStmt);
            
            if (mysqli_affected_rows($con) > 0) {
                error_log("자동 취소 처리: transaction_id={$transactionId}");
                $transaction['status'] = 'cancelled';
            }
        }
    }
    
    // 응답 데이터
    echo json_encode([
        'success' => true,
        'status' => $transaction['status'],
        'payment_status' => $transaction['payment_status'],
        'remaining_seconds' => max(0, $remainingSeconds),
        'is_expired' => $remainingSeconds <= 0,
        'updated_at' => $transaction['updated_at'],
        'elapsed_seconds' => $elapsedSeconds,
        'transaction' => [
            'id' => $transaction['id'],
            'user_id' => $transaction['user_id'],
            'status' => $transaction['status'],
            'payment_status' => $transaction['payment_status'],
            'amount' => $transaction['amount'],
            'depositor_name' => $transaction['depositor_name'],
            'created_at' => $transaction['created_at'],
            'updated_at' => $transaction['updated_at']
        ]
    ]);
    
} catch (Exception $e) {
    error_log("Error in check_transaction_status.php: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

// DB 연결 정리
if (isset($con)) {
    mysqli_close($con);
}
?>