<?php
// confirm_bank_transfer.php - 은행 송금 완료 확인 처리
header('Content-Type: application/json; charset=utf-8');

try {
    include 'inc/db_connect.php';
    
    // POST 요청 확인
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('POST 요청만 허용됩니다.');
    }
    
    // JSON 데이터 읽기
    $json_input = file_get_contents('php://input');
    error_log("Bank transfer confirm input: " . $json_input);
    
    if (empty($json_input)) {
        throw new Exception('요청 데이터가 비어있습니다.');
    }
    
    $input = json_decode($json_input, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('JSON 파싱 에러: ' . json_last_error_msg());
    }
    
    $transactionId = $input['transaction_id'] ?? '';
    
    if (!$transactionId) {
        throw new Exception('거래 ID가 필요합니다.');
    }
    
    // 거래 정보 조회
    $query = "SELECT * FROM transfer WHERE kakao_tid = ? AND status = 'pending' AND payment_status = 'bank_requested'";
    $stmt = mysqli_prepare($con, $query);
    mysqli_stmt_bind_param($stmt, 's', $transactionId);
    mysqli_stmt_execute($stmt);
    $transfer = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt));
    
    if (!$transfer) {
        throw new Exception('해당 거래를 찾을 수 없거나 이미 처리되었습니다.');
    }
    
    // 송금 완료 상태로 업데이트
    $updateQuery = "UPDATE transfer SET 
                    payment_status = 'bank_completed', 
                    status = 'requested',
                    updated_at = NOW() 
                    WHERE id = ?";
    $updateStmt = mysqli_prepare($con, $updateQuery);
    mysqli_stmt_bind_param($updateStmt, 'i', $transfer['id']);
    
    if (!mysqli_stmt_execute($updateStmt)) {
        throw new Exception('송금 완료 처리에 실패했습니다.');
    }
    
    // 상태 변경 로그 추가 (transfer_status_logs 테이블이 있다면)
    $logQuery = "INSERT INTO transfer_status_logs (transfer_id, old_status, new_status, created_at) 
                 VALUES (?, 'bank_requested', 'bank_completed', NOW())";
    $logStmt = mysqli_prepare($con, $logQuery);
    mysqli_stmt_bind_param($logStmt, 'i', $transfer['id']);
    mysqli_stmt_execute($logStmt);
    
    // 성공 로그
    error_log("은행 송금 완료 처리: Transfer ID={$transfer['id']}, Transaction ID={$transactionId}, Amount={$transfer['amount']}원, Depositor={$transfer['depositor_name']}");
    
    echo json_encode([
        'success' => true,
        'message' => '송금 완료가 접수되었습니다. 입금 확인 후 처리됩니다.',
        'data' => [
            'transfer_id' => $transfer['id'],
            'transaction_id' => $transactionId,
            'amount' => $transfer['amount'],
            'depositor_name' => $transfer['depositor_name'],
            'status' => 'bank_completed'
        ]
    ]);
    
} catch (Exception $e) {
    error_log("은행 송금 완료 처리 오류: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>