<?php
// create_bank_transfer_request.php - 유저 페이지 점검 체크 추가 버전
header('Content-Type: application/json; charset=utf-8');

try {
    include 'inc/db_connect.php';
    
    // ========== 유저 페이지 점검 상태 확인 ==========
    $maintenanceQuery = "SELECT user_maintenance_enabled FROM settings WHERE id = 1";
    $maintenanceResult = mysqli_query($con, $maintenanceQuery);
    
    if ($maintenanceResult && $maintenanceRow = mysqli_fetch_assoc($maintenanceResult)) {
        if ($maintenanceRow['user_maintenance_enabled'] == 1) {
            // 유저 페이지가 점검 중인 경우
            echo json_encode([
                'success' => false,
                'maintenance' => true,
                'message' => '시스템 점검 중입니다. 잠시 후 다시 시도해주세요.',
                'maintenance_info' => [
                    'status' => 'under_maintenance',
                    'message' => '서비스 업그레이드를 위한 정기 점검 중입니다.',
                    'type' => 'bank_transfer'
                ]
            ]);
            exit;
        }
    }
    
    // POST 요청 확인
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('POST 요청만 허용됩니다.');
    }
    
    // JSON 데이터 읽기
    $json_input = file_get_contents('php://input');
    error_log("Bank transfer request input: " . $json_input);
    
    if (empty($json_input)) {
        throw new Exception('요청 데이터가 비어있습니다.');
    }
    
    $input = json_decode($json_input, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('JSON 파싱 에러: ' . json_last_error_msg());
    }
    
    $storeId = $input['store_id'] ?? '';
    $amount = $input['amount'] ?? 0;
    $depositorName = $input['depositor_name'] ?? '';
    $transferType = $input['transfer_type'] ?? '';
    $sessionId = $input['session_id'] ?? '';
    
    if (!$storeId || !$amount || !$depositorName || $transferType !== 'bank' || !$sessionId) {
        throw new Exception('필수 정보가 누락되었습니다. (가맹점, 금액, 입금자명, 송금타입, 세션ID)');
    }
    
    // 입금자명 유효성 검사
    if (strlen(trim($depositorName)) < 2) {
        throw new Exception('입금자명은 2글자 이상 입력해주세요.');
    }
    
    // ========== 트랜잭션 시작 ==========
    mysqli_begin_transaction($con, MYSQLI_TRANS_START_READ_WRITE);
    
    try {
        // ========== 중복 거래 체크 (FOR UPDATE로 락 설정) ==========
        $checkQuery = "
            SELECT id, kakao_tid, status, created_at, transfer_link,
                   TIMESTAMPDIFF(SECOND, created_at, NOW()) as elapsed_seconds
            FROM transfer 
            WHERE session_id = ? 
              AND type = 'deposit'
              AND payment_status = 'bank_requested'
              AND status IN ('pending', 'requested')
              AND created_at >= DATE_SUB(NOW(), INTERVAL 10 MINUTE)
            ORDER BY created_at DESC
            LIMIT 1
            FOR UPDATE
        ";
        
        $checkStmt = mysqli_prepare($con, $checkQuery);
        mysqli_stmt_bind_param($checkStmt, 's', $sessionId);
        mysqli_stmt_execute($checkStmt);
        $existingTransfer = mysqli_fetch_assoc(mysqli_stmt_get_result($checkStmt));
        
        if ($existingTransfer) {
            // 10분 이내에 동일 세션으로 생성된 진행 중인 은행 송금 거래가 있음
            $remainingSeconds = 600 - $existingTransfer['elapsed_seconds'];
            
            if ($remainingSeconds > 0) {
                mysqli_rollback($con);
                
                // 기존 거래 정보로 응답 (중복 생성하지 않음)
                error_log("은행 송금 중복 거래 방지: 세션 {$sessionId}에 이미 진행 중인 거래가 있음 (ID: {$existingTransfer['id']})");
                
                // 가맹점 정보 조회
                $storeQuery = "SELECT m.*, ba.bank, ba.account_holder, ba.account_number
                               FROM member m 
                               LEFT JOIN bank_account ba ON m.bank_account_id = ba.id
                               WHERE m.code = ? OR m.id = ?";
                $storeStmt = mysqli_prepare($con, $storeQuery);
                mysqli_stmt_bind_param($storeStmt, 'si', $storeId, $storeId);
                mysqli_stmt_execute($storeStmt);
                $store = mysqli_fetch_assoc(mysqli_stmt_get_result($storeStmt));
                
                // 기존 은행 정보 파싱
                $bank_info = json_decode($existingTransfer['transfer_link'], true);
                
                echo json_encode([
                    'success' => true,
                    'message' => '이미 진행 중인 은행 송금 거래가 있습니다.',
                    'existing_transaction' => true,
                    'data' => [
                        'transaction_id' => $existingTransfer['kakao_tid'],
                        'transfer_id' => $existingTransfer['id'],
                        'amount' => $amount,
                        'store_name' => $store['user_name'],
                        'store_id' => $storeId,
                        'depositor_name' => $depositorName,
                        'bank_info' => $bank_info,
                        'remaining_seconds' => $remainingSeconds
                    ]
                ]);
                exit;
            }
        }
        
        // ========== 추가 중복 체크: 동일 사용자, 동일 금액, 짧은 시간 내 요청 ==========
        $recentCheckQuery = "
            SELECT id, kakao_tid, created_at
            FROM transfer 
            WHERE user_id = (SELECT id FROM member WHERE code = ? OR id = ?)
              AND amount = ?
              AND type = 'deposit'
              AND depositor_name = ?
              AND payment_status = 'bank_requested'
              AND status IN ('pending', 'requested')
              AND created_at >= DATE_SUB(NOW(), INTERVAL 10 MINUTE)
            ORDER BY created_at DESC
            LIMIT 1
            FOR UPDATE
        ";
        
        $recentStmt = mysqli_prepare($con, $recentCheckQuery);
        mysqli_stmt_bind_param($recentStmt, 'sids', $storeId, $storeId, $amount, $depositorName);
        mysqli_stmt_execute($recentStmt);
        $recentTransfer = mysqli_fetch_assoc(mysqli_stmt_get_result($recentStmt));
        
        if ($recentTransfer) {
            mysqli_rollback($con);
            
            error_log("은행 송금 중복 거래 방지: 30초 이내 동일한 요청 감지 (사용자: {$storeId}, 금액: {$amount})");
            
            throw new Exception('너무 빠른 요청입니다. 잠시 후 다시 시도해주세요.');
        }
        
        // ========== 가맹점 정보 + 은행 계좌 정보 조회 ==========
        $storeQuery = "SELECT m.*, ba.bank, ba.account_holder, ba.account_number
                       FROM member m 
                       LEFT JOIN bank_account ba ON m.bank_account_id = ba.id
                       WHERE m.code = ? OR m.id = ?";
        $storeStmt = mysqli_prepare($con, $storeQuery);
        mysqli_stmt_bind_param($storeStmt, 'si', $storeId, $storeId);
        mysqli_stmt_execute($storeStmt);
        $store = mysqli_fetch_assoc(mysqli_stmt_get_result($storeStmt));
        
        if (!$store) {
            mysqli_rollback($con);
            throw new Exception('가맹점을 찾을 수 없습니다.');
        }
        
        // 은행 계좌 정보 확인
        $bank_info = [
            'bank' => $store['bank'],
            'account_holder' => $store['account_holder'], 
            'account_number' => $store['account_number']
        ];
        
        if (!$bank_info['bank'] || !$bank_info['account_number']) {
            mysqli_rollback($con);
            throw new Exception('가맹점의 은행 계좌 정보가 설정되지 않았습니다.');
        }
        
        // ========== 고유 거래 ID 생성 (세션 ID 포함하여 중복 방지) ==========
        $transactionId = 'BANK_' . substr($sessionId, 0, 8) . '_' . time() . '_' . rand(1000, 9999);
        
        // ========== 거래 ID 중복 체크 ==========
        $tidCheckQuery = "SELECT id FROM transfer WHERE kakao_tid = ? LIMIT 1";
        $tidCheckStmt = mysqli_prepare($con, $tidCheckQuery);
        mysqli_stmt_bind_param($tidCheckStmt, 's', $transactionId);
        mysqli_stmt_execute($tidCheckStmt);
        $tidExists = mysqli_fetch_assoc(mysqli_stmt_get_result($tidCheckStmt));
        
        if ($tidExists) {
            $transactionId = 'BANK_' . substr($sessionId, 0, 8) . '_' . time() . '_' . rand(10000, 99999);
        }
        
        // ========== DB에 은행 송금 요청 저장 ==========
        $insertQuery = "
            INSERT INTO transfer 
            (user_id, amount, type, kakao_tid, depositor_name, transfer_link, status, payment_status, session_id, created_at) 
            VALUES (?, ?, 'deposit', ?, ?, ?, 'pending', 'bank_requested', ?, NOW())
        ";
        
        $insertStmt = mysqli_prepare($con, $insertQuery);
        
        // transfer_link에는 은행 정보를 JSON으로 저장
        $bankInfoJson = json_encode($bank_info, JSON_UNESCAPED_UNICODE);
        
        mysqli_stmt_bind_param($insertStmt, 'iissss', 
            $store['id'], 
            $amount, 
            $transactionId, 
            $depositorName, 
            $bankInfoJson, 
            $sessionId
        );
        
        if (!mysqli_stmt_execute($insertStmt)) {
            mysqli_rollback($con);
            throw new Exception('송금 요청 저장에 실패했습니다.');
        }
        
        $transferId = mysqli_insert_id($con);
        
        if (!$transferId) {
            mysqli_rollback($con);
            throw new Exception('송금 요청 저장에 실패했습니다.');
        }
        
        // ========== 트랜잭션 커밋 ==========
        mysqli_commit($con);
        
        error_log("은행 송금 요청 생성 완료: ID={$transferId}, TID={$transactionId}, Amount={$amount}원, Bank={$bank_info['bank']}, Session={$sessionId}");
        
        echo json_encode([
            'success' => true,
            'message' => '은행 송금 요청이 생성되었습니다.',
            'data' => [
                'transaction_id' => $transactionId,
                'transfer_id' => $transferId,
                'amount' => $amount,
                'store_name' => $store['user_name'],
                'store_id' => $storeId,
                'depositor_name' => $depositorName,
                'bank_info' => $bank_info
            ]
        ]);
        
    } catch (Exception $e) {
        mysqli_rollback($con);
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("은행 송금 요청 오류: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>