<?php
// chk_y_history_stats.php
// 매일 00:00:01에 실행 - 전날 거래 통계 및 모든 회원의 현재 잔액 정산

error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// 데이터베이스 연결
require_once __DIR__ . '/../inc/db_connect.php';

// 실행 시작 시간 기록
$startTime = microtime(true);

echo "===== Daily History Settlement Script =====\n";
echo "Started at: " . date('Y-m-d H:i:s') . "\n\n";

// 데이터베이스 연결 확인
if (!isset($con) || !$con) {
    die("Database connection failed\n");
}

try {
    // 트랜잭션 시작
    mysqli_begin_transaction($con);
    
    // 어제 날짜 설정 (00:00:01에 실행되므로 어제 데이터 처리)
    $targetDate = date('Y-m-d', strtotime('-1 day'));
    $targetTimestamp = strtotime($targetDate);
    $targetDateStart = $targetDate . ' 00:00:00';
    $targetDateEnd = date('Y-m-d') . ' 00:00:00'; // 오늘 00:00:00
    
    echo "Processing date: {$targetDate} (timestamp: {$targetTimestamp})\n";
    echo "Date range: {$targetDateStart} ~ {$targetDateEnd}\n\n";
    
    $updateCount = 0;
    $insertCount = 0;
    $skipCount = 0;
    $errors = [];
    
    // 1. 모든 회원 조회 (잔액 스냅샷용) - status 조건 제거하고 모든 회원 조회
    $allMembersQuery = "SELECT id as user_id, amount FROM member";
    $allMembersStmt = mysqli_prepare($con, $allMembersQuery);
    
    if (!mysqli_stmt_execute($allMembersStmt)) {
        throw new Exception("Failed to execute members query: " . mysqli_error($con));
    }
    
    $allMembersResult = mysqli_stmt_get_result($allMembersStmt);
    
    $allMembers = [];
    while ($row = mysqli_fetch_assoc($allMembersResult)) {
        $allMembers[$row['user_id']] = intval($row['amount'] ?? 0);
    }
    mysqli_stmt_close($allMembersStmt);
    
    echo "Total members to process: " . count($allMembers) . "\n";
    
    if (count($allMembers) == 0) {
        throw new Exception("No members found in database!");
    }
    
    // 2. 어제 거래한 사용자들의 거래 통계 조회
    $statsQuery = "
        SELECT 
            th.user_id,
            SUM(CASE 
                WHEN th.type = 'deposit' THEN th.requested_amount 
                ELSE 0 
            END) as total_deposit,
            SUM(CASE 
                WHEN th.type IN ('withdraw', 'withdrawal') THEN th.requested_amount 
                ELSE 0 
            END) as total_withdraw,
            SUM(CASE 
                WHEN th.type = 'deposit' THEN IFNULL(th.fee_amount, 0)
                ELSE 0 
            END) as total_deposit_fee,
            SUM(CASE 
                WHEN th.type IN ('withdraw', 'withdrawal') THEN IFNULL(th.fee_amount, 0)
                ELSE 0 
            END) as total_withdraw_fee,
            SUM(IFNULL(th.total_fee_amount, 0)) as total_actual_fee,
            SUM(IFNULL(th.total_fee_amount, 0) - IFNULL(th.fee_amount, 0)) as total_rolling_paid
        FROM transaction_history th
        WHERE th.status = 'completed'
        AND th.created_at >= ?
        AND th.created_at < ?
        GROUP BY th.user_id
    ";
    
    $statsStmt = mysqli_prepare($con, $statsQuery);
    mysqli_stmt_bind_param($statsStmt, "ss", $targetDateStart, $targetDateEnd);
    
    if (!mysqli_stmt_execute($statsStmt)) {
        echo "Warning: Failed to execute transaction stats query: " . mysqli_error($con) . "\n";
    }
    
    $statsResult = mysqli_stmt_get_result($statsStmt);
    
    $transactionStats = [];
    while ($row = mysqli_fetch_assoc($statsResult)) {
        $transactionStats[$row['user_id']] = $row;
    }
    mysqli_stmt_close($statsStmt);
    
    echo "Transactions Found: " . count($transactionStats) . "\n";
    
    // 3. 롤링 수익 통계 조회
    $rollingQuery = "
        SELECT 
            rh.parent_member_id as user_id,
            SUM(rh.rolling_amount) as total_rolling_received
        FROM rolling_history rh
        JOIN transaction_history th ON rh.original_transaction_id = th.id
        WHERE th.status = 'completed'
        AND th.created_at >= ?
        AND th.created_at < ?
        GROUP BY rh.parent_member_id
    ";
    
    $rollingStmt = mysqli_prepare($con, $rollingQuery);
    mysqli_stmt_bind_param($rollingStmt, "ss", $targetDateStart, $targetDateEnd);
    
    if (!mysqli_stmt_execute($rollingStmt)) {
        echo "Warning: Failed to execute rolling stats query: " . mysqli_error($con) . "\n";
    }
    
    $rollingResult = mysqli_stmt_get_result($rollingStmt);
    
    $rollingStats = [];
    while ($row = mysqli_fetch_assoc($rollingResult)) {
        $rollingStats[$row['user_id']] = intval($row['total_rolling_received'] ?? 0);
    }
    mysqli_stmt_close($rollingStmt);
    
    echo "Rolling Recipients: " . count($rollingStats) . "\n\n";
    echo "Processing members...\n";
    
    // 4. 모든 회원에 대해 history 레코드 생성/업데이트
    foreach ($allMembers as $userId => $currentBalance) {
        // 거래 통계 (있는 경우)
        $totalDeposit = isset($transactionStats[$userId]) ? intval($transactionStats[$userId]['total_deposit'] ?? 0) : 0;
        $totalWithdraw = isset($transactionStats[$userId]) ? intval($transactionStats[$userId]['total_withdraw'] ?? 0) : 0;
        $totalDepositFee = isset($transactionStats[$userId]) ? intval($transactionStats[$userId]['total_deposit_fee'] ?? 0) : 0;
        $totalWithdrawFee = isset($transactionStats[$userId]) ? intval($transactionStats[$userId]['total_withdraw_fee'] ?? 0) : 0;
        $totalActualFee = isset($transactionStats[$userId]) ? intval($transactionStats[$userId]['total_actual_fee'] ?? 0) : 0;
        $totalRollingPaid = isset($transactionStats[$userId]) ? intval($transactionStats[$userId]['total_rolling_paid'] ?? 0) : 0;
        
        // 롤링 수익 (있는 경우)
        $totalRollingReceived = isset($rollingStats[$userId]) ? $rollingStats[$userId] : 0;
        
        // 현재 잔액 (모든 회원)
        $lastBalance = $currentBalance;
        
        // 거래가 없고 잔액도 0인 경우 스킵할지 결정 (선택사항)
        // if ($totalDeposit == 0 && $totalWithdraw == 0 && $totalRollingReceived == 0 && $lastBalance == 0) {
        //     $skipCount++;
        //     continue; // 이 줄을 주석처리하면 모든 회원 기록
        // }
        
        // 기존 레코드 확인
        $checkQuery = "SELECT id FROM history WHERE stat_date = ? AND user_id = ?";
        $checkStmt = mysqli_prepare($con, $checkQuery);
        mysqli_stmt_bind_param($checkStmt, "ii", $targetTimestamp, $userId);
        mysqli_stmt_execute($checkStmt);
        $checkResult = mysqli_stmt_get_result($checkStmt);
        
        if (mysqli_num_rows($checkResult) > 0) {
            // UPDATE
            $updateQuery = "
                UPDATE history 
                SET total_deposit = ?,
                    total_withdraw = ?,
                    total_deposit_fee = ?,
                    total_withdraw_fee = ?,
                    total_actual_fee = ?,
                    total_rolling_paid = ?,
                    total_rolling_received = ?,
                    last_balance = ?,
                    updated_at = NOW()
                WHERE stat_date = ? AND user_id = ?
            ";
            
            $updateStmt = mysqli_prepare($con, $updateQuery);
            mysqli_stmt_bind_param($updateStmt, "iiiiiiiiii", 
                $totalDeposit, 
                $totalWithdraw, 
                $totalDepositFee, 
                $totalWithdrawFee,
                $totalActualFee,
                $totalRollingPaid,
                $totalRollingReceived,
                $lastBalance,
                $targetTimestamp,
                $userId
            );
            
            if (mysqli_stmt_execute($updateStmt)) {
                $updateCount++;
                echo "  Updated user {$userId}\n";
            } else {
                $errors[] = "Update failed for user {$userId}: " . mysqli_error($con);
            }
            mysqli_stmt_close($updateStmt);
            
        } else {
            // INSERT
            $insertQuery = "
                INSERT INTO history 
                (stat_date, user_id, total_deposit, total_withdraw, 
                 total_deposit_fee, total_withdraw_fee, total_actual_fee,
                 total_rolling_paid, total_rolling_received, last_balance,
                 created_at, updated_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
            ";
            
            $insertStmt = mysqli_prepare($con, $insertQuery);
            mysqli_stmt_bind_param($insertStmt, "iiiiiiiiii", 
                $targetTimestamp,
                $userId,
                $totalDeposit, 
                $totalWithdraw, 
                $totalDepositFee, 
                $totalWithdrawFee,
                $totalActualFee,
                $totalRollingPaid,
                $totalRollingReceived,
                $lastBalance
            );
            
            if (mysqli_stmt_execute($insertStmt)) {
                $insertCount++;
                echo "  Inserted user {$userId}\n";
            } else {
                $errors[] = "Insert failed for user {$userId}: " . mysqli_error($con);
            }
            mysqli_stmt_close($insertStmt);
        }
        
        mysqli_stmt_close($checkStmt);
    }
    
    // 트랜잭션 커밋
    mysqli_commit($con);
    
    // 실행 시간 계산
    $executionTime = round(microtime(true) - $startTime, 3);
    
    // 결과 출력
    echo "\n========================================\n";
    echo "=== Daily Settlement Completed ===\n";
    echo "========================================\n";
    echo "Target Date: {$targetDate}\n";
    echo "Execution Time: {$executionTime} seconds\n";
    echo "Total Members: " . count($allMembers) . "\n";
    echo "Updated: {$updateCount}\n";
    echo "Inserted: {$insertCount}\n";
    echo "Skipped (no activity): {$skipCount}\n";
    echo "Transactions Found: " . count($transactionStats) . "\n";
    echo "Rolling Recipients: " . count($rollingStats) . "\n";
    
    if (count($errors) > 0) {
        echo "\n⚠ Errors: " . count($errors) . "\n";
        foreach ($errors as $error) {
            echo "  - {$error}\n";
        }
    }
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    mysqli_rollback($con);
    
    echo "\n========================================\n";
    echo "=== ERROR OCCURRED ===\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
    echo "Line: " . $e->getLine() . "\n";
    echo "File: " . $e->getFile() . "\n";
    
    error_log("Daily Settlement Error: " . $e->getMessage());
    
    exit(1);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
        echo "\nDatabase connection closed.\n";
    }
}
?>