<?php
// update_history_stats.php
// 1분마다 실행되는 통계 업데이트 스크립트 (최적화 버전)

error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('memory_limit', '256M'); // 메모리 제한 설정

// 데이터베이스 연결
require_once __DIR__ . '/../inc/db_connect.php';

// ==========================================
// 1. 중복 실행 방지
// ==========================================
$lockFile = '/tmp/update_history_stats.lock';
$lockHandle = fopen($lockFile, 'w');

if (!flock($lockHandle, LOCK_EX | LOCK_NB)) {
    echo "Another instance is already running. Exiting...\n";
    exit(0);
}

// 실행 시작 시간 기록
$startTime = microtime(true);

echo "========================================\n";
echo "Script started at: " . date('Y-m-d H:i:s') . "\n";
echo "========================================\n";

// 데이터베이스 연결 확인
if (!isset($con) || !$con) {
    flock($lockHandle, LOCK_UN);
    fclose($lockHandle);
    @unlink($lockFile);
    die("Database connection failed\n");
}

echo "✓ Database connected\n";

try {
    // 트랜잭션 시작
    mysqli_begin_transaction($con);
    
    // 현재 시간 처리
    $currentTimestamp = time();
    $todayTimestamp = strtotime('today');
    $currentDate = date('Y-m-d');
    $currentDateTime = date('Y-m-d H:i:s');
    $todayStart = $currentDate . ' 00:00:00';
    $tomorrowStart = date('Y-m-d', strtotime('+1 day')) . ' 00:00:00';
    
    echo "✓ Current date: {$currentDate}\n";
    echo "✓ Processing range: {$todayStart} ~ {$tomorrowStart}\n\n";
    
    // ==========================================
    // 2. 최적화된 통계 쿼리 (N+1 문제 해결)
    // ==========================================
    echo "=== Fetching statistics ===\n";
    
    $statsQuery = "
        SELECT 
            th.user_id,
            ? as stat_date,
            -- 입출금 집계
            SUM(CASE 
                WHEN th.type = 'deposit' THEN th.requested_amount 
                ELSE 0 
            END) as total_deposit,
            SUM(CASE 
                WHEN th.type = 'withdraw' OR th.type = 'withdrawal' THEN th.requested_amount 
                ELSE 0 
            END) as total_withdraw,
            -- 수수료 집계
            SUM(CASE 
                WHEN th.type = 'deposit' THEN IFNULL(th.fee_amount, 0)
                ELSE 0 
            END) as total_deposit_fee,
            SUM(CASE 
                WHEN th.type = 'withdraw' OR th.type = 'withdrawal' THEN IFNULL(th.fee_amount, 0)
                ELSE 0 
            END) as total_withdraw_fee,
            -- 실제 수수료
            SUM(CASE 
                WHEN th.type = 'deposit' THEN IFNULL(th.total_fee_amount, 0)
                WHEN th.type = 'withdraw' OR th.type = 'withdrawal' THEN IFNULL(th.total_fee_amount, 0)
                ELSE 0 
            END) as total_actual_fee,
            -- 롤링 지급액 (하위에게 지급한 금액)
            SUM(CASE 
                WHEN th.type = 'deposit' THEN IFNULL(th.total_fee_amount, 0) - IFNULL(th.fee_amount, 0)
                WHEN th.type = 'withdraw' OR th.type = 'withdrawal' THEN IFNULL(th.total_fee_amount, 0) - IFNULL(th.fee_amount, 0)
                ELSE 0 
            END) as total_rolling_paid,
            -- 롤링 수익 (상위로부터 받은 금액) - JOIN으로 최적화
            IFNULL(rolling_received.total_rolling_received, 0) as total_rolling_received
        FROM transaction_history th
        LEFT JOIN (
            SELECT 
                rh.parent_member_id,
                SUM(rh.rolling_amount) as total_rolling_received
            FROM rolling_history rh
            JOIN transaction_history th2 ON rh.original_transaction_id = th2.id
            WHERE th2.status = 'completed'
            AND th2.created_at >= ?
            AND th2.created_at < ?
            GROUP BY rh.parent_member_id
        ) rolling_received ON th.user_id = rolling_received.parent_member_id
        WHERE th.status = 'completed'
        AND th.created_at >= ?
        AND th.created_at < ?
        GROUP BY th.user_id
    ";
    
    $statsStmt = mysqli_prepare($con, $statsQuery);
    
    if (!$statsStmt) {
        throw new Exception("Stats query prepare failed: " . mysqli_error($con));
    }
    
    // 바인딩: i(stat_date), ss(rolling 서브쿼리용), ss(메인 쿼리용)
    mysqli_stmt_bind_param($statsStmt, "issss", 
        $todayTimestamp,
        $todayStart, 
        $tomorrowStart,
        $todayStart, 
        $tomorrowStart
    );
    
    if (!mysqli_stmt_execute($statsStmt)) {
        throw new Exception("Stats query execute failed: " . mysqli_stmt_error($statsStmt));
    }
    
    $statsResult = mysqli_stmt_get_result($statsStmt);
    
    if (!$statsResult) {
        throw new Exception("Stats query get result failed: " . mysqli_stmt_error($statsStmt));
    }
    
    $totalRows = mysqli_num_rows($statsResult);
    echo "✓ Found {$totalRows} users with transactions\n\n";
    
    // ==========================================
    // 3. UPSERT 방식으로 INSERT/UPDATE 통합
    // ==========================================
    echo "=== Processing user statistics ===\n";
    
    // history 테이블에 UNIQUE KEY가 있어야 함: UNIQUE KEY (stat_date, user_id)
    $upsertQuery = "
        INSERT INTO history 
        (stat_date, user_id, total_deposit, total_withdraw, 
         total_deposit_fee, total_withdraw_fee, total_actual_fee,
         total_rolling_paid, total_rolling_received, created_at, updated_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
            total_deposit = VALUES(total_deposit),
            total_withdraw = VALUES(total_withdraw),
            total_deposit_fee = VALUES(total_deposit_fee),
            total_withdraw_fee = VALUES(total_withdraw_fee),
            total_actual_fee = VALUES(total_actual_fee),
            total_rolling_paid = VALUES(total_rolling_paid),
            total_rolling_received = VALUES(total_rolling_received),
            updated_at = VALUES(updated_at)
    ";
    
    $upsertStmt = mysqli_prepare($con, $upsertQuery);
    
    if (!$upsertStmt) {
        throw new Exception("Upsert query prepare failed: " . mysqli_error($con));
    }
    
    $processedCount = 0;
    $errorCount = 0;
    $errors = [];
    
    // 각 사용자별 통계 처리
    while ($row = mysqli_fetch_assoc($statsResult)) {
        $userId = intval($row['user_id']);
        $statDate = intval($row['stat_date']);
        $totalDeposit = intval($row['total_deposit'] ?? 0);
        $totalWithdraw = intval($row['total_withdraw'] ?? 0);
        $totalDepositFee = intval($row['total_deposit_fee'] ?? 0);
        $totalWithdrawFee = intval($row['total_withdraw_fee'] ?? 0);
        $totalActualFee = intval($row['total_actual_fee'] ?? 0);
        $totalRollingPaid = intval($row['total_rolling_paid'] ?? 0);
        $totalRollingReceived = intval($row['total_rolling_received'] ?? 0);
        
        echo "Processing user_id: {$userId}\n";
        echo "  Deposit: {$totalDeposit} | Withdraw: {$totalWithdraw}\n";
        echo "  Fee (D/W): {$totalDepositFee}/{$totalWithdrawFee} | Actual Fee: {$totalActualFee}\n";
        echo "  Rolling (Paid/Received): {$totalRollingPaid}/{$totalRollingReceived}\n";
        
        // UPSERT 실행
        mysqli_stmt_bind_param($upsertStmt, "iiiiiiiisss", 
            $statDate,
            $userId,
            $totalDeposit, 
            $totalWithdraw, 
            $totalDepositFee, 
            $totalWithdrawFee,
            $totalActualFee,
            $totalRollingPaid,
            $totalRollingReceived,
            $currentDateTime,
            $currentDateTime
        );
        
        if (mysqli_stmt_execute($upsertStmt)) {
            $processedCount++;
            echo "  ✓ Processed successfully\n\n";
        } else {
            $errorCount++;
            $errorMsg = "Failed for user_id: {$userId} - " . mysqli_stmt_error($upsertStmt);
            $errors[] = $errorMsg;
            echo "  ✗ {$errorMsg}\n\n";
            
            // 개별 에러가 발생해도 계속 진행할지, 아니면 전체 롤백할지 결정
            // 현재는 에러를 기록하고 계속 진행
        }
    }
    
    mysqli_stmt_close($statsStmt);
    mysqli_stmt_close($upsertStmt);
    
    // ==========================================
    // 4. 롤링만 받은 사용자 처리
    // ==========================================
    echo "\n=== Processing rolling-only users ===\n";
    
    $rollingOnlyQuery = "
        SELECT DISTINCT
            rh.parent_member_id as user_id,
            ? as stat_date,
            SUM(rh.rolling_amount) as total_rolling_received
        FROM rolling_history rh
        JOIN transaction_history th ON rh.original_transaction_id = th.id
        WHERE th.status = 'completed'
        AND th.created_at >= ?
        AND th.created_at < ?
        AND rh.parent_member_id NOT IN (
            SELECT DISTINCT user_id 
            FROM transaction_history 
            WHERE status = 'completed'
            AND created_at >= ?
            AND created_at < ?
        )
        GROUP BY rh.parent_member_id
    ";
    
    $rollingOnlyStmt = mysqli_prepare($con, $rollingOnlyQuery);
    
    if (!$rollingOnlyStmt) {
        throw new Exception("Rolling-only query prepare failed: " . mysqli_error($con));
    }
    
    mysqli_stmt_bind_param($rollingOnlyStmt, "issss", 
        $todayTimestamp, 
        $todayStart, 
        $tomorrowStart,
        $todayStart, 
        $tomorrowStart
    );
    
    if (!mysqli_stmt_execute($rollingOnlyStmt)) {
        throw new Exception("Rolling-only query execute failed: " . mysqli_stmt_error($rollingOnlyStmt));
    }
    
    $rollingOnlyResult = mysqli_stmt_get_result($rollingOnlyStmt);
    $rollingOnlyCount = mysqli_num_rows($rollingOnlyResult);
    
    echo "✓ Found {$rollingOnlyCount} rolling-only users\n\n";
    
    // UPSERT 준비 (롤링만 있는 사용자용)
    $upsertRollingQuery = "
        INSERT INTO history 
        (stat_date, user_id, total_deposit, total_withdraw, 
         total_deposit_fee, total_withdraw_fee, total_actual_fee,
         total_rolling_paid, total_rolling_received, created_at, updated_at)
        VALUES (?, ?, 0, 0, 0, 0, 0, 0, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
            total_rolling_received = total_rolling_received + VALUES(total_rolling_received),
            updated_at = VALUES(updated_at)
    ";
    
    $upsertRollingStmt = mysqli_prepare($con, $upsertRollingQuery);
    
    if (!$upsertRollingStmt) {
        throw new Exception("Upsert rolling query prepare failed: " . mysqli_error($con));
    }
    
    while ($row = mysqli_fetch_assoc($rollingOnlyResult)) {
        $userId = intval($row['user_id']);
        $statDate = intval($row['stat_date']);
        $totalRollingReceived = intval($row['total_rolling_received'] ?? 0);
        
        echo "Processing rolling-only user_id: {$userId}\n";
        echo "  Rolling Received: {$totalRollingReceived}\n";
        
        mysqli_stmt_bind_param($upsertRollingStmt, "iisss", 
            $statDate,
            $userId,
            $totalRollingReceived,
            $currentDateTime,
            $currentDateTime
        );
        
        if (mysqli_stmt_execute($upsertRollingStmt)) {
            $processedCount++;
            echo "  ✓ Processed successfully\n\n";
        } else {
            $errorCount++;
            $errorMsg = "Rolling-only failed for user_id: {$userId} - " . mysqli_stmt_error($upsertRollingStmt);
            $errors[] = $errorMsg;
            echo "  ✗ {$errorMsg}\n\n";
        }
    }
    
    mysqli_stmt_close($rollingOnlyStmt);
    mysqli_stmt_close($upsertRollingStmt);
    
    // ==========================================
    // 5. 트랜잭션 커밋
    // ==========================================
    
    // 에러가 있으면 롤백할지 결정 (선택사항)
    if ($errorCount > 0 && $errorCount >= $processedCount * 0.1) {
        // 에러가 10% 이상이면 롤백
        throw new Exception("Too many errors occurred ({$errorCount}/{$processedCount}). Rolling back.");
    }
    
    mysqli_commit($con);
    
    // ==========================================
    // 6. 실행 결과 출력
    // ==========================================
    $executionTime = round(microtime(true) - $startTime, 3);
    
    echo "\n========================================\n";
    echo "=== Execution Completed ===\n";
    echo "========================================\n";
    echo "Date: {$currentDate}\n";
    echo "Timestamp: {$todayTimestamp}\n";
    echo "✓ Successfully processed: {$processedCount} records\n";
    echo "✗ Errors: {$errorCount} records\n";
    echo "⏱ Execution time: {$executionTime} seconds\n";
    
    if ($errorCount > 0) {
        echo "\n⚠ Error Details:\n";
        foreach ($errors as $i => $error) {
            echo "  " . ($i + 1) . ". {$error}\n";
        }
    } else {
        echo "\n✓ All operations completed without errors\n";
    }
    
    echo "========================================\n";
    
    // 로그 기록 (선택사항)
    if (function_exists('writeLog')) {
        $logData = [
            'date' => $currentDate,
            'timestamp' => $todayTimestamp,
            'processed' => $processedCount,
            'errors' => $errorCount,
            'execution_time' => $executionTime
        ];
        
        $detail = "통계 업데이트 완료 - 날짜: {$currentDate}, 처리: {$processedCount}건, 오류: {$errorCount}건, 실행시간: {$executionTime}초";
        
        if ($errorCount > 0) {
            $logData['error_details'] = $errors;
        }
        
        writeLog($con, 'HISTORY_STATS_UPDATE', null, $logData, 1, $detail, 0);
    }
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    mysqli_rollback($con);
    
    // 에러 출력
    echo "\n========================================\n";
    echo "=== CRITICAL ERROR ===\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
    echo "File: " . $e->getFile() . "\n";
    echo "Line: " . $e->getLine() . "\n";
    echo "========================================\n";
    
    error_log("History Stats Update Error: " . $e->getMessage());
    
    // 로그 기록 (선택사항)
    if (function_exists('writeLog') && isset($con) && $con) {
        $logData = [
            'error' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ];
        writeLog($con, 'HISTORY_STATS_UPDATE_ERROR', null, $logData, 0, '통계 업데이트 실패: ' . $e->getMessage(), 0);
    }
    
    exit(1);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
        echo "\n✓ Database connection closed\n";
    }
    
    // 락 파일 해제
    if (isset($lockHandle) && $lockHandle) {
        flock($lockHandle, LOCK_UN);
        fclose($lockHandle);
        @unlink($lockFile);
        echo "✓ Lock file released\n";
    }
    
    echo "\nScript finished at: " . date('Y-m-d H:i:s') . "\n";
}
?>