<?php

include 'layout/header.php';

// 사용자 권한 확인
$isAdmin = isset($_SESSION['grade']) && $_SESSION['grade'] === 'admin';

// 파일 상단 PHP 부분에 추가
$today = date('Y-m-d');
$yesterday = date('Y-m-d', strtotime('-1 day'));
?>

<style>
/* 자동승인 관련 스타일 */
@keyframes pulse-subtle {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.8; }
}

.animate-pulse-subtle {
    animation: pulse-subtle 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
}

/* 자동승인 대기 행 */
.auto-approval-row {
    background: linear-gradient(90deg, rgba(16, 185, 129, 0.15) 0%, rgba(16, 185, 129, 0.05) 100%);
    border-left: 4px solid #10b981;
    animation: pulse-subtle 3s ease-in-out infinite;
}

.auto-approval-row:hover {
    background: linear-gradient(90deg, rgba(16, 185, 129, 0.25) 0%, rgba(16, 185, 129, 0.1) 100%);
}

/* 자동승인 카운트다운 */
.countdown-badge {
    background: linear-gradient(135deg, #10b981, #059669);
    color: white;
    font-size: 11px;
    font-weight: 600;
    padding: 3px 8px;
    border-radius: 9999px;
    display: inline-flex;
    align-items: center;
    gap: 4px;
    font-variant-numeric: tabular-nums;
}

/* 중복 거래 행 스타일 */
.duplicate-row {
    background: linear-gradient(90deg, rgba(251, 146, 60, 0.15) 0%, rgba(251, 146, 60, 0.05) 100%);
    border-left: 4px solid #f59e0b;
}

.duplicate-row:hover {
    background: linear-gradient(90deg, rgba(251, 146, 60, 0.25) 0%, rgba(251, 146, 60, 0.1) 100%);
}

/* 중복 표시기 스타일 */
.duplicate-indicator {
    background: linear-gradient(135deg, #f59e0b, #d97706);
    color: white;
    font-size: 10px;
    font-weight: 600;
    padding: 2px 6px;
    border-radius: 9999px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* 알림 박스 스타일 */
.notification-box {
    border-radius: 8px;
    backdrop-filter: blur(8px);
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* 상태 인디케이터 */
.status-indicators {
    display: flex;
    align-items: center;
    gap: 4px;
}

.indicator-dot {
    width: 8px;
    height: 8px;
    border-radius: 50%;
    display: inline-block;
}

.indicator-new {
    background: #ef4444;
    animation: pulse 1.5s cubic-bezier(0.4, 0, 0.6, 1) infinite;
}

.indicator-auto-approval {
    background: #10b981;
    animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
}

.indicator-duplicate {
    background: #f59e0b;
}

/* 승인 유형 배지 애니메이션 */
.auto-approval-type-badge {
    animation: subtle-glow 2s ease-in-out infinite;
}

@keyframes subtle-glow {
    0%, 100% { 
        opacity: 1;
        box-shadow: 0 0 5px rgba(59, 130, 246, 0.3);
    }
    50% { 
        opacity: 0.9;
        box-shadow: 0 0 10px rgba(59, 130, 246, 0.5);
    }
}

/* 반응형 */
@media (max-width: 768px) {
    .duplicate-indicator, .countdown-badge {
        font-size: 9px;
        padding: 2px 5px;
    }
    
    .notification-box {
        font-size: 14px;
        padding: 12px;
    }
}

/* 추가 애니메이션 스타일 */
@keyframes slide-in {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

@keyframes fade-out {
    from {
        opacity: 1;
    }
    to {
        opacity: 0;
    }
}

@keyframes pulse {
    0%, 100% {
        opacity: 1;
    }
    50% {
        opacity: .5;
    }
}

@keyframes spin {
    from {
        transform: rotate(0deg);
    }
    to {
        transform: rotate(360deg);
    }
}

.animate-slide-in {
    animation: slide-in 0.3s ease-out;
}

.animate-fade-out {
    animation: fade-out 0.3s ease-out;
}

.animate-spin {
    animation: spin 2s linear infinite;
}

/* 커스텀 스크롤바 */
.overflow-x-auto::-webkit-scrollbar {
    height: 8px;
}

.overflow-x-auto::-webkit-scrollbar-track {
    background: rgba(55, 65, 81, 0.3);
    border-radius: 4px;
}

.overflow-x-auto::-webkit-scrollbar-thumb {
    background: rgba(107, 114, 128, 0.5);
    border-radius: 4px;
}

.overflow-x-auto::-webkit-scrollbar-thumb:hover {
    background: rgba(107, 114, 128, 0.7);
}

/* 테이블 정렬 화살표 */
.sort-indicator {
    opacity: 0.4;
    transition: opacity 0.3s ease;
}

th:hover .sort-indicator {
    opacity: 0.8;
}
</style>

<!-- 모바일 메뉴 열기 버튼 -->
<button class="md:hidden fixed top-4 left-4 z-40 bg-gradient-to-r from-blue-600 to-purple-600 text-white p-3 rounded-full shadow-lg hover:shadow-xl transition-all duration-300" onclick="toggleSidebar()">
    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
    </svg>
</button>

<!-- 금액 수정 모달 -->
<div id="amountEditModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
    <div class="bg-gradient-to-br from-gray-800 to-gray-900 p-6 rounded-2xl shadow-2xl border border-gray-700/50 backdrop-blur-sm w-full max-w-md">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-white">금액 수정</h3>
            <button onclick="closeAmountModal()" class="text-gray-400 hover:text-white transition-colors">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        
        <input type="hidden" id="editDepositId">
        <div class="space-y-4">
            <div>
                <label class="block text-sm font-medium text-gray-300 mb-2">수정할 금액</label>
                <div class="relative">
                    <input type="text" id="editAmount"
                        class="w-full px-4 py-3 bg-gray-700/50 border border-gray-600 text-white rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300 pr-12"
                        placeholder="금액을 입력하세요"
                        inputmode="numeric">
                    <span class="absolute right-4 top-1/2 transform -translate-y-1/2 text-gray-400">원</span>
                </div>
            </div>
            
            <div class="flex justify-end space-x-3">
                <button onclick="closeAmountModal()" 
                    class="px-4 py-2 bg-gray-600 hover:bg-gray-700 text-white rounded-lg transition-all duration-300">
                    취소
                </button>
                <button onclick="saveAmount()" 
                    class="px-4 py-2 bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white rounded-lg transition-all duration-300">
                    저장
                </button>
            </div>
        </div>
    </div>
</div>

<main class="flex-1 p-6 space-y-8 bg-gradient-to-br from-gray-900 via-gray-800 to-gray-900 min-h-screen">
    <!-- 대시보드 요약 (5개 카드로 변경) -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6 max-w-7xl mx-auto">
        <!-- 오늘 입금 신청 -->
        <div class="bg-gradient-to-br from-blue-500/10 to-blue-600/10 border border-blue-500/20 rounded-xl p-6">
            <div class="flex items-center justify-between mb-4">
                <div class="w-12 h-12 bg-blue-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                    </svg>
                </div>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">오늘 입금 신청</h3>
                <p class="text-2xl font-bold text-white" id="todayCount">0건</p>
                <p class="text-sm text-blue-400" id="todayAmount">0원</p>
            </div>
        </div>

        <!-- 자동승인 대기 (새로 추가) -->
        <div class="bg-gradient-to-br from-emerald-500/10 to-emerald-600/10 border border-emerald-500/20 rounded-xl p-6">
            <div class="flex items-center justify-between mb-4">
                <div class="w-12 h-12 bg-emerald-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-emerald-400 animate-pulse" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                </div>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">자동승인 대기</h3>
                <p class="text-2xl font-bold text-white" id="autoApprovalCount">0건</p>
                <p class="text-sm text-emerald-400" id="autoApprovalAmount">0원</p>
            </div>
        </div>

        <!-- 처리 대기중 -->
        <div class="bg-gradient-to-br from-yellow-500/10 to-yellow-600/10 border border-yellow-500/20 rounded-xl p-6">
            <div class="flex items-center justify-between mb-4">
                <div class="w-12 h-12 bg-yellow-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-yellow-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                </div>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">처리 대기중</h3>
                <p class="text-2xl font-bold text-white" id="pendingCount">0건</p>
                <p class="text-sm text-yellow-400" id="pendingAmount">0원</p>
            </div>
        </div>

        <!-- 승인된 입금 -->
        <div class="bg-gradient-to-br from-green-500/10 to-green-600/10 border border-green-500/20 rounded-xl p-6">
            <div class="flex items-center justify-between mb-4">
                <div class="w-12 h-12 bg-green-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                    </svg>
                </div>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">오늘 승인 완료</h3>
                <p class="text-2xl font-bold text-white" id="completedCount">0건</p>
                <p class="text-sm text-green-400" id="completedAmount">0원</p>
            </div>
        </div>

        <!-- 총 수수료 -->
        <div class="bg-gradient-to-br from-purple-500/10 to-purple-600/10 border border-purple-500/20 rounded-xl p-6">
            <div class="flex items-center justify-between mb-4">
                <div class="w-12 h-12 bg-purple-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-purple-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2-2V7a2 2 0 012-2h2a2 2 0 002 2v2a2 2 002 2h2a2 2 0 012-2V7a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 00-2 2h-2a2 2 0 00-2 2v6"></path>
                    </svg>
                </div>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">오늘 총 수수료</h3>
                <p class="text-2xl font-bold text-white" id="totalFees">0원</p>
            </div>
        </div>
    </div>

    <!-- 입금 내역 -->
    <div class="w-full">
        <div class="bg-gradient-to-br from-gray-800 to-gray-900 rounded-2xl shadow-2xl border border-gray-700/50 overflow-hidden">
            <div class="p-6 border-b border-gray-700/50">
                <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-6">
                    <div>
                        <h3 class="text-2xl font-bold text-white">입금 신청 내역</h3>
                        <p class="text-gray-400 text-sm mt-1">입금 신청 현황을 관리하세요</p>
                    </div>
                    <div class="flex items-center space-x-3">
                        <?php if ($isAdmin): ?>
                        <!-- 엑셀 다운로드 버튼 -->
                        <button onclick="downloadExcel()" class="flex items-center px-4 py-2 text-sm text-green-400 hover:text-green-300 hover:bg-green-500/10 rounded-lg transition-all duration-300">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                            </svg>
                            엑셀 다운로드
                        </button>
                        <?php endif; ?>
                        
                        <!-- 자동 새로고침 토글 버튼 -->
                        <button id="autoRefreshToggle" onclick="toggleAutoRefresh()" class="flex items-center px-4 py-2 text-sm bg-gray-700 hover:bg-gray-600 rounded-lg transition-all duration-300">
                            <svg id="autoRefreshIcon" class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                            </svg>
                            <span id="autoRefreshText" class="text-white">자동 새로고침 ON</span>
                        </button>
                        
                        <!-- 수동 새로고침 버튼 -->
                        <button onclick="fetchDeposits()" class="flex items-center px-4 py-2 text-sm text-blue-400 hover:text-blue-300 hover:bg-blue-500/10 rounded-lg transition-all duration-300">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                            </svg>
                            새로고침
                        </button>
                    </div>
                </div>

                <!-- 알림 컨테이너 (동적으로 추가됨) -->
                <div id="notificationContainer"></div>
                
                <!-- 검색 필터 -->
                <form id="searchForm" method="GET" class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-7 gap-4">
                    <input type="hidden" name="sort_by" id="sort_by" value="created_at">
                    <input type="hidden" name="sort_order" id="sort_order" value="desc">
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">시작일</label>
                        <input type="date" name="start_date" id="start_date"
                            class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">종료일</label>
                        <input type="date" name="end_date" id="end_date"
                            class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">입금자명</label>
                        <input type="text" name="search" id="search" placeholder="입금자 이름"
                            class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                    </div>
                    
                    <!-- 상태 검색 필터 -->
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">상태</label>
                        <div class="max-h-32 overflow-y-auto bg-gray-700/50 border border-gray-600 rounded-lg p-2">
                            <label class="flex items-center space-x-2 py-1 hover:bg-gray-600/50 px-2 rounded cursor-pointer">
                                <input type="checkbox" name="status[]" value="requested" class="rounded border-gray-500">
                                <span class="text-sm text-white">대기</span>
                            </label>
                            <label class="flex items-center space-x-2 py-1 hover:bg-gray-600/50 px-2 rounded cursor-pointer">
                                <input type="checkbox" name="status[]" value="pending" class="rounded border-gray-500">
                                <span class="text-sm text-white">보류</span>
                            </label>
                            <label class="flex items-center space-x-2 py-1 hover:bg-gray-600/50 px-2 rounded cursor-pointer">
                                <input type="checkbox" name="status[]" value="completed" class="rounded border-gray-500">
                                <span class="text-sm text-white">승인</span>
                            </label>
                            <label class="flex items-center space-x-2 py-1 hover:bg-gray-600/50 px-2 rounded cursor-pointer">
                                <input type="checkbox" name="status[]" value="cancelled" class="rounded border-gray-500">
                                <span class="text-sm text-white">취소</span>
                            </label>
                        </div>
                    </div>

                    <?php if ($isAdmin): ?>
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">가맹점</label>
                        <select name="store_code" class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                            <option value="">전체 가맹점</option>
                            <?php
                            $storeQuery = "SELECT id, user_name, user_id FROM member WHERE grade != 'admin' ORDER BY user_name";
                            $storeResult = mysqli_query($con, $storeQuery);
                            while ($store = mysqli_fetch_assoc($storeResult)) {
                                echo "<option value=\"{$store['id']}\">{$store['user_name']} ({$store['user_id']})</option>";
                            }
                            ?>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">카카오 연결 계정</label>
                        <div class="max-h-32 overflow-y-auto bg-gray-700/50 border border-gray-600 rounded-lg p-2">
                            <?php
                            $kakaoQuery = "SELECT id, kakao_tid, name FROM kakao_account WHERE 1";
                            $kakaoResult = mysqli_query($con, $kakaoQuery);
                            while ($kakao = mysqli_fetch_assoc($kakaoResult)) {
                                echo '<label class="flex items-center space-x-2 py-1 hover:bg-gray-600/50 px-2 rounded cursor-pointer">';
                                echo '<input type="checkbox" name="kakao_code[]" value="' . $kakao['id'] . '" class="rounded border-gray-500">';
                                echo '<span class="text-sm text-white">' . $kakao['kakao_tid'] . ' (' . $kakao['name'] . ')</span>';
                                echo '</label>';
                            }
                            ?>
                        </div>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">은행 연결 계정</label>
                        <select name="bank_code" class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                            <option value="">전체 은행계정</option>
                            <?php
                            $bankQuery = "SELECT id, bank, account_holder FROM bank_account WHERE 1";
                            $bankResult = mysqli_query($con, $bankQuery);
                            while ($bank = mysqli_fetch_assoc($bankResult)) {
                                echo "<option value=\"{$bank['id']}\">{$bank['bank']} ({$bank['account_holder']})</option>";
                            }
                            ?>
                        </select>
                    </div>
                    <?php endif; ?>
                    
                    <div class="flex items-end">
                        <button type="submit" 
                            class="w-full px-4 py-2 bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white rounded-lg transition-all duration-300">
                            검색
                        </button>
                    </div>
                    
                    <div class="flex items-end">
                        <button type="button" onclick="resetForm()" 
                            class="w-full px-4 py-2 bg-gray-600 hover:bg-gray-500 text-white rounded-lg transition-all duration-300">
                            초기화
                        </button>
                    </div>
                </form>
            </div>
            
            <div class="overflow-x-auto">
                <table class="w-full text-sm">
                    <thead class="bg-gray-700/50">
                        <tr>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" onclick="sortTable('id')">
                                ID 
                                <span class="sort-indicator ml-1" data-column="id">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" onclick="sortTable('depositor_name')">
                                입금자
                                <span class="sort-indicator ml-1" data-column="depositor_name">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-right text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" onclick="sortTable('amount')">
                                요청금액
                                <span class="sort-indicator ml-1" data-column="amount">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-center text-xs font-medium text-gray-400 uppercase tracking-wider">수수료 상세</th>
                            <th class="px-4 py-4 text-right text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" onclick="sortTable('actual_amount')">
                                처리금액
                                <span class="sort-indicator ml-1" data-column="actual_amount">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-right text-xs font-medium text-gray-400 uppercase tracking-wider">잔액</th>
                            <?php if ($isAdmin): ?>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" onclick="sortTable('store_name')">
                                업체명
                                <span class="sort-indicator ml-1" data-column="store_name">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider">업체ID</th>
                            <?php endif; ?>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" onclick="sortTable('status')">
                                상태
                                <span class="sort-indicator ml-1" data-column="status">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" onclick="sortTable('created_at')">
                                신청일시
                                <span class="sort-indicator ml-1" data-column="created_at">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" onclick="sortTable('updated_at')">
                                처리일시
                                <span class="sort-indicator ml-1" data-column="updated_at">⇅</span>
                            </th>
                            <?php if ($isAdmin): ?>
                            <th class="px-4 py-4 text-center text-xs font-medium text-gray-400 uppercase tracking-wider">작업</th>
                            <?php endif; ?>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-700/30" id="depositsTableBody">
                        <!-- 데이터가 여기에 동적으로 로드됩니다 -->
                    </tbody>
                </table>
            </div>
            
            <!-- 로딩 표시 -->
            <div id="loadingIndicator" class="hidden py-12 text-center">
                <div class="inline-flex items-center px-4 py-2 font-semibold leading-6 text-sm text-blue-400">
                    <svg class="animate-spin -ml-1 mr-3 h-5 w-5 text-blue-400" fill="none" viewBox="0 0 24 24">
                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l-3-2.647z"></path>
                    </svg>
                    데이터를 불러오는 중...
                </div>
            </div>
            
            <!-- 페이지네이션 -->
            <div id="pagination" class="p-6 border-t border-gray-700/30">
                <!-- 페이지네이션은 JavaScript로 동적 생성 -->
            </div>
        </div>
    </div>
</main>

<script>
// ============================================
// deposit.php <script> 섹션 - 완전한 통합 버전
// ============================================

// 전역 변수
const isAdmin = <?php echo json_encode($isAdmin); ?>;
let autoRefreshInterval = null;
let autoRefreshEnabled = true;
let countdownIntervals = new Map();
let lastFetchTime = null;
let globalAutoApprovalWaitTime = 60;

// ============================================
// 페이지 로드 시 초기화
// ============================================
document.addEventListener('DOMContentLoaded', function() {
    const today = new Date().toISOString().split('T')[0];
    const yesterday = new Date();
    yesterday.setDate(yesterday.getDate() - 1);
    const yesterdayStr = yesterday.toISOString().split('T')[0];
    
    document.getElementById('start_date').value = '<?php echo $yesterday; ?>';
    document.getElementById('end_date').value = '<?php echo $today; ?>';
    
    const savedAutoRefresh = localStorage.getItem('autoRefreshEnabled');
    if (savedAutoRefresh !== null) {
        autoRefreshEnabled = savedAutoRefresh === 'true';
    }
    
    updateAutoRefreshUI();
    if (autoRefreshEnabled) {
        startAutoRefresh();
    }
    
    fetchDeposits();
    
    document.getElementById('searchForm').addEventListener('submit', function(e) {
        e.preventDefault();
        fetchDeposits();
    });
    
    const editAmountInput = document.getElementById('editAmount');
    if (editAmountInput) {
        editAmountInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                saveAmount();
            }
        });
    }
    
    document.getElementById('amountEditModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeAmountModal();
        }
    });
});

// ============================================
// 자동승인 취소 함수 (전역)
// ============================================
function cancelAutoApproval(id) {
    if (!confirm('이 거래의 자동승인을 취소하시겠습니까?\n⚠️ 입금 내역이 완전히 삭제됩니다.')) {
        return;
    }

    const button = event.target.closest('button');
    const originalHTML = button.innerHTML;
    button.disabled = true;
    button.innerHTML = `
        <svg class="w-3 h-3 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
        </svg>
        처리중...
    `;

    fetch('/ajax/cancel_auto_approval.php', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({id: id})
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('success', '자동승인이 취소되고 내역이 삭제되었습니다.');
            fetchDeposits();
        } else {
            showAlert('error', data.message || '자동승인 취소 중 오류가 발생했습니다.');
            button.disabled = false;
            button.innerHTML = originalHTML;
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('error', '처리 중 오류가 발생했습니다.');
        button.disabled = false;
        button.innerHTML = originalHTML;
    });
}

// ============================================
// 카운트다운 시작 함수
// ============================================
function startAutoApprovalCountdowns() {
    // 기존 모든 interval 정리
    countdownIntervals.forEach((interval, id) => {
        clearInterval(interval);
    });
    countdownIntervals.clear();
    
    const countdownElements = document.querySelectorAll('[data-countdown]');
    
    console.log(`카운트다운 시작: ${countdownElements.length}개 항목, 설정 대기시간: ${globalAutoApprovalWaitTime}초`);
    
    countdownElements.forEach(element => {
        const transferId = element.dataset.transferId;
        let remainingSeconds = parseInt(element.dataset.countdown);
        
        // 이미 만료된 경우
        if (remainingSeconds <= 0) {
            const textElement = element.querySelector('.countdown-text');
            if (textElement) {
                textElement.textContent = '승인 처리중...';
            }
            return;
        }
        
        // 초기 표시
        updateCountdownDisplay(element, remainingSeconds);
        
        // 1초마다 업데이트
        const interval = setInterval(() => {
            remainingSeconds--;
            element.dataset.countdown = remainingSeconds;
            
            if (remainingSeconds <= 0) {
                clearInterval(interval);
                countdownIntervals.delete(transferId);
                
                const textElement = element.querySelector('.countdown-text');
                if (textElement) {
                    textElement.textContent = '승인 처리중...';
                }
                
                // 3초 후 자동으로 새로고침
                setTimeout(() => {
                    console.log(`Transfer ID ${transferId} 자동승인 완료 예상 - 새로고침`);
                    fetchDeposits();
                }, 3000);
            } else {
                updateCountdownDisplay(element, remainingSeconds);
            }
        }, 1000);
        
        countdownIntervals.set(transferId, interval);
    });
}

// ============================================
// 카운트다운 표시 업데이트
// ============================================
function updateCountdownDisplay(element, seconds) {
    const textElement = element.querySelector('.countdown-text');
    if (!textElement) return;
    
    textElement.textContent = formatCountdown(seconds);
    
    // 설정된 대기시간의 50% 기준으로 긴급 상태 판단
    const urgentThreshold = Math.floor(globalAutoApprovalWaitTime / 2);
    
    // 긴급 상태 클래스 토글
    if (seconds <= urgentThreshold) {
        element.classList.remove('from-emerald-500', 'to-emerald-600');
        element.classList.add('from-orange-500', 'to-red-500', 'urgent');
    } else {
        element.classList.remove('from-orange-500', 'to-red-500', 'urgent');
        element.classList.add('from-emerald-500', 'to-emerald-600');
    }
    
    // 10초 미만일 때 pulse 애니메이션
    if (seconds <= 10) {
        element.classList.add('animate-pulse');
    } else {
        element.classList.remove('animate-pulse');
    }
}

// ============================================
// 카운트다운 포맷 함수
// ============================================
function formatCountdown(seconds) {
    if (seconds <= 0) return '곧 승인';
    
    if (seconds < 60) {
        return `${seconds}초 후`;
    }
    
    const minutes = Math.floor(seconds / 60);
    const remainingSeconds = seconds % 60;
    
    if (remainingSeconds === 0) {
        return `${minutes}분 후`;
    }
    
    return `${minutes}분 ${remainingSeconds}초`;
}

// ============================================
// 엑셀 다운로드
// ============================================
function downloadExcel() {
    const formData = new FormData(document.getElementById('searchForm'));
    const params = new URLSearchParams();
    
    for (let [key, value] of formData.entries()) {
        if (value) params.append(key, value);
    }
    
    params.append('export', 'excel');
    showAlert('info', '엑셀 파일을 생성 중입니다...');
    
    const downloadUrl = `/ajax/export_deposits_excel.php?${params.toString()}`;
    
    const iframe = document.createElement('iframe');
    iframe.style.display = 'none';
    iframe.src = downloadUrl;
    document.body.appendChild(iframe);
    
    iframe.onload = function() {
        setTimeout(() => {
            document.body.removeChild(iframe);
            showAlert('success', '엑셀 파일 다운로드가 시작되었습니다.');
        }, 1000);
    };
    
    iframe.onerror = function() {
        document.body.removeChild(iframe);
        showAlert('error', '엑셀 다운로드 중 오류가 발생했습니다.');
    };
}

// ============================================
// 자동 새로고침 관련
// ============================================
function toggleAutoRefresh() {
    autoRefreshEnabled = !autoRefreshEnabled;
    localStorage.setItem('autoRefreshEnabled', autoRefreshEnabled);
    
    if (autoRefreshEnabled) {
        startAutoRefresh();
        showAlert('success', '자동 새로고침이 활성화되었습니다.');
    } else {
        stopAutoRefresh();
        showAlert('info', '자동 새로고침이 비활성화되었습니다.');
    }
    
    updateAutoRefreshUI();
}

function updateAutoRefreshUI() {
    const button = document.getElementById('autoRefreshToggle');
    const icon = document.getElementById('autoRefreshIcon');
    const text = document.getElementById('autoRefreshText');
    
    if (autoRefreshEnabled) {
        button.classList.remove('bg-gray-700', 'hover:bg-gray-600');
        button.classList.add('bg-green-600', 'hover:bg-green-700');
        icon.classList.add('animate-spin');
        text.textContent = '자동 새로고침 ON';
    } else {
        button.classList.remove('bg-green-600', 'hover:bg-green-700');
        button.classList.add('bg-gray-700', 'hover:bg-gray-600');
        icon.classList.remove('animate-spin');
        text.textContent = '자동 새로고침 OFF';
    }
}

function startAutoRefresh() {
    if (autoRefreshInterval) {
        clearInterval(autoRefreshInterval);
    }
    
    autoRefreshInterval = setInterval(() => {
        const countdownElements = document.querySelectorAll('[data-countdown]');
        
        if (countdownElements.length > 0) {
            const hasUrgent = Array.from(countdownElements).some(el => {
                const remainingSeconds = parseInt(el.dataset.countdown);
                return remainingSeconds > 0 && remainingSeconds <= 5;
            });
            
            if (hasUrgent) {
                console.log('긴급 자동승인 대기중 (5초 이하) - 새로고침 스킵');
                return;
            }
        }
        
        fetchDeposits();
    }, 30000);
}

function stopAutoRefresh() {
    if (autoRefreshInterval) {
        clearInterval(autoRefreshInterval);
        autoRefreshInterval = null;
    }
}

// ============================================
// 테이블 관련
// ============================================
function sortTable(column) {
    const sortByInput = document.getElementById('sort_by');
    const sortOrderInput = document.getElementById('sort_order');
    
    if (sortByInput.value === column) {
        sortOrderInput.value = sortOrderInput.value === 'asc' ? 'desc' : 'asc';
    } else {
        sortByInput.value = column;
        sortOrderInput.value = 'desc';
    }
    
    fetchDeposits();
}

function resetForm() {
    const today = new Date().toISOString().split('T')[0];
    const yesterday = new Date();
    yesterday.setDate(yesterday.getDate() - 1);
    const yesterdayStr = yesterday.toISOString().split('T')[0];

    document.getElementById('searchForm').reset();
    document.getElementById('start_date').value = yesterdayStr;
    document.getElementById('end_date').value = today;
    document.getElementById('sort_by').value = 'created_at';
    document.getElementById('sort_order').value = 'desc';

    fetchDeposits();
}

// ============================================
// 유틸리티
// ============================================
function escapeHtml(text) {
    const map = {
        '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, m => map[m]);
}

function formatNumber(num) {
    return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ',');
}

function removeCommas(str) {
    return str.replace(/,/g, '');
}

// ============================================
// 금액 수정 모달
// ============================================
function openAmountModal(id, currentAmount) {
    document.getElementById('editDepositId').value = id;
    const amountInput = document.getElementById('editAmount');
    amountInput.value = formatNumber(currentAmount);
    document.getElementById('amountEditModal').classList.remove('hidden');
    document.body.style.overflow = 'hidden';
    
    setTimeout(() => {
        amountInput.focus();
        amountInput.select();
    }, 100);
}

function closeAmountModal() {
    document.getElementById('amountEditModal').classList.add('hidden');
    document.body.style.overflow = '';
}

function saveAmount() {
    const id = document.getElementById('editDepositId').value;
    const amountWithCommas = document.getElementById('editAmount').value;
    const newAmount = removeCommas(amountWithCommas);
    
    if (!newAmount || newAmount <= 0) {
        showAlert('error', '올바른 금액을 입력해주세요.');
        return;
    }
    
    fetch('/ajax/update_deposit_amount.php', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({id: id, amount: newAmount})
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('success', '금액이 수정되었습니다.');
            closeAmountModal();
            fetchDeposits();
        } else {
            showAlert('error', data.message || '금액 수정 중 오류가 발생했습니다.');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('error', '처리 중 오류가 발생했습니다.');
    });
}

// ============================================
// 수수료 표시
// ============================================
function createFeeDisplay(d) {
    if (!d.has_transaction) {
        return '<span class="text-gray-500">-</span>';
    }

    let feeHtml = '';
    
    feeHtml += `<div class="text-yellow-400 text-sm font-medium">`;
    feeHtml += `총: ${Number(d.total_fee_amount || 0).toLocaleString()}원`;
    feeHtml += ` <span class="text-xs text-gray-400">(${d.total_fee_rate || 0}%)</span>`;
    feeHtml += `</div>`;
    
    feeHtml += `<div class="text-green-400 text-xs">`;
    feeHtml += `회사: ${Number(d.company_fee_amount || 0).toLocaleString()}원`;
    feeHtml += ` <span class="text-gray-400">(${d.company_fee_rate || 0}%)</span>`;
    feeHtml += `</div>`;
    
    if (d.has_rolling && d.rolling_info && d.rolling_info.length > 0) {
        if (d.rolling_levels > 1) {
            feeHtml += `<div class="text-blue-400 text-xs font-medium border-t border-gray-600 pt-1 mt-1">`;
            feeHtml += `총 롤링: ${Number(d.total_rolling_amount).toLocaleString()}원`;
            feeHtml += ` <span class="text-gray-400">(${d.rolling_levels}단계)</span>`;
            feeHtml += `</div>`;
        }
        
        d.rolling_info.forEach((rolling, index) => {
            const stepNum = index + 1;
            feeHtml += `<div class="text-blue-300 text-xs pl-2">`;
            feeHtml += `${stepNum}단계: ${Number(rolling.rolling_amount).toLocaleString()}원`;
            feeHtml += ` <span class="text-gray-400">(${rolling.rolling_rate}%)</span>`;
            feeHtml += `</div>`;
        });
    }
    
    return feeHtml;
}

// ============================================
// 데이터 가져오기
// ============================================
function fetchDeposits(page = null) {
    const formData = new FormData(document.getElementById('searchForm'));
    const params = new URLSearchParams();
    
    for (let [key, value] of formData.entries()) {
        if (key === 'status[]' || key === 'kakao_code[]') {
            params.append(key, value);
        } else if (value) {
            params.append(key, value);
        }
    }
    
    const currentPage = page || new URLSearchParams(window.location.search).get('page') || 1;
    params.append('page', currentPage);
    
    const tbody = document.getElementById('depositsTableBody');
    const loadingIndicator = document.getElementById('loadingIndicator');
    
    tbody.style.display = 'none';
    loadingIndicator.classList.remove('hidden');
    
    lastFetchTime = Date.now();
    
    fetch(`/ajax/fetch_deposits.php?${params.toString()}`)
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            tbody.style.display = '';
            loadingIndicator.classList.add('hidden');
            
            if (data.success && Array.isArray(data.deposits)) {
                // 서버에서 받은 자동승인 대기시간 저장
                if (data.autoApprovalWaitTime) {
                    const newWaitTime = parseInt(data.autoApprovalWaitTime);
                    if (globalAutoApprovalWaitTime !== newWaitTime) {
                        console.log(`자동승인 대기시간 변경: ${globalAutoApprovalWaitTime}초 → ${newWaitTime}초`);
                        globalAutoApprovalWaitTime = newWaitTime;
                    }
                }
                
                renderDeposits(data.deposits);
                updatePagination(data.totalPages, parseInt(currentPage));
                updateDashboard(data.deposits);
                showNotifications(data.deposits);
            } else {
                showErrorMessage('데이터를 불러올 수 없습니다: ' + (data.message || '알 수 없는 오류'));
            }
        })
        .catch(error => {
            console.error('Fetch error:', error);
            tbody.style.display = '';
            loadingIndicator.classList.add('hidden');
            showErrorMessage(`네트워크 오류: ${error.message}`);
        });
}

// ============================================
// 알림 표시
// ============================================
function showNotifications(deposits) {
    const container = document.getElementById('notificationContainer');
    container.innerHTML = '';
    
    const autoApprovalCount = deposits.filter(d => d.is_auto_approval_queued).length;
    const duplicateCount = deposits.filter(d => d.is_duplicate).length;
    
    if (autoApprovalCount > 0) {
        const notice = document.createElement('div');
        notice.className = 'notification-box bg-emerald-500/10 border border-emerald-500/30 rounded-lg p-3 mb-4 flex items-center';
        notice.innerHTML = `
            <svg class="w-5 h-5 text-emerald-400 mr-2 animate-pulse" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
            </svg>
            <span class="text-emerald-300">
                <strong>${autoApprovalCount}건</strong>의 거래가 자동승인 대기 중입니다. 
                (설정 대기시간: ${formatCountdown(globalAutoApprovalWaitTime)})
            </span>
        `;
        container.appendChild(notice);
    }
    
    if (duplicateCount > 0) {
        const notice = document.createElement('div');
        notice.className = 'notification-box bg-orange-500/10 border border-orange-500/30 rounded-lg p-3 mb-4 flex items-center';
        notice.innerHTML = `
            <svg class="w-5 h-5 text-orange-400 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L8.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
            </svg>
            <span class="text-orange-300">
                <strong>${duplicateCount}건</strong>의 중복 가능성이 있는 거래가 발견되었습니다.
            </span>
        `;
        container.appendChild(notice);
    }
}

// ============================================
// 테이블 렌더링 (핵심 부분)
// ============================================
function renderDeposits(deposits) {
    const tbody = document.getElementById('depositsTableBody');
    tbody.innerHTML = '';
    
    deposits.forEach(d => {
        let statusClass = '';
        let rowClass = 'hover:bg-gray-700/20 transition-all duration-200';

        // 자동승인 대기 중인 경우
        if (d.is_auto_approval_queued) {
            rowClass = 'auto-approval-row';
        } else if (d.is_duplicate) {
            rowClass = 'duplicate-row';
        }

        // 상태별 스타일
        switch (d.status) {
            case '승인':
            case 'completed':
                statusClass = 'bg-green-500/20 text-green-400 border border-green-500/30';
                break;
            case '취소':
            case 'cancelled':
                statusClass = 'bg-red-500/20 text-red-400 border border-red-500/30';
                break;
            case '보류':
            case 'pending':
                statusClass = 'bg-orange-500/20 text-orange-400 border border-orange-500/30';
                break;
            case '대기':
            case 'requested':
                statusClass = 'bg-yellow-500/20 text-yellow-400 border border-yellow-500/30';
                if (!d.is_duplicate && !d.is_auto_approval_queued) {
                    rowClass = 'bg-blue-900/20 hover:bg-blue-900/30 transition-all duration-200';
                }
                break;
            default:
                statusClass = 'bg-gray-500/20 text-gray-400 border border-gray-500/30';
        }

        // 상태 버튼
        let statusButtons = '';
        let actionButtons = '';

        if (isAdmin) {
            if (d.is_auto_approval_queued) {
                statusButtons = `
                    <button onclick="cancelAutoApproval(${d.id})" 
                        class="ml-2 px-3 py-1 text-xs bg-orange-600 hover:bg-orange-700 text-white rounded-lg inline-flex items-center gap-1 transition-all duration-300 shadow-lg">
                        <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                        자동승인 취소
                    </button>
                `;
            } else if (d.status === '대기') {
                statusButtons = `
                    <button onclick="processDeposit(${d.id}, 'completed')" 
                        class="ml-2 px-2 py-1 text-xs bg-green-600 hover:bg-green-700 text-white rounded transition-all duration-300">
                        승인
                    </button>
                    <button onclick="processDeposit(${d.id}, 'cancelled')" 
                        class="ml-2 px-2 py-1 text-xs bg-red-600 hover:bg-red-700 text-white rounded transition-all duration-300">
                        취소
                    </button>
                `;
            } else if (d.status === '보류') {
                statusButtons = `
                    <button onclick="processDeposit(${d.id}, 'requested')" 
                        class="ml-2 px-2 py-1 text-xs bg-yellow-600 hover:bg-yellow-700 text-white rounded transition-all duration-300">
                        대기
                    </button>
                    <button onclick="processDeposit(${d.id}, 'completed')" 
                        class="ml-2 px-2 py-1 text-xs bg-green-600 hover:bg-green-700 text-white rounded transition-all duration-300">
                        승인
                    </button>
                    <button onclick="processDeposit(${d.id}, 'cancelled')" 
                        class="ml-2 px-2 py-1 text-xs bg-red-600 hover:bg-red-700 text-white rounded transition-all duration-300">
                        취소
                    </button>
                `;
            } else if (d.status === '승인') {
                actionButtons = `
                    <button onclick="processDeposit(${d.id}, 'cancelled')" 
                        class="px-2 py-1 text-xs bg-red-600 hover:bg-red-700 text-white rounded transition-all duration-300">
                        취소
                    </button>
                `;
            } else if (d.status === '취소') {
                actionButtons = `
                    <button onclick="processDeposit(${d.id}, 'completed')" 
                        class="px-2 py-1 text-xs bg-green-600 hover:bg-green-700 text-white rounded transition-all duration-300">
                        승인
                    </button>
                `;
            }
        }

        const amountDisplay = isAdmin
            ? `<span class="cursor-pointer hover:text-blue-400 font-medium transition-colors duration-200" onclick="openAmountModal(${d.id}, ${d.amount})">${Number(d.amount).toLocaleString()}원</span>`
            : `<span class="font-medium">${Number(d.amount).toLocaleString()}원</span>`;

        const feeDisplay = createFeeDisplay(d);

        const actualAmountDisplay = d.has_transaction && d.actual_amount !== null
            ? `<span class="text-green-400 font-medium">${Number(d.actual_amount).toLocaleString()}원</span>`
            : '<span class="text-gray-500">-</span>';

        const balanceDisplay = d.has_transaction && d.balance_after !== null
            ? `<span class="text-blue-400 font-medium">${Number(d.balance_after).toLocaleString()}원</span>`
            : '<span class="text-gray-500">-</span>';

        const storeColumns = isAdmin
            ? `<td class="px-4 py-4 text-blue-400 font-medium">${escapeHtml(d.store_name || '미지정')}</td>
               <td class="px-4 py-4 text-gray-300">${escapeHtml(d.store_user_id || '-')}</td>`
            : '';

        const actionColumn = isAdmin
            ? `<td class="px-4 py-4 text-center">${actionButtons}</td>`
            : '';

        // 신규 및 자동승인 표시
        const createdTime = new Date(d.created_at);
        const currentTime = new Date();
        const timeDiff = (currentTime - createdTime) / 1000;
        const isNew = timeDiff <= 60;

        let idDisplay = parseInt(d.id);
        let indicators = [];
        
        if (isNew) {
            indicators.push('<span class="indicator-dot indicator-new" title="신규"></span>');
        }
        
        if (d.is_auto_approval_queued) {
            indicators.push('<span class="indicator-dot indicator-auto-approval" title="자동승인 대기"></span>');
        }
        
        if (d.is_duplicate) {
            indicators.push('<span class="indicator-dot indicator-duplicate" title="중복 가능성"></span>');
        }

        if (indicators.length > 0) {
            idDisplay = `<div class="flex items-center gap-2">${idDisplay} <div class="status-indicators">${indicators.join('')}</div></div>`;
        }

        // 입금자명 표시 (★★ 핵심 부분: 카운트다운 배지 생성 ★★)
        let depositorName = `<span class="font-medium text-white">${escapeHtml(d.name)}</span>`;
        let badges = [];
        
        if (d.is_auto_approval_queued) {
            const remainingSeconds = d.seconds_until_auto_approval;
            // 동적으로 긴급 상태 판단
            const urgentThreshold = Math.floor(globalAutoApprovalWaitTime / 2);
            const urgentClass = remainingSeconds <= urgentThreshold ? 'urgent' : '';
            
            badges.push(`
                <span class="countdown-badge bg-gradient-to-r ${urgentClass}" 
                      data-transfer-id="${d.id}"
                      data-countdown="${remainingSeconds}">
                    <svg class="w-3 h-3 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    <span class="countdown-text">${formatCountdown(remainingSeconds)}</span>
                </span>
            `);
        }
        
        if (d.is_duplicate) {
            badges.push('<span class="duplicate-indicator">중복?</span>');
        }

        if (badges.length > 0) {
            depositorName = `
                <div class="flex items-center gap-2">
                    ${depositorName}
                    ${badges.join(' ')}
                </div>
            `;
        }

        // 상태 표시
        let statusDisplay = '';
        if (d.is_auto_approval_queued) {
            statusDisplay = `
                <div class="flex items-center">
                    <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-emerald-500/20 text-emerald-400 border border-emerald-500/30">
                        자동승인 대기
                    </span>
                    ${statusButtons}
                </div>
            `;
        } else {
            let approvalTypeBadge = '';
            if (d.status === '승인' && d.approval_type) {
                if (d.approval_type === 'auto') {
                    approvalTypeBadge = `
                        <span class="inline-flex items-center px-2 py-0.5 rounded text-xs font-medium bg-blue-500/20 text-blue-400 border border-blue-500/30 ml-2">
                            자동
                        </span>
                    `;
                } else if (d.approval_type === 'manual') {
                    approvalTypeBadge = `
                        <span class="inline-flex items-center px-2 py-0.5 rounded text-xs font-medium bg-purple-500/20 text-purple-400 border border-purple-500/30 ml-2">
                            수동
                        </span>
                    `;
                }
            }

            statusDisplay = `
                <div class="flex items-center">
                    <div class="flex items-center">
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium ${statusClass}">
                            ${escapeHtml(d.status)}
                        </span>
                        ${approvalTypeBadge}
                    </div>
                    ${statusButtons}
                </div>
            `;
        }

        const row = `
            <tr class="${rowClass}">
                <td class="px-4 py-4 font-medium">${idDisplay}</td>
                <td class="px-4 py-4">${depositorName}</td>
                <td class="px-4 py-4 text-right">${amountDisplay}</td>
                <td class="px-4 py-4 text-center text-sm">${feeDisplay}</td>
                <td class="px-4 py-4 text-right">${actualAmountDisplay}</td>
                <td class="px-4 py-4 text-right">${balanceDisplay}</td>
                ${storeColumns}
                <td class="px-4 py-4">${statusDisplay}</td>
                <td class="px-4 py-4 text-gray-400 text-xs">${formatDateTime(d.created_at)}</td>
                <td class="px-4 py-4 text-gray-400 text-xs">${formatDateTime(d.updated_at)}</td>
                ${actionColumn}
            </tr>
        `;
        tbody.insertAdjacentHTML('beforeend', row);
    });

    // ★★ 카운트다운 시작 (중요!) ★★
    startAutoApprovalCountdowns();
}

// ============================================
// 대시보드 업데이트
// ============================================
function updateDashboard(deposits) {
    const today = new Date().toDateString();
    
    let todayCount = 0;
    let todayAmount = 0;
    let autoApprovalCount = 0;
    let autoApprovalAmount = 0;
    let pendingCount = 0;
    let pendingAmount = 0;
    let completedCount = 0;
    let completedAmount = 0;
    let totalFees = 0;
    
    let autoCompletedCount = 0;
    let manualCompletedCount = 0;
    
    deposits.forEach(d => {
        const createdDate = new Date(d.created_at).toDateString();
        const isToday = createdDate === today;
        
        if (isToday) {
            todayCount++;
            todayAmount += Number(d.amount || 0);
        }
        
        if (d.is_auto_approval_queued) {
            autoApprovalCount++;
            autoApprovalAmount += Number(d.amount || 0);
        }
        
        if (d.original_status === 'requested' || d.original_status === 'pending') {
            pendingCount++;
            pendingAmount += Number(d.amount || 0);
        }
        
        if (d.original_status === 'completed' && isToday) {
            completedCount++;
            completedAmount += Number(d.actual_amount || 0);
            totalFees += Number(d.total_fee_amount || 0);
            
            if (d.approval_type === 'auto') {
                autoCompletedCount++;
            } else if (d.approval_type === 'manual') {
                manualCompletedCount++;
            }
        }
    });
    
    document.getElementById('todayCount').textContent = todayCount + '건';
    document.getElementById('todayAmount').textContent = formatNumber(todayAmount) + '원';
    document.getElementById('autoApprovalCount').textContent = autoApprovalCount + '건';
    document.getElementById('autoApprovalAmount').textContent = formatNumber(autoApprovalAmount) + '원';
    document.getElementById('pendingCount').textContent = pendingCount + '건';
    document.getElementById('pendingAmount').textContent = formatNumber(pendingAmount) + '원';
    
    let completedText = completedCount + '건';
    if (autoCompletedCount > 0 || manualCompletedCount > 0) {
        completedText += ` (자동:${autoCompletedCount}, 수동:${manualCompletedCount})`;
    }
    document.getElementById('completedCount').textContent = completedText;
    document.getElementById('completedAmount').textContent = formatNumber(completedAmount) + '원';
    document.getElementById('totalFees').textContent = formatNumber(totalFees) + '원';
}

// ============================================
// 페이지네이션
// ============================================
function updatePagination(totalPages, currentPage) {
    const pagination = document.getElementById('pagination');
    pagination.innerHTML = '';

    if (totalPages <= 1) {
        return;
    }

    const formData = new FormData(document.getElementById('searchForm'));
    const searchParams = new URLSearchParams();
    
    for (let [key, value] of formData.entries()) {
        if (value) searchParams.append(key, value);
    }

    let html = '<div class="flex items-center justify-center space-x-2">';
    
    if (currentPage > 1) {
        const prevParams = new URLSearchParams(searchParams);
        prevParams.set('page', currentPage - 1);
        html += `
            <a href="?${prevParams.toString()}" onclick="return handlePageClick(event, ${currentPage - 1})"
                class="flex items-center px-3 py-2 text-sm text-gray-400 hover:text-white hover:bg-gray-700 rounded-lg transition-all duration-300">
                <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                </svg>
                이전
            </a>
        `;
    }
    
    for (let i = Math.max(1, currentPage - 2); i <= Math.min(totalPages, currentPage + 2); i++) {
        const pageParams = new URLSearchParams(searchParams);
        pageParams.set('page', i);
        html += `
            <a href="?${pageParams.toString()}" onclick="return handlePageClick(event, ${i})"
                class="px-3 py-2 text-sm rounded-lg transition-all duration-300 ${
                    i === currentPage 
                        ? 'bg-gradient-to-r from-blue-600 to-purple-600 text-white font-medium' 
                        : 'text-gray-400 hover:text-white hover:bg-gray-700'
                }">
                ${i}
            </a>
        `;
    }
    
    if (currentPage < totalPages) {
        const nextParams = new URLSearchParams(searchParams);
        nextParams.set('page', currentPage + 1);
        html += `
            <a href="?${nextParams.toString()}" onclick="return handlePageClick(event, ${currentPage + 1})"
                class="flex items-center px-3 py-2 text-sm text-gray-400 hover:text-white hover:bg-gray-700 rounded-lg transition-all duration-300">
                다음
                <svg class="w-4 h-4 ml-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            </a>
        `;
    }
    
    html += '</div>';
    pagination.innerHTML = html;
}

function handlePageClick(event, page) {
    event.preventDefault();
    
    const formData = new FormData(document.getElementById('searchForm'));
    const params = new URLSearchParams();
    
    for (let [key, value] of formData.entries()) {
        if (value) params.append(key, value);
    }
    params.set('page', page);
    
    window.history.pushState({page: page}, '', `?${params.toString()}`);
    fetchDeposits(page);
    window.scrollTo({top: 0, behavior: 'smooth'});
    
    return false;
}

// ============================================
// 입금 처리 함수
// ============================================
function processDeposit(id, action) {
    let actionText = '';
    let confirmMessage = '';

    switch (action) {
        case 'requested':
            actionText = '대기 처리';
            confirmMessage = '이 입금 신청을 대기 상태로 변경하시겠습니까?';
            break;
        case 'completed':
            actionText = '승인 처리';
            confirmMessage = '이 입금 신청을 승인 처리하시겠습니까?';
            break;
        case 'cancelled':
            actionText = '취소';
            confirmMessage = '이 입금 신청을 취소하시겠습니까?';
            break;
        default:
            showAlert('error', '잘못된 액션입니다.');
            return;
    }

    if (!confirm(confirmMessage)) {
        return;
    }

    const button = event.target;
    const originalText = button.textContent;
    button.disabled = true;
    button.textContent = '처리중...';

    fetch('/ajax/process_deposit.php', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({id: id, action: action})
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('success', `입금 신청이 ${actionText}되었습니다.`);
            fetchDeposits();
        } else {
            showAlert('error', data.message || `${actionText} 중 오류가 발생했습니다.`);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('error', '처리 중 오류가 발생했습니다.');
    })
    .finally(() => {
        if (button) {
            button.disabled = false;
            button.textContent = originalText;
        }
    });
}

// ============================================
// 날짜시간 포맷팅
// ============================================
function formatDateTime(dateTimeString) {
    const date = new Date(dateTimeString);
    const dateStr = date.toLocaleDateString('ko-KR', { 
        year: 'numeric', 
        month: '2-digit', 
        day: '2-digit' 
    });
    const timeStr = date.toLocaleTimeString('ko-KR', { 
        hour: '2-digit', 
        minute: '2-digit' 
    });
    
    return `${dateStr} ${timeStr}`;
}

// ============================================
// 알림 표시
// ============================================
function showAlert(type, message) {
    const existingAlert = document.querySelector('.alert-message');
    if (existingAlert) {
        existingAlert.remove();
    }
    
    let alertClass, alertIcon;
    
    switch(type) {
        case 'success':
            alertClass = 'bg-gradient-to-r from-green-600 to-emerald-600 border-green-500';
            alertIcon = '<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path></svg>';
            break;
        case 'info':
            alertClass = 'bg-gradient-to-r from-blue-600 to-cyan-600 border-blue-500';
            alertIcon = '<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
            break;
        default:
            alertClass = 'bg-gradient-to-r from-red-600 to-rose-600 border-red-500';
            alertIcon = '<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
    }
    
    const alert = document.createElement('div');
    alert.className = `alert-message fixed top-4 right-4 ${alertClass} text-white px-6 py-4 rounded-lg shadow-2xl z-[60] border flex items-center animate-slide-in`;
    alert.innerHTML = `
        <div class="flex items-center">
            ${alertIcon}
            <span class="ml-3 font-medium">${message}</span>
        </div>
    `;
    
    document.body.appendChild(alert);
    
    setTimeout(() => {
        alert.classList.add('animate-fade-out');
        setTimeout(() => alert.remove(), 300);
    }, 3000);
}

function showErrorMessage(message) {
    console.error('Error:', message);
    showAlert('error', message);
}

function toggleSidebar() {
    const sidebar = document.querySelector('.sidebar');
    if (sidebar) {
        sidebar.classList.toggle('hidden');
    }
}

// ============================================
// 페이지 언로드 및 포커스 처리
// ============================================
window.addEventListener('beforeunload', function() {
    stopAutoRefresh();
    countdownIntervals.forEach((interval, id) => {
        clearInterval(interval);
    });
    countdownIntervals.clear();
});

document.addEventListener('visibilitychange', function() {
    if (!document.hidden && lastFetchTime) {
        const timeSinceLastFetch = Date.now() - lastFetchTime;
        
        if (timeSinceLastFetch > 60000) {
            console.log('페이지 포커스 복원 - 새로고침');
            fetchDeposits();
        }
    }
});
</script>

</body>
</html>