<?php
// direct_payment.php - URL 노출 없이 직접 결제 페이지 표시
// URL: domain.com/direct_payment.php?store=6789

$store_code = $_GET['store'] ?? '';

// 기본 보안 검사
if (empty($store_code)) {
    showNotFoundPage();
    exit;
}

// DB 연결
include 'inc/db_connect.php';

// DB에서 가맹점 존재 여부 확인
$store_query = "
    SELECT m.*, 
           ka.name as kakao_name, ka.qr_links,
           ba.bank as bank_name, ba.account_holder, ba.account_number
    FROM member m 
    LEFT JOIN kakao_account ka ON m.kakao_account_id = ka.id 
    LEFT JOIN bank_account ba ON m.bank_account_id = ba.id
    WHERE m.code = ? AND m.grade != 'admin'
";

$store_stmt = mysqli_prepare($con, $store_query);
mysqli_stmt_bind_param($store_stmt, 's', $store_code);
mysqli_stmt_execute($store_stmt);
$store = mysqli_fetch_assoc(mysqli_stmt_get_result($store_stmt));

if (!$store) {
    showNotFoundPage();
    exit;
}

// 결제 방식 확인
$payment_type = 'none';
if (!empty($store['qr_links']) && $store['kakao_account_id'] > 0) {
    $payment_type = 'kakao';
} elseif (!empty($store['bank_name']) && $store['bank_account_id'] > 0) {
    $payment_type = 'bank';
}

if ($payment_type === 'none') {
    showNotFoundPage();
    exit;
}

// 접근 성공 로그
error_log("Direct payment access: Store={$store_code}, Type={$payment_type}, IP=" . ($_SERVER['REMOTE_ADDR'] ?? 'unknown'));

// 이제 payment_select.php의 내용을 직접 포함하되, URL을 바꾸지 않고 처리
// 하지만 더 좋은 방법은 payment_select.php의 내용을 직접 여기에 구현하는 것

?>
<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($store['user_name']); ?> - 간편 송금</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-900 text-white min-h-screen">
    <div class="container mx-auto px-4 py-8 max-w-md">
        <!-- 헤더 -->
        <div class="text-center mb-8">
            <div class="w-20 h-20 bg-<?php echo $payment_type === 'kakao' ? 'yellow' : 'blue'; ?>-400 rounded-full flex items-center justify-center mx-auto mb-4">
                <span class="text-3xl"><?php echo $payment_type === 'kakao' ? '💳' : '🏦'; ?></span>
            </div>
            <h1 class="text-2xl font-bold text-<?php echo $payment_type === 'kakao' ? 'yellow' : 'blue'; ?>-400 mb-2">
                <?php echo htmlspecialchars($store['user_name']); ?>
            </h1>
            <p class="text-gray-300">
                <?php echo $payment_type === 'kakao' ? '카카오페이' : '은행'; ?> 송금할 금액을 선택해주세요
            </p>
            <?php if ($payment_type === 'kakao' && $store['kakao_name']): ?>
            <p class="text-sm text-gray-400 mt-1">
                카카오페이: <?php echo htmlspecialchars($store['kakao_name']); ?>
            </p>
            <?php elseif ($payment_type === 'bank'): ?>
            <p class="text-sm text-gray-400 mt-1">
                <?php echo htmlspecialchars($store['bank_name']); ?> | <?php echo htmlspecialchars($store['account_holder']); ?>
            </p>
            <?php endif; ?>
        </div>

        <!-- 입금자명 입력 -->
        <div class="bg-gray-800 rounded-xl p-4 mb-6">
            <h3 class="text-lg font-semibold mb-3 text-<?php echo $payment_type === 'kakao' ? 'yellow' : 'blue'; ?>-400">입금자 정보</h3>
            <div class="space-y-3">
                <div>
                    <label class="block text-sm font-medium text-gray-300 mb-2">입금자명 *</label>
                    <input type="text" 
                           id="depositorName" 
                           placeholder="입금하시는 분의 성함을 입력하세요" 
                           class="w-full px-4 py-3 bg-gray-700 text-white rounded-lg border border-gray-600 focus:outline-none focus:ring-2 focus:ring-<?php echo $payment_type === 'kakao' ? 'yellow' : 'blue'; ?>-400"
                           maxlength="20"
                           required>
                    <p class="text-xs text-gray-400 mt-1">* 필수 입력 항목입니다</p>
                </div>
            </div>
        </div>

        <!-- 금액 선택 버튼들 -->
        <div class="bg-gray-800 rounded-xl p-4 mb-6">
            <h3 class="text-lg font-semibold mb-4 text-green-400">송금 금액 선택</h3>
            <div class="grid grid-cols-2 gap-3 mb-4">
                <button onclick="selectAmount(10000)" 
                        class="bg-gray-700 hover:bg-green-600 border border-gray-600 hover:border-green-400 rounded-lg p-3 transition">
                    <div class="text-center">
                        <div class="text-lg font-semibold">1만원</div>
                    </div>
                </button>
                <button onclick="selectAmount(30000)" 
                        class="bg-gray-700 hover:bg-green-600 border border-gray-600 hover:border-green-400 rounded-lg p-3 transition">
                    <div class="text-center">
                        <div class="text-lg font-semibold">3만원</div>
                    </div>
                </button>
                <button onclick="selectAmount(50000)" 
                        class="bg-gray-700 hover:bg-green-600 border border-gray-600 hover:border-green-400 rounded-lg p-3 transition">
                    <div class="text-center">
                        <div class="text-lg font-semibold">5만원</div>
                    </div>
                </button>
                <button onclick="selectAmount(100000)" 
                        class="bg-gray-700 hover:bg-green-600 border border-gray-600 hover:border-green-400 rounded-lg p-3 transition">
                    <div class="text-center">
                        <div class="text-lg font-semibold">10만원</div>
                    </div>
                </button>
                <button onclick="selectAmount(500000)" 
                        class="bg-gray-700 hover:bg-green-600 border border-gray-600 hover:border-green-400 rounded-lg p-3 transition">
                    <div class="text-center">
                        <div class="text-lg font-semibold">50만원</div>
                    </div>
                </button>
                <button onclick="selectAmount(1000000)" 
                        class="bg-gray-700 hover:bg-green-600 border border-gray-600 hover:border-green-400 rounded-lg p-3 transition">
                    <div class="text-center">
                        <div class="text-lg font-semibold">100만원</div>
                    </div>
                </button>
            </div>
            
            <!-- 직접 입력 -->
            <div class="border-t border-gray-600 pt-4">
                <h4 class="text-md font-medium mb-3 text-gray-300">직접 입력</h4>
                <div class="flex space-x-2">
                    <input type="number" 
                           id="customAmount" 
                           placeholder="금액을 입력하세요" 
                           class="flex-1 px-4 py-3 bg-gray-700 text-white rounded-lg border border-gray-600 focus:outline-none focus:ring-2 focus:ring-green-400"
                           min="1000"
                           step="1000">
                    <button onclick="selectCustomAmount()" 
                            class="px-6 py-3 bg-green-400 text-gray-900 font-semibold rounded-lg hover:bg-green-500 transition">
                        선택
                    </button>
                </div>
                <p class="text-xs text-gray-400 mt-2">* 최소 1,000원부터 입력 가능합니다</p>
            </div>
        </div>

        <!-- 안내 메시지 -->
        <div class="bg-blue-900/20 border border-blue-500/30 rounded-lg p-4">
            <div class="flex items-start space-x-3">
                <span class="text-blue-400 text-xl">ℹ️</span>
                <div>
                    <h4 class="font-semibold text-blue-400 mb-1">이용 안내</h4>
                    <ul class="text-sm text-blue-200 space-y-1">
                        <li>• 입금자명과 금액을 정확히 입력하세요</li>
                        <li>• 금액 선택 후 <?php echo $payment_type === 'kakao' ? '카카오페이' : '은행 계좌'; ?>로 연결됩니다</li>
                        <li>• 송금 완료 후 자동으로 처리됩니다</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>

    <script>
        // 현재 가맹점 정보
        const storeData = {
            code: '<?php echo $store_code; ?>',
            name: '<?php echo htmlspecialchars($store['user_name']); ?>',
            type: '<?php echo $payment_type; ?>',
            kakao_tid: '<?php echo $store['kakao_tid'] ?? ''; ?>',
            kakao_account_id: <?php echo $store['kakao_account_id'] ?? 0; ?>,
            bank_account_id: <?php echo $store['bank_account_id'] ?? 0; ?>,
            has_qr_links: <?php echo !empty($store['qr_links']) ? 'true' : 'false'; ?>
        };

        // 금액 선택 함수
        function selectAmount(amount) {
            const depositorName = document.getElementById('depositorName').value.trim();
            
            if (!depositorName) {
                alert('입금자명을 입력해주세요.');
                document.getElementById('depositorName').focus();
                return;
            }
            
            if (depositorName.length < 2) {
                alert('입금자명은 2글자 이상 입력해주세요.');
                document.getElementById('depositorName').focus();
                return;
            }
            
            // 결제 방식에 따라 분기
            if (storeData.type === 'kakao') {
                proceedToKakaoPay(amount, depositorName);
            } else {
                proceedToBankTransfer(amount, depositorName);
            }
        }

        // 직접 입력 금액 선택
        function selectCustomAmount() {
            const customAmount = document.getElementById('customAmount').value;
            const amount = parseInt(customAmount);
            
            if (!amount || amount < 1000) {
                alert('1,000원 이상의 금액을 입력해주세요.');
                return;
            }
            
            if (amount > 10000000) {
                alert('1,000만원 이하의 금액을 입력해주세요.');
                return;
            }
            
            selectAmount(amount);
        }

        // 카카오페이 송금 처리
        function proceedToKakaoPay(amount, depositorName) {
            if (confirm(`가맹점: ${storeData.name}\n입금자: ${depositorName}\n금액: ${amount.toLocaleString()}원\n\n카카오페이로 송금하시겠습니까?`)) {
                // 로딩 표시
                document.body.innerHTML = `
                    <div class="fixed inset-0 bg-gray-900 flex items-center justify-center">
                        <div class="text-center">
                            <div class="w-16 h-16 border-4 border-yellow-400 border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
                            <p class="text-white text-lg">카카오페이로 연결 중...</p>
                            <p class="text-gray-400 text-sm mt-2">가맹점: ${storeData.name}</p>
                            <p class="text-gray-400 text-sm">입금자: ${depositorName}</p>
                            <p class="text-gray-400 text-sm">금액: ${amount.toLocaleString()}원</p>
                        </div>
                    </div>
                `;
                
                // 카카오페이 송금 요청 생성
                const requestData = {
                    store_id: storeData.code,
                    amount: amount,
                    kakao_tid: storeData.kakao_tid,
                    depositor_name: depositorName
                };
                
                fetch('create_payment_request.php', {
                    method: 'POST',
                    headers: { 
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify(requestData)
                })
                .then(response => response.text())
                .then(text => {
                    try {
                        const data = JSON.parse(text);
                        if (data.success) {
                            window.location.href = data.payment_url;
                        } else {
                            alert('오류: ' + data.message);
                            location.reload();
                        }
                    } catch (e) {
                        alert('서버 응답 형식 오류: ' + text.substring(0, 100));
                        location.reload();
                    }
                })
                .catch(error => {
                    alert('연결 중 오류가 발생했습니다: ' + error.message);
                    location.reload();
                });
            }
        }

        // 은행 송금 처리
        function proceedToBankTransfer(amount, depositorName) {
            if (confirm(`가맹점: ${storeData.name}\n입금자: ${depositorName}\n금액: ${amount.toLocaleString()}원\n\n은행 송금으로 진행하시겠습니까?`)) {
                // 은행 송금 페이지로 이동 (기존 방식 유지)
                const bankTransferUrl = `bank_transfer.php?store=${storeData.code}&amount=${amount}&depositor=${encodeURIComponent(depositorName)}`;
                window.location.href = bankTransferUrl;
            }
        }

        // Enter 키 이벤트
        document.getElementById('customAmount').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                selectCustomAmount();
            }
        });
        
        document.getElementById('depositorName').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                // 마지막으로 선택된 금액이 있으면 진행
                const lastAmount = sessionStorage.getItem('lastSelectedAmount');
                if (lastAmount) {
                    selectAmount(parseInt(lastAmount));
                }
            }
        });

        // 숫자만 입력 허용
        document.getElementById('customAmount').addEventListener('input', function(e) {
            this.value = this.value.replace(/[^0-9]/g, '');
        });
    </script>
</body>
</html>

<?php
function showNotFoundPage() {
    http_response_code(404);
    ?>
    <!DOCTYPE html>
    <html lang="ko">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Page Not Found</title>
        <style>
            body { font-family: Arial, sans-serif; text-align: center; padding: 50px; background: #f8f9fa; }
            .error-container { max-width: 500px; margin: 0 auto; background: white; padding: 40px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
            h1 { color: #dc3545; margin-bottom: 20px; }
            p { color: #6c757d; line-height: 1.6; }
            .back-btn { background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block; margin-top: 20px; }
        </style>
    </head>
    <body>
        <div class="error-container">
            <h1>404 - Page Not Found</h1>
            <p>요청하신 페이지를 찾을 수 없습니다.</p>
            <p>링크가 올바른지 확인해주세요.</p>
            <a href="/" class="back-btn">홈으로 돌아가기</a>
        </div>
    </body>
    </html>
    <?php
}
?>