<?php
// get_remaining_time.php - 거래의 남은 시간 계산
header('Content-Type: application/json; charset=utf-8');

// 타임존 설정 추가 (한국 시간)
date_default_timezone_set('Asia/Seoul');
try {
    include 'inc/db_connect.php';
    
    // POST 요청 확인
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('POST 요청만 허용됩니다.');
    }
    
    // JSON 데이터 읽기
    $json_input = file_get_contents('php://input');
    error_log("Get remaining time input: " . $json_input);
    
    if (empty($json_input)) {
        throw new Exception('요청 데이터가 비어있습니다.');
    }
    
    $input = json_decode($json_input, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('JSON 파싱 에러: ' . json_last_error_msg());
    }
    
    $transactionId = $input['transaction_id'] ?? '';
    $sessionId = $input['session_id'] ?? '';
    $checkExisting = $input['check_existing'] ?? false;
    
    // 기존 거래 확인 모드
    if ($checkExisting) {
        if (!$sessionId) {
            throw new Exception('세션 ID가 필요합니다.');
        }
        
        // 같은 세션 ID의 최근 거래 찾기
        $query = "SELECT t.kakao_tid, t.created_at, t.status 
                  FROM transfer t
                  WHERE t.session_id = ? 
                  AND t.status IN ('pending', 'requested')
                  AND t.created_at > DATE_SUB(NOW(), INTERVAL 10 MINUTE)
                  ORDER BY t.created_at DESC 
                  LIMIT 1";
        
        $stmt = mysqli_prepare($con, $query);
        mysqli_stmt_bind_param($stmt, 's', $sessionId);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        $existingTransaction = mysqli_fetch_assoc($result);
        
        if ($existingTransaction) {
            // 기존 거래가 있으면 해당 거래 정보 반환
            $createdAt = new DateTime($existingTransaction['created_at']);
            $currentTime = new DateTime();
            $elapsedSeconds = $currentTime->getTimestamp() - $createdAt->getTimestamp();
            $remainingSeconds = 600 - $elapsedSeconds;
            
            echo json_encode([
                'success' => true,
                'existing_transaction' => $existingTransaction,
                'remaining_seconds' => max(0, $remainingSeconds),
                'is_expired' => $remainingSeconds <= 0
            ]);
            exit;
        } else {
            // 기존 거래가 없음
            echo json_encode([
                'success' => true,
                'existing_transaction' => null
            ]);
            exit;
        }
    }
    
    // 일반 남은 시간 확인 모드
    if (!$transactionId) {
        throw new Exception('거래 ID가 필요합니다.');
    }
    
    // 거래 정보 조회
    $query = "SELECT created_at, status FROM transfer WHERE kakao_tid = ?";
    $stmt = mysqli_prepare($con, $query);
    mysqli_stmt_bind_param($stmt, 's', $transactionId);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $transfer = mysqli_fetch_assoc($result);
    
    if (!$transfer) {
        throw new Exception('해당 거래를 찾을 수 없습니다.');
    }
    
    // 거래가 이미 완료되었거나 취소된 경우
    if (in_array($transfer['status'], ['completed', 'cancelled'])) {
        echo json_encode([
            'success' => true,
            'remaining_seconds' => 0,
            'is_expired' => true,
            'status' => $transfer['status']
        ]);
        exit;
    }
    
    // 생성 시간과 현재 시간 비교
    $createdAt = new DateTime($transfer['created_at']);
    $currentTime = new DateTime();
    $elapsedSeconds = $currentTime->getTimestamp() - $createdAt->getTimestamp();
    
    // 10분 = 600초
    $timeLimit = 600;
    $remainingSeconds = $timeLimit - $elapsedSeconds;
    
    // 시간이 만료된 경우
    if ($remainingSeconds <= 0) {
        echo json_encode([
            'success' => true,
            'remaining_seconds' => 0,
            'is_expired' => true,
            'status' => $transfer['status']
        ]);
    } else {
        echo json_encode([
            'success' => true,
            'remaining_seconds' => $remainingSeconds,
            'is_expired' => false,
            'status' => $transfer['status'],
            'created_at' => $transfer['created_at'],
            'elapsed_seconds' => $elapsedSeconds
        ]);
    }
    
    error_log("남은 시간 계산 완료: Transaction ID={$transactionId}, Remaining={$remainingSeconds}초");
    
} catch (Exception $e) {
    error_log("남은 시간 계산 오류: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>