<?php 
// 데이터베이스 연결
date_default_timezone_set('Asia/Seoul');

$con = mysqli_connect("localhost", "test_pay", "tp1324!#@$", "test_pay");

// 연결 확인
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// UTF-8 설정
mysqli_set_charset($con, "utf8mb4");

/**
 * 실제 IP 주소 가져오기 (프록시, 로드밸런서 고려)
 * 
 * @return string IP 주소
 */
function getRealIpAddress() {
    // 프록시를 통한 접속인지 확인
    if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) {
        // Cloudflare
        return $_SERVER['HTTP_CF_CONNECTING_IP'];
    } elseif (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        // 공유 인터넷을 통한 접속
        return $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        // 프록시를 통한 접속
        $forwardedIps = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
        return trim($forwardedIps[0]);
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED'])) {
        return $_SERVER['HTTP_X_FORWARDED'];
    } elseif (!empty($_SERVER['HTTP_X_CLUSTER_CLIENT_IP'])) {
        return $_SERVER['HTTP_X_CLUSTER_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_FORWARDED_FOR'])) {
        return $_SERVER['HTTP_FORWARDED_FOR'];
    } elseif (!empty($_SERVER['HTTP_FORWARDED'])) {
        return $_SERVER['HTTP_FORWARDED'];
    } else {
        // 일반적인 경우
        return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    }
}



/**
 * all_log 테이블에 로그 기록 (IP 정보 포함)
 * 
 * @param mysqli $con 데이터베이스 연결
 * @param string $type 로그 타입
 * @param mixed $oldData 이전 데이터 (배열 또는 문자열)
 * @param mixed $newData 새로운 데이터 (배열 또는 문자열)
 * @param int $status 상태 (1: 성공, 0: 실패)
 * @param string $detail 상세 설명
 * @param int $userId 사용자 ID (기본값: 세션에서 가져옴)
 * @param string $ipAddress IP 주소 (기본값: 자동 감지)
 * @return bool 성공 여부
 */
function writeLog($con, $type, $oldData = null, $newData = null, $status = 1, $detail = '', $userId = null, $ipAddress = null) {
    // 사용자 ID 설정
    if ($userId === null) {
        $userId = isset($_SESSION['id']) ? $_SESSION['id'] : 0;
    }
    
    // IP 주소 설정
    if ($ipAddress === null) {
        $ipAddress = getRealIpAddress();
    }
    
    // 데이터를 JSON으로 변환
    if (is_array($oldData) || is_object($oldData)) {
        $oldData = json_encode($oldData, JSON_UNESCAPED_UNICODE);
    }
    if (is_array($newData) || is_object($newData)) {
        $newData = json_encode($newData, JSON_UNESCAPED_UNICODE);
    }
    
    // 로그 삽입 쿼리 (IP 정보 포함)
    $query = "INSERT INTO all_log (user_id, ip_address, type, old_data, new_data, status, detail, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, NOW())";
    
    $stmt = mysqli_prepare($con, $query);
    
    if (!$stmt) {
        error_log("Log prepare failed: " . mysqli_error($con));
        return false;
    }
    
    mysqli_stmt_bind_param($stmt, "issssis", $userId, $ipAddress, $type, $oldData, $newData, $status, $detail);
    
    $result = mysqli_stmt_execute($stmt);
    
    if (!$result) {
        error_log("Log insert failed: " . mysqli_stmt_error($stmt));
    }
    
    mysqli_stmt_close($stmt);
    
    return $result;
}

/**
 * 성공 로그 기록 (간편 함수)
 * 
 * @param mysqli $con 데이터베이스 연결
 * @param string $type 로그 타입
 * @param string $detail 상세 설명
 * @param mixed $oldData 이전 데이터
 * @param mixed $newData 새로운 데이터
 * @return bool 성공 여부
 */
function logSuccess($con, $type, $detail, $oldData = null, $newData = null) {
    return writeLog($con, $type, $oldData, $newData, 1, $detail);
}

/**
 * 실패 로그 기록 (간편 함수)
 * 
 * @param mysqli $con 데이터베이스 연결
 * @param string $type 로그 타입
 * @param string $detail 상세 설명
 * @param mixed $oldData 이전 데이터
 * @param mixed $newData 새로운 데이터
 * @return bool 성공 여부
 */
function logError($con, $type, $detail, $oldData = null, $newData = null) {
    return writeLog($con, $type, $oldData, $newData, 0, $detail);
}

/**
 * 특정 IP에서 로그 기록 (IP 정보 명시적 전달)
 * 
 * @param mysqli $con 데이터베이스 연결
 * @param string $type 로그 타입
 * @param string $detail 상세 설명
 * @param string $ipAddress 특정 IP 주소
 * @param mixed $oldData 이전 데이터
 * @param mixed $newData 새로운 데이터
 * @param int $status 상태
 * @return bool 성공 여부
 */
function logWithIp($con, $type, $detail, $ipAddress, $oldData = null, $newData = null, $status = 1) {
    return writeLog($con, $type, $oldData, $newData, $status, $detail, null, $ipAddress);
}

/**
 * 관리자 액션 로그 (IP 추적 강화)
 * 
 * @param mysqli $con 데이터베이스 연결
 * @param string $action 관리자 액션
 * @param string $detail 상세 설명
 * @param mixed $oldData 이전 데이터
 * @param mixed $newData 새로운 데이터
 * @param int $adminId 관리자 ID
 * @return bool 성공 여부
 */
function logAdminAction($con, $action, $detail, $oldData = null, $newData = null, $adminId = null) {
    $ipAddress = getRealIpAddress();
    
    // 관리자 액션은 특별히 상세하게 기록
    $enhancedDetail = "[ADMIN_ACTION] {$detail} | IP: {$ipAddress}";
    
    return writeLog($con, "ADMIN_{$action}", $oldData, $newData, 1, $enhancedDetail, $adminId, $ipAddress);
}

/**
 * 트랜잭션 시작
 */
function beginTransaction($con) {
    mysqli_autocommit($con, false);
}

/**
 * 트랜잭션 커밋
 */
function commitTransaction($con) {
    mysqli_commit($con);
    mysqli_autocommit($con, true);
}

/**
 * 트랜잭션 롤백
 */
function rollbackTransaction($con) {
    mysqli_rollback($con);
    mysqli_autocommit($con, true);
}

/**
 * 안전한 문자열 이스케이프
 */
function escapeString($con, $str) {
    return mysqli_real_escape_string($con, $str);
}

/**
 * 페이징 계산 헬퍼 함수
 * 
 * @param int $totalCount 전체 개수
 * @param int $perPage 페이지당 개수
 * @param int $currentPage 현재 페이지
 * @return array ['total_pages', 'current_page', 'offset', 'limit']
 */
function calculatePagination($totalCount, $perPage = 20, $currentPage = 1) {
    $totalPages = ceil($totalCount / $perPage);
    $currentPage = max(1, min($currentPage, $totalPages));
    $offset = ($currentPage - 1) * $perPage;
    
    return [
        'total_pages' => $totalPages,
        'current_page' => $currentPage,
        'offset' => $offset,
        'limit' => $perPage
    ];
}

/**
 * IP 기반 보안 체크
 * 
 * @param string $ipAddress 체크할 IP
 * @return bool 허용된 IP인지 여부
 */
function isAllowedIp($ipAddress) {
    // 허용된 IP 대역 (예시)
    $allowedIps = [
        '127.0.0.1',           // localhost
        '192.168.1.0/24',      // 내부 네트워크
        '10.0.0.0/8',          // 사설 네트워크
    ];
    
    // 실제 구현에서는 더 정교한 IP 체크 로직 필요
    foreach ($allowedIps as $allowedIp) {
        if (strpos($allowedIp, '/') !== false) {
            // CIDR 표기법 체크
            if (ipInCIDR($ipAddress, $allowedIp)) {
                return true;
            }
        } else {
            // 직접 비교
            if ($ipAddress === $allowedIp) {
                return true;
            }
        }
    }
    
    return false;
}

/**
 * CIDR 체크 헬퍼 함수
 */
function ipInCIDR($ip, $cidr) {
    list($subnet, $mask) = explode('/', $cidr);
    return (ip2long($ip) & ~((1 << (32 - $mask)) - 1)) == ip2long($subnet);
}
?>