<?php
/**
 * 입금 승인 처리 함수
 * 파일 위치: /inc/deposit_functions.php
 * 
 * 사용법:
 * require_once '/inc/deposit_functions.php';
 * $result = processDepositApproval($con, $transferId);
 */

/**
 * 트랜잭션 ID 생성 함수
 */
if (!function_exists('generateTransactionId')) {
    function generateTransactionId() {
        return 'TXN' . date('YmdHis') . sprintf('%06d', mt_rand(0, 999999));
    }
}

/**
 * 입금 승인 처리 함수 (자동 승인용)
 * 
 * @param mysqli $con 데이터베이스 연결 객체
 * @param int $transferId 처리할 transfer ID
 * @param bool $callCallback 콜백 호출 여부 (기본값: true)
 * @return array 처리 결과 배열
 */
function processDepositApproval($con, $transferId, $callCallback = true) {
    // 자동 승인 시스템이므로 처리자는 항상 'system'
    $processedBy = 'system';
    $result = [
        'success' => false,
        'message' => '',
        'data' => null,
        'error' => null
    ];
    
    try {
        // autocommit 비활성화
        mysqli_autocommit($con, FALSE);
        
        // 트랜잭션 시작
        mysqli_begin_transaction($con, MYSQLI_TRANS_START_READ_WRITE);
        
        // 1. transfer 정보 및 사용자 정보 조회 (FOR UPDATE로 잠금)
        $checkQuery = "SELECT t.id, t.user_id, t.amount, t.status, t.type, t.created_at, t.trans_id,
                              m.amount as current_balance, m.user_name, m.user_id as member_user_id, 
                              m.rate, m.parents, m.callback_url
                       FROM transfer t 
                       JOIN member m ON t.user_id = m.id 
                       WHERE t.id = ? AND t.type = 'deposit'
                       FOR UPDATE";
        
        $checkStmt = mysqli_prepare($con, $checkQuery);
        mysqli_stmt_bind_param($checkStmt, "i", $transferId);
        mysqli_stmt_execute($checkStmt);
        $transferResult = mysqli_stmt_get_result($checkStmt);
        
        if (!$transfer = mysqli_fetch_assoc($transferResult)) {
            throw new Exception('입금 신청을 찾을 수 없습니다.');
        }
        mysqli_stmt_close($checkStmt);
        
        // 기본 정보 추출
        $oldStatus = $transfer['status'];
        $memberId = $transfer['user_id'];
        $memberUserId = $transfer['member_user_id'];
        $depositAmount = $transfer['amount'];
        $currentBalance = $transfer['current_balance'];
        $userName = $transfer['user_name'];
        $memberRate = $transfer['rate'] ?? 0;
        $existingTransId = $transfer['trans_id'] ?? null;
        $parentsJson = $transfer['parents'] ?? '[]';
        $callbackUrl = $transfer['callback_url'];
        
        // 상태 체크
        if ($oldStatus === 'completed') {
            throw new Exception('이미 승인 처리된 입금입니다.');
        }
        
        // 중복 처리 방지 체크
        if (!empty($existingTransId)) {
            $historyCheckQuery = "SELECT id FROM transaction_history WHERE trans_id = ? OR transfer_id = ?";
            $historyCheckStmt = mysqli_prepare($con, $historyCheckQuery);
            mysqli_stmt_bind_param($historyCheckStmt, "si", $existingTransId, $transferId);
            mysqli_stmt_execute($historyCheckStmt);
            $historyCheckResult = mysqli_stmt_get_result($historyCheckStmt);
            
            if (mysqli_num_rows($historyCheckResult) > 0) {
                mysqli_stmt_close($historyCheckStmt);
                throw new Exception('이미 처리된 거래입니다. (Transaction ID: ' . $existingTransId . ')');
            }
            mysqli_stmt_close($historyCheckStmt);
        }
        
        // transfer_id로 중복 체크
        $dupCheckQuery = "SELECT id FROM transaction_history WHERE transfer_id = ?";
        $dupCheckStmt = mysqli_prepare($con, $dupCheckQuery);
        mysqli_stmt_bind_param($dupCheckStmt, "i", $transferId);
        mysqli_stmt_execute($dupCheckStmt);
        $dupResult = mysqli_stmt_get_result($dupCheckStmt);
        
        if (mysqli_num_rows($dupResult) > 0) {
            mysqli_stmt_close($dupCheckStmt);
            throw new Exception('이미 처리 내역이 존재하는 거래입니다.');
        }
        mysqli_stmt_close($dupCheckStmt);
        
        // 트랜잭션 ID 생성
        $transId = generateTransactionId();
        
        // 총 수수료 계산
        $totalFeeRate = $memberRate;
        $totalFeeAmount = round($depositAmount * ($totalFeeRate / 100), 2);
        $actualAmount = $depositAmount - $totalFeeAmount;
        
        // member 테이블 잔액 증가 (FOR UPDATE로 잠금)
        $lockMemberQuery = "SELECT id, amount FROM member WHERE id = ? FOR UPDATE";
        $lockMemberStmt = mysqli_prepare($con, $lockMemberQuery);
        mysqli_stmt_bind_param($lockMemberStmt, "i", $memberId);
        mysqli_stmt_execute($lockMemberStmt);
        $lockResult = mysqli_stmt_get_result($lockMemberStmt);
        $lockedMember = mysqli_fetch_assoc($lockResult);
        mysqli_stmt_close($lockMemberStmt);
        
        $currentBalance = $lockedMember['amount'];
        
        $balanceQuery = "UPDATE member SET amount = amount + ? WHERE id = ?";
        $balanceStmt = mysqli_prepare($con, $balanceQuery);
        mysqli_stmt_bind_param($balanceStmt, "di", $actualAmount, $memberId);
        
        if (!mysqli_stmt_execute($balanceStmt)) {
            throw new Exception('잔액 업데이트 실패: ' . mysqli_error($con));
        }
        mysqli_stmt_close($balanceStmt);
        
        $newBalance = $currentBalance + $actualAmount;
        
        // 다단계 롤링 처리
        $rollingDetails = [];
        $totalRollingAmount = 0;
        $parentsArray = json_decode($parentsJson, true);
        if (!is_array($parentsArray)) {
            $parentsArray = [];
        }
        
        if (count($parentsArray) > 0) {
            $currentMemberId = $memberId;
            $currentRate = $memberRate;
            $reversedParents = array_reverse($parentsArray);
            
            foreach ($reversedParents as $parentId) {
                // 상위 회원 정보 조회 (FOR UPDATE로 잠금)
                $parentInfoQuery = "SELECT id, rate, user_name, user_id, amount FROM member WHERE id = ? FOR UPDATE";
                $parentInfoStmt = mysqli_prepare($con, $parentInfoQuery);
                mysqli_stmt_bind_param($parentInfoStmt, "i", $parentId);
                mysqli_stmt_execute($parentInfoStmt);
                $parentInfoResult = mysqli_stmt_get_result($parentInfoStmt);
                
                if ($parentInfo = mysqli_fetch_assoc($parentInfoResult)) {
                    $parentRate = floatval($parentInfo['rate']);
                    $parentName = $parentInfo['user_name'];
                    $parentUserId = $parentInfo['user_id'];
                    
                    if ($currentRate > $parentRate) {
                        $rollingRate = $currentRate - $parentRate;
                        $rollingAmount = round($depositAmount * ($rollingRate / 100), 2);
                        
                        if ($rollingAmount > 0) {
                            // 상위 회원 잔액 증가
                            $parentBalanceQuery = "UPDATE member SET amount = amount + ? WHERE id = ?";
                            $parentBalanceStmt = mysqli_prepare($con, $parentBalanceQuery);
                            mysqli_stmt_bind_param($parentBalanceStmt, "di", $rollingAmount, $parentId);
                            
                            if (!mysqli_stmt_execute($parentBalanceStmt)) {
                                throw new Exception("상위 회원({$parentName}) 잔액 업데이트 실패");
                            }
                            mysqli_stmt_close($parentBalanceStmt);
                            
                            $rollingDetails[] = [
                                'parent_member_id' => $parentId,
                                'parent_name' => $parentName,
                                'parent_user_id' => $parentUserId,
                                'child_member_id' => $currentMemberId,
                                'rolling_rate' => $rollingRate,
                                'rolling_amount' => $rollingAmount,
                                'current_rate' => $currentRate,
                                'parent_rate' => $parentRate
                            ];
                            
                            $totalRollingAmount += $rollingAmount;
                        }
                    }
                    
                    $currentMemberId = $parentId;
                    $currentRate = $parentRate;
                }
                mysqli_stmt_close($parentInfoStmt);
            }
        }
        
        // 회사 수수료 계산
        $companyFeeAmount = $totalFeeAmount - $totalRollingAmount;
        $companyFeeRate = ($depositAmount > 0) ? ($companyFeeAmount / $depositAmount) * 100 : 0;
        
        // transaction_history 기록
        $historyQuery = "INSERT INTO transaction_history 
                        (trans_id, transfer_id, user_id, type, requested_amount, 
                        total_fee_rate, total_fee_amount, fee_rate, fee_amount, 
                        actual_amount, balance_before, balance_after, status, 
                        processed_by, created_at) 
                        VALUES (?, ?, ?, 'deposit', ?, ?, ?, ?, ?, ?, ?, ?, 'completed', ?, NOW())";
        $historyStmt = mysqli_prepare($con, $historyQuery);
        mysqli_stmt_bind_param($historyStmt, "siidddddddds", 
            $transId, $transferId, $memberId, $depositAmount, 
            $totalFeeRate, $totalFeeAmount, $companyFeeRate, $companyFeeAmount,
            $actualAmount, $currentBalance, $newBalance, $processedBy
        );
        
        if (!mysqli_stmt_execute($historyStmt)) {
            throw new Exception('트랜잭션 내역 저장 실패: ' . mysqli_error($con));
        }
        $transactionHistoryId = mysqli_insert_id($con);
        mysqli_stmt_close($historyStmt);
        
        // rolling_history 저장
        if (!empty($rollingDetails)) {
            foreach ($rollingDetails as $rollingDetail) {
                $rollingHistoryQuery = "INSERT INTO rolling_history 
                                    (parent_member_id, child_member_id, rolling_rate, rolling_amount, 
                                    original_transaction_id, created_at) 
                                    VALUES (?, ?, ?, ?, ?, NOW())";
                $rollingHistoryStmt = mysqli_prepare($con, $rollingHistoryQuery);
                mysqli_stmt_bind_param($rollingHistoryStmt, "iiddi", 
                    $rollingDetail['parent_member_id'], 
                    $rollingDetail['child_member_id'],
                    $rollingDetail['rolling_rate'], 
                    $rollingDetail['rolling_amount'], 
                    $transactionHistoryId
                );
                
                if (!mysqli_stmt_execute($rollingHistoryStmt)) {
                    throw new Exception('롤링 내역 저장 실패');
                }
                mysqli_stmt_close($rollingHistoryStmt);
            }
        }
        
        // transfer 상태 업데이트
        $updateQuery = "UPDATE transfer SET 
                    status = 'completed', 
                    trans_id = ?, 
                    updated_at = NOW() 
                    WHERE id = ?";
        $updateStmt = mysqli_prepare($con, $updateQuery);
        mysqli_stmt_bind_param($updateStmt, "si", $transId, $transferId);
        
        if (!mysqli_stmt_execute($updateStmt)) {
            throw new Exception('상태 업데이트 실패: ' . mysqli_error($con));
        }
        mysqli_stmt_close($updateStmt);
        
        // 로그 기록
        if (function_exists('writeLog')) {
            $detail = "입금 자동 승인 완료 - Trans ID: {$transId}, Transfer ID: {$transferId}, " .
                    "사용자: {$userName}({$memberUserId}), 요청액: " . number_format($depositAmount) . "원, " .
                    "수수료: " . number_format($totalFeeAmount) . "원, 실지급: " . number_format($actualAmount) . "원";
            writeLog($con, 'DEPOSIT_AUTO_APPROVED', null, ['trans_id' => $transId, 'transfer_id' => $transferId], 1, $detail, 'system');
        }
        
        // 트랜잭션 커밋
        mysqli_commit($con);
        
        // 결과 데이터 구성
        $result['success'] = true;
        $result['message'] = '입금 승인이 성공적으로 처리되었습니다.';
        $result['data'] = [
            'trans_id' => $transId,
            'transfer_id' => $transferId,
            'user_name' => $userName,
            'member_user_id' => $memberUserId,
            'requested_amount' => $depositAmount,
            'total_fee_rate' => $totalFeeRate,
            'total_fee_amount' => $totalFeeAmount,
            'company_fee_amount' => $companyFeeAmount,
            'rolling_amount' => $totalRollingAmount,
            'rolling_levels' => count($rollingDetails),
            'actual_amount' => $actualAmount,
            'old_balance' => $currentBalance,
            'new_balance' => $newBalance,
            'processed_at' => date('Y-m-d H:i:s')
        ];
        
        // 콜백 호출 (옵션)
        if ($callCallback && !empty($callbackUrl)) {
            $callbackData = [
                'trans_id' => $transId,
                'transfer_id' => $transferId,
                'user_id' => $memberUserId,
                'amount' => $depositAmount,
                'actual_amount' => $actualAmount,
                'fee_amount' => $totalFeeAmount,
                'status' => 'completed'
            ];
            
            $logInfo = [
                'trans_id' => $transId,
                'transfer_id' => $transferId,
                'member_id' => $memberId
            ];
            
            // callWebhook 함수가 있다면 호출
            if (function_exists('callWebhook')) {
                $callbackResult = callWebhook($con, $callbackUrl, 0, $callbackData, $logInfo, 'system');
                $result['data']['callback_result'] = $callbackResult;
            }
        }
        
    } catch (Exception $e) {
        // 트랜잭션 롤백
        mysqli_rollback($con);
        
        // 에러 로깅
        error_log("Process Deposit Approval Error: " . $e->getMessage());
        
        if (function_exists('writeLog')) {
            $errorData = [
                'transfer_id' => $transferId,
                'error' => $e->getMessage()
            ];
            writeLog($con, 'DEPOSIT_AUTO_APPROVAL_ERROR', null, $errorData, 0, $e->getMessage(), 'system');
        }
        
        $result['success'] = false;
        $result['message'] = $e->getMessage();
        $result['error'] = [
            'code' => 'APPROVAL_ERROR',
            'message' => $e->getMessage(),
            'timestamp' => date('Y-m-d H:i:s')
        ];
        
    } finally {
        // autocommit 복원
        mysqli_autocommit($con, TRUE);
    }
    
    return $result;
}

/**
 * 입금 취소 처리 함수
 * 
 * @param mysqli $con 데이터베이스 연결 객체
 * @param int $transferId 취소할 transfer ID
 * @return array 처리 결과 배열
 */
function cancelDepositApproval($con, $transferId) {
    // 취소 처리자도 system으로 고정
    $processedBy = 'system';
    $result = [
        'success' => false,
        'message' => '',
        'data' => null,
        'error' => null
    ];
    
    try {
        // autocommit 비활성화
        mysqli_autocommit($con, FALSE);
        
        // 트랜잭션 시작
        mysqli_begin_transaction($con, MYSQLI_TRANS_START_READ_WRITE);
        
        // transfer 정보 조회
        $checkQuery = "SELECT t.id, t.user_id, t.amount, t.status, t.trans_id,
                              m.amount as current_balance, m.user_name, m.user_id as member_user_id
                       FROM transfer t 
                       JOIN member m ON t.user_id = m.id 
                       WHERE t.id = ? AND t.type = 'deposit'
                       FOR UPDATE";
        
        $checkStmt = mysqli_prepare($con, $checkQuery);
        mysqli_stmt_bind_param($checkStmt, "i", $transferId);
        mysqli_stmt_execute($checkStmt);
        $transferResult = mysqli_stmt_get_result($checkStmt);
        
        if (!$transfer = mysqli_fetch_assoc($transferResult)) {
            throw new Exception('입금 신청을 찾을 수 없습니다.');
        }
        mysqli_stmt_close($checkStmt);
        
        $oldStatus = $transfer['status'];
        $memberId = $transfer['user_id'];
        $existingTransId = $transfer['trans_id'];
        $userName = $transfer['user_name'];
        $memberUserId = $transfer['member_user_id'];
        
        if ($oldStatus !== 'completed') {
            throw new Exception('승인된 입금만 취소할 수 있습니다.');
        }
        
        if (empty($existingTransId)) {
            throw new Exception('취소할 거래 정보를 찾을 수 없습니다.');
        }
        
        // transaction_history 조회
        $getHistoryQuery = "SELECT * FROM transaction_history WHERE trans_id = ?";
        $getHistoryStmt = mysqli_prepare($con, $getHistoryQuery);
        mysqli_stmt_bind_param($getHistoryStmt, "s", $existingTransId);
        mysqli_stmt_execute($getHistoryStmt);
        $historyResult = mysqli_stmt_get_result($getHistoryStmt);
        $historyData = mysqli_fetch_assoc($historyResult);
        mysqli_stmt_close($getHistoryStmt);
        
        if (!$historyData) {
            throw new Exception('거래 내역을 찾을 수 없습니다.');
        }
        
        $actualAmountToRevert = $historyData['actual_amount'];
        $transactionHistoryId = $historyData['id'];
        $depositAmount = $historyData['requested_amount'];
        
        // member 잔액 확인 및 차감
        $lockMemberQuery = "SELECT id, amount FROM member WHERE id = ? FOR UPDATE";
        $lockMemberStmt = mysqli_prepare($con, $lockMemberQuery);
        mysqli_stmt_bind_param($lockMemberStmt, "i", $memberId);
        mysqli_stmt_execute($lockMemberStmt);
        $lockResult = mysqli_stmt_get_result($lockMemberStmt);
        $lockedMember = mysqli_fetch_assoc($lockResult);
        mysqli_stmt_close($lockMemberStmt);
        
        $currentBalance = $lockedMember['amount'];
        
        if ($currentBalance < $actualAmountToRevert) {
            throw new Exception("취소 불가: 잔액 부족 (현재: " . number_format($currentBalance) . 
                              "원, 필요: " . number_format($actualAmountToRevert) . "원)");
        }
        
        // 롤링 내역 조회 및 복구
        $getRollingQuery = "SELECT * FROM rolling_history WHERE original_transaction_id = ?";
        $getRollingStmt = mysqli_prepare($con, $getRollingQuery);
        mysqli_stmt_bind_param($getRollingStmt, "i", $transactionHistoryId);
        mysqli_stmt_execute($getRollingStmt);
        $rollingResult = mysqli_stmt_get_result($getRollingStmt);
        
        while ($rollingRow = mysqli_fetch_assoc($rollingResult)) {
            // 상위 회원 잔액 차감
            $parentLockQuery = "SELECT amount FROM member WHERE id = ? FOR UPDATE";
            $parentLockStmt = mysqli_prepare($con, $parentLockQuery);
            mysqli_stmt_bind_param($parentLockStmt, "i", $rollingRow['parent_member_id']);
            mysqli_stmt_execute($parentLockStmt);
            mysqli_stmt_close($parentLockStmt);
            
            $parentBalanceQuery = "UPDATE member SET amount = amount - ? WHERE id = ?";
            $parentBalanceStmt = mysqli_prepare($con, $parentBalanceQuery);
            mysqli_stmt_bind_param($parentBalanceStmt, "di", $rollingRow['rolling_amount'], $rollingRow['parent_member_id']);
            
            if (!mysqli_stmt_execute($parentBalanceStmt)) {
                throw new Exception("상위 회원 잔액 차감 실패");
            }
            mysqli_stmt_close($parentBalanceStmt);
        }
        mysqli_stmt_close($getRollingStmt);
        
        // 회원 잔액 차감
        $balanceQuery = "UPDATE member SET amount = amount - ? WHERE id = ?";
        $balanceStmt = mysqli_prepare($con, $balanceQuery);
        mysqli_stmt_bind_param($balanceStmt, "di", $actualAmountToRevert, $memberId);
        
        if (!mysqli_stmt_execute($balanceStmt)) {
            throw new Exception("잔액 차감 실패");
        }
        mysqli_stmt_close($balanceStmt);
        
        $newBalance = $currentBalance - $actualAmountToRevert;
        
        // rolling_history 삭제
        $deleteRollingQuery = "DELETE FROM rolling_history WHERE original_transaction_id = ?";
        $deleteRollingStmt = mysqli_prepare($con, $deleteRollingQuery);
        mysqli_stmt_bind_param($deleteRollingStmt, "i", $transactionHistoryId);
        mysqli_stmt_execute($deleteRollingStmt);
        mysqli_stmt_close($deleteRollingStmt);
        
        // transaction_history 삭제
        $deleteHistoryQuery = "DELETE FROM transaction_history WHERE id = ?";
        $deleteHistoryStmt = mysqli_prepare($con, $deleteHistoryQuery);
        mysqli_stmt_bind_param($deleteHistoryStmt, "i", $transactionHistoryId);
        mysqli_stmt_execute($deleteHistoryStmt);
        mysqli_stmt_close($deleteHistoryStmt);
        
        // transfer 상태 업데이트
        $updateQuery = "UPDATE transfer SET 
                       status = 'cancelled', 
                       trans_id = NULL, 
                       cancelled_at = NOW(), 
                       updated_at = NOW() 
                       WHERE id = ?";
        $updateStmt = mysqli_prepare($con, $updateQuery);
        mysqli_stmt_bind_param($updateStmt, "i", $transferId);
        
        if (!mysqli_stmt_execute($updateStmt)) {
            throw new Exception('상태 업데이트 실패');
        }
        mysqli_stmt_close($updateStmt);
        
        // 로그 기록
        if (function_exists('writeLog')) {
            $detail = "입금 취소 완료 - Trans ID: {$existingTransId}, Transfer ID: {$transferId}, " .
                     "사용자: {$userName}({$memberUserId}), 취소액: " . number_format($actualAmountToRevert) . "원";
            writeLog($con, 'DEPOSIT_CANCELLED', null, ['trans_id' => $existingTransId, 'transfer_id' => $transferId], 1, $detail, 'system');
        }
        
        // 트랜잭션 커밋
        mysqli_commit($con);
        
        // 결과 구성
        $result['success'] = true;
        $result['message'] = '입금 취소가 성공적으로 처리되었습니다.';
        $result['data'] = [
            'trans_id' => $existingTransId,
            'transfer_id' => $transferId,
            'cancelled_amount' => $actualAmountToRevert,
            'old_balance' => $currentBalance,
            'new_balance' => $newBalance,
            'cancelled_at' => date('Y-m-d H:i:s')
        ];
        
    } catch (Exception $e) {
        // 트랜잭션 롤백
        mysqli_rollback($con);
        
        // 에러 로깅
        error_log("Cancel Deposit Error: " . $e->getMessage());
        
        if (function_exists('writeLog')) {
            $errorData = [
                'transfer_id' => $transferId,
                'error' => $e->getMessage()
            ];
            writeLog($con, 'DEPOSIT_CANCEL_ERROR', null, $errorData, 0, $e->getMessage(), 'system');
        }
        
        $result['success'] = false;
        $result['message'] = $e->getMessage();
        $result['error'] = [
            'code' => 'CANCEL_ERROR',
            'message' => $e->getMessage(),
            'timestamp' => date('Y-m-d H:i:s')
        ];
        
    } finally {
        // autocommit 복원
        mysqli_autocommit($con, TRUE);
    }
    
    return $result;
}

/**
 * 입금 상태 변경 함수 (requested, transferred 등)
 * 
 * @param mysqli $con 데이터베이스 연결 객체
 * @param int $transferId 변경할 transfer ID
 * @param string $newStatus 새로운 상태 값
 * @return array 처리 결과 배열
 */
function changeDepositStatus($con, $transferId, $newStatus) {
    // 상태 변경 처리자도 system으로 고정
    $processedBy = 'system';
    $result = [
        'success' => false,
        'message' => '',
        'data' => null,
        'error' => null
    ];
    
    $allowedStatuses = ['requested', 'transferred'];
    if (!in_array($newStatus, $allowedStatuses)) {
        $result['message'] = '허용되지 않은 상태값입니다.';
        return $result;
    }
    
    try {
        // 트랜잭션 시작
        mysqli_begin_transaction($con);
        
        // transfer 정보 조회
        $checkQuery = "SELECT t.*, m.user_name, m.user_id as member_user_id 
                       FROM transfer t 
                       JOIN member m ON t.user_id = m.id 
                       WHERE t.id = ? AND t.type = 'deposit'";
        
        $checkStmt = mysqli_prepare($con, $checkQuery);
        mysqli_stmt_bind_param($checkStmt, "i", $transferId);
        mysqli_stmt_execute($checkStmt);
        $transferResult = mysqli_stmt_get_result($checkStmt);
        
        if (!$transfer = mysqli_fetch_assoc($transferResult)) {
            throw new Exception('입금 신청을 찾을 수 없습니다.');
        }
        mysqli_stmt_close($checkStmt);
        
        $oldStatus = $transfer['status'];
        $userName = $transfer['user_name'];
        $memberUserId = $transfer['member_user_id'];
        $depositAmount = $transfer['amount'];
        
        if ($oldStatus === $newStatus) {
            throw new Exception('이미 해당 상태입니다.');
        }
        
        // 상태 업데이트
        $updateQuery = "UPDATE transfer SET status = ?, updated_at = NOW() WHERE id = ?";
        $updateStmt = mysqli_prepare($con, $updateQuery);
        mysqli_stmt_bind_param($updateStmt, "si", $newStatus, $transferId);
        
        if (!mysqli_stmt_execute($updateStmt)) {
            throw new Exception('상태 업데이트 실패');
        }
        mysqli_stmt_close($updateStmt);
        
        // 로그 기록
        if (function_exists('writeLog')) {
            $statusMessages = [
                'requested' => '대기',
                'transferred' => '송금완료'
            ];
            
            $statusMessage = $statusMessages[$newStatus] ?? $newStatus;
            $detail = "입금 신청 {$statusMessage} 처리 - ID: {$transferId}, 사용자: {$userName}({$memberUserId}), " .
                     "금액: " . number_format($depositAmount) . "원, 변경: {$oldStatus} → {$newStatus}";
            
            $oldData = [
                'transfer_id' => $transferId,
                'status' => $oldStatus
            ];
            
            $newData = [
                'transfer_id' => $transferId,
                'status' => $newStatus
            ];
            
            writeLog($con, 'DEPOSIT_STATUS_CHANGE', $oldData, $newData, 1, $detail, 'system');
        }
        
        // 트랜잭션 커밋
        mysqli_commit($con);
        
        // 결과 구성
        $result['success'] = true;
        $result['message'] = '상태가 성공적으로 변경되었습니다.';
        $result['data'] = [
            'transfer_id' => $transferId,
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
    } catch (Exception $e) {
        // 트랜잭션 롤백
        mysqli_rollback($con);
        
        $result['success'] = false;
        $result['message'] = $e->getMessage();
        $result['error'] = [
            'code' => 'STATUS_CHANGE_ERROR',
            'message' => $e->getMessage(),
            'timestamp' => date('Y-m-d H:i:s')
        ];
    }
    
    return $result;
}
?>