<?php
// admin/ip_blacklist.php
include 'layout/header.php';

// 경고 메시지를 출력하지 않도록 설정
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', 0);

session_start();

// 관리자 권한 확인
if (!isset($_SESSION['grade']) || $_SESSION['grade'] < 9) {
    echo "<script>alert('관리자만 접근 가능합니다.'); location.href='/';</script>";
    exit;
}
?>

<!-- 모바일 메뉴 열기 버튼 (상단에 배치 필요) -->
<button class="md:hidden fixed top-4 left-4 z-40 bg-primary text-white p-2 rounded-full shadow-lg"
    onclick="toggleSidebar()">
    ☰
</button>

<main class="flex-1 p-8 space-y-8">
    <h2 class="text-3xl font-bold text-primary mb-6">🛡️ IP 블랙리스트 관리</h2>

    <!-- 통계 카드 -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
        <div class="bg-gradient-to-r from-red-500 to-red-600 rounded-xl p-4 shadow">
            <h4 class="text-white/90 text-sm font-medium mb-2">활성 차단</h4>
            <div class="text-2xl font-bold text-white" id="activeCount">0</div>
        </div>
        <div class="bg-gradient-to-r from-orange-500 to-orange-600 rounded-xl p-4 shadow">
            <h4 class="text-white/90 text-sm font-medium mb-2">만료된 차단</h4>
            <div class="text-2xl font-bold text-white" id="expiredCount">0</div>
        </div>
        <div class="bg-gradient-to-r from-green-500 to-green-600 rounded-xl p-4 shadow">
            <h4 class="text-white/90 text-sm font-medium mb-2">해제된 차단</h4>
            <div class="text-2xl font-bold text-white" id="releasedCount">0</div>
        </div>
        <div class="bg-gradient-to-r from-purple-500 to-purple-600 rounded-xl p-4 shadow">
            <h4 class="text-white/90 text-sm font-medium mb-2">전체 기록</h4>
            <div class="text-2xl font-bold text-white" id="totalCount">0</div>
        </div>
    </div>

    <!-- IP 차단 추가 폼 -->
    <div class="bg-white/5 border border-white/10 rounded-xl overflow-hidden shadow mb-6">
        <div class="p-4">
            <h3 class="text-lg font-semibold text-white mb-4">수동 IP 차단 추가</h3>
            <form id="addBlacklistForm" class="flex flex-wrap gap-4 items-end">
                <div class="flex-1 min-w-[150px]">
                    <label class="block text-sm text-gray-400 mb-1">IP 주소</label>
                    <input type="text" name="ip_address" required pattern="^(?:[0-9]{1,3}\.){3}[0-9]{1,3}$"
                        class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
                        placeholder="예: 192.168.1.1">
                </div>
                <div class="flex-1 min-w-[200px]">
                    <label class="block text-sm text-gray-400 mb-1">차단 사유</label>
                    <input type="text" name="reason" required
                        class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
                        placeholder="차단 사유 입력">
                </div>
                <div class="flex-1 min-w-[150px]">
                    <label class="block text-sm text-gray-400 mb-1">차단 시간</label>
                    <select name="block_hours"
                        class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
                        <option value="1">1시간</option>
                        <option value="24" selected>24시간</option>
                        <option value="72">3일</option>
                        <option value="168">7일</option>
                        <option value="720">30일</option>
                        <option value="0">영구 차단</option>
                    </select>
                </div>
                <button type="submit" class="px-6 py-2 bg-red-600 text-white rounded hover:bg-red-700 transition">
                    차단 추가
                </button>
            </form>
        </div>
    </div>

    <!-- 검색 필터 -->
    <form method="GET" class="mb-6 flex gap-4 flex-wrap">
        <input type="text" name="search_ip" placeholder="IP 주소 검색"
            value="<?php echo $_GET['search_ip'] ?? ''; ?>"
            class="px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 w-64 focus:outline-none focus:ring-2 focus:ring-primary">
        
        <select name="status_filter"
            class="px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            <option value="">전체 상태</option>
            <option value="active" <?php echo ($_GET['status_filter'] ?? '') === 'active' ? 'selected' : ''; ?>>활성</option>
            <option value="expired" <?php echo ($_GET['status_filter'] ?? '') === 'expired' ? 'selected' : ''; ?>>만료</option>
            <option value="released" <?php echo ($_GET['status_filter'] ?? '') === 'released' ? 'selected' : ''; ?>>해제</option>
        </select>

        <button type="submit" class="px-4 py-2 bg-primary text-white rounded hover:bg-teal-600">
            검색
        </button>
        
        <?php if (!empty($_GET['search_ip']) || !empty($_GET['status_filter'])): ?>
            <a href="ip_blacklist.php" class="px-4 py-2 bg-orange-600 text-white rounded hover:bg-orange-700">
                초기화
            </a>
        <?php endif; ?>
    </form>

    <!-- IP 블랙리스트 테이블 -->
    <div class="bg-white/5 border border-white/10 rounded-xl overflow-hidden shadow">
        <div class="overflow-x-auto">
            <table class="min-w-full text-sm text-left">
                <thead class="bg-gray-800 text-gray-300">
                    <tr>
                        <th class="px-4 py-3 border-b border-white/10">IP 주소</th>
                        <th class="px-4 py-3 border-b border-white/10">사용자</th>
                        <th class="px-4 py-3 border-b border-white/10">차단 사유</th>
                        <th class="px-4 py-3 border-b border-white/10">차단 시작</th>
                        <th class="px-4 py-3 border-b border-white/10">차단 종료</th>
                        <th class="px-4 py-3 border-b border-white/10">실패 횟수</th>
                        <th class="px-4 py-3 border-b border-white/10">상태</th>
                        <th class="px-4 py-3 border-b border-white/10">차단자</th>
                        <th class="px-4 py-3 border-b border-white/10">관리</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-white/10" id="blacklistTableBody">
                    <!-- 동적으로 데이터가 로드됩니다 -->
                </tbody>
            </table>
        </div>
    </div>

    <!-- 페이지네이션 -->
    <div class="flex justify-center space-x-2 text-sm mt-4" id="pagination">
        <!-- 동적으로 생성됩니다 -->
    </div>

    <!-- 최근 로그인 시도 기록 -->
    <div class="mt-8">
        <h3 class="text-xl font-bold text-white mb-4">최근 로그인 시도 기록</h3>
        <div class="bg-white/5 border border-white/10 rounded-xl overflow-hidden shadow">
            <div class="overflow-x-auto">
                <table class="min-w-full text-sm text-left">
                    <thead class="bg-gray-800 text-gray-300">
                        <tr>
                            <th class="px-4 py-3 border-b border-white/10">시간</th>
                            <th class="px-4 py-3 border-b border-white/10">IP 주소</th>
                            <th class="px-4 py-3 border-b border-white/10">사용자 ID</th>
                            <th class="px-4 py-3 border-b border-white/10">결과</th>
                            <th class="px-4 py-3 border-b border-white/10">실패 사유</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-white/10" id="loginAttemptsTableBody">
                        <!-- 동적으로 데이터가 로드됩니다 -->
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</main>

<script>
    // 페이지 로드 시 데이터 가져오기
    document.addEventListener('DOMContentLoaded', function () {
        fetchBlacklist();
        fetchStatistics();
        fetchLoginAttempts();
    });

    // IP 차단 추가 폼 제출
    document.getElementById('addBlacklistForm').addEventListener('submit', function (e) {
        e.preventDefault();

        const formData = new FormData(this);

        fetch('/ajax/add_ip_blacklist.php', {
            method: 'POST',
            body: formData
        })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    alert('IP가 차단되었습니다.');
                    this.reset();
                    fetchBlacklist();
                    fetchStatistics();
                } else {
                    alert(data.message || '추가 중 오류가 발생했습니다.');
                }
            })
            .catch(err => {
                console.error(err);
                alert('오류가 발생했습니다.');
            });
    });

    // 블랙리스트 목록 가져오기
    function fetchBlacklist() {
        const urlParams = new URLSearchParams(window.location.search);
        const searchIp = urlParams.get('search_ip') || '';
        const statusFilter = urlParams.get('status_filter') || '';
        const page = urlParams.get('page') || 1;

        fetch(`/ajax/fetch_ip_blacklist.php?search_ip=${encodeURIComponent(searchIp)}&status_filter=${statusFilter}&page=${page}`)
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    displayBlacklist(data.blacklist);
                    updatePagination(data.totalPages, data.currentPage);
                }
            })
            .catch(console.error);
    }

    // 블랙리스트 목록 표시
    function displayBlacklist(blacklist) {
        const tbody = document.getElementById('blacklistTableBody');
        tbody.innerHTML = '';

        if (blacklist.length === 0) {
            tbody.innerHTML = '<tr><td colspan="9" class="px-4 py-8 text-center text-gray-400">차단된 IP가 없습니다.</td></tr>';
            return;
        }

        blacklist.forEach(item => {
            const row = document.createElement('tr');
            row.className = 'hover:bg-white/5 transition';

            // 상태별 색상 클래스
            let statusClass = '';
            let statusText = '';
            switch(item.status) {
                case 'active':
                    statusClass = 'text-red-500 font-bold';
                    statusText = '활성';
                    break;
                case 'expired':
                    statusClass = 'text-orange-500';
                    statusText = '만료';
                    break;
                case 'released':
                    statusClass = 'text-green-500';
                    statusText = '해제';
                    break;
            }

            row.innerHTML = `
                <td class="px-4 py-2 font-mono font-bold">${escapeHtml(item.ip_address)}</td>
                <td class="px-4 py-2">
                    ${item.user_name ? `${escapeHtml(item.user_name)} (${escapeHtml(item.user_id)})` : '-'}
                </td>
                <td class="px-4 py-2">${escapeHtml(item.reason)}</td>
                <td class="px-4 py-2 text-gray-400">${formatDateTime(item.blocked_at)}</td>
                <td class="px-4 py-2 text-gray-400">
                    ${item.blocked_until ? formatDateTime(item.blocked_until) : '<span class="text-red-400">영구</span>'}
                </td>
                <td class="px-4 py-2">
                    <span class="px-2 py-1 bg-gray-700 rounded text-xs">${item.failed_attempts}회</span>
                </td>
                <td class="px-4 py-2">
                    <span class="${statusClass}">${statusText}</span>
                </td>
                <td class="px-4 py-2 text-gray-400">${item.admin_id || '자동'}</td>
                <td class="px-4 py-2">
                    ${item.status === 'active' ? 
                        `<button onclick="releaseIp('${item.ip_address}')" 
                            class="px-2 py-1 text-xs bg-orange-600 hover:bg-orange-700 text-white rounded">
                            해제
                        </button>` : '-'
                    }
                </td>
            `;
            tbody.appendChild(row);
        });
    }

    // 통계 가져오기
    function fetchStatistics() {
        fetch('/ajax/fetch_ip_statistics.php')
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('activeCount').textContent = data.stats.active_count || 0;
                    document.getElementById('expiredCount').textContent = data.stats.expired_count || 0;
                    document.getElementById('releasedCount').textContent = data.stats.released_count || 0;
                    document.getElementById('totalCount').textContent = data.stats.total_count || 0;
                }
            })
            .catch(console.error);
    }

    // 최근 로그인 시도 가져오기
    function fetchLoginAttempts() {
        fetch('/ajax/fetch_login_attempts.php?limit=10')
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    displayLoginAttempts(data.attempts);
                }
            })
            .catch(console.error);
    }

    // 로그인 시도 기록 표시
    function displayLoginAttempts(attempts) {
        const tbody = document.getElementById('loginAttemptsTableBody');
        tbody.innerHTML = '';

        if (attempts.length === 0) {
            tbody.innerHTML = '<tr><td colspan="5" class="px-4 py-8 text-center text-gray-400">로그인 시도 기록이 없습니다.</td></tr>';
            return;
        }

        attempts.forEach(attempt => {
            const row = document.createElement('tr');
            row.className = 'hover:bg-white/5 transition';

            let typeClass = '';
            let typeText = '';
            switch(attempt.attempt_type) {
                case 'success':
                    typeClass = 'text-green-500';
                    typeText = '성공';
                    break;
                case 'failed':
                    typeClass = 'text-red-500';
                    typeText = '실패';
                    break;
                case 'blocked':
                    typeClass = 'text-orange-500';
                    typeText = '차단';
                    break;
            }

            row.innerHTML = `
                <td class="px-4 py-2 text-gray-400">${formatDateTime(attempt.attempted_at)}</td>
                <td class="px-4 py-2 font-mono">${escapeHtml(attempt.ip_address)}</td>
                <td class="px-4 py-2">${attempt.user_id || '-'}</td>
                <td class="px-4 py-2">
                    <span class="${typeClass} font-bold">${typeText}</span>
                </td>
                <td class="px-4 py-2 text-gray-400">${attempt.fail_reason || '-'}</td>
            `;
            tbody.appendChild(row);
        });
    }

    // IP 차단 해제
    function releaseIp(ip) {
        if (!confirm(`정말로 ${ip} 차단을 해제하시겠습니까?`)) return;

        fetch('/ajax/release_ip_blacklist.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ ip_address: ip })
        })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    alert('IP 차단이 해제되었습니다.');
                    fetchBlacklist();
                    fetchStatistics();
                } else {
                    alert(data.message || '해제 중 오류가 발생했습니다.');
                }
            })
            .catch(err => {
                console.error(err);
                alert('오류가 발생했습니다.');
            });
    }

    // 페이지네이션 업데이트
    function updatePagination(totalPages, currentPage) {
        const pagination = document.getElementById('pagination');
        pagination.innerHTML = '';

        const urlParams = new URLSearchParams(window.location.search);
        const searchIp = urlParams.get('search_ip') || '';
        const statusFilter = urlParams.get('status_filter') || '';

        // 이전 버튼
        if (currentPage > 1) {
            const prevLink = createPageLink(currentPage - 1, '이전', searchIp, statusFilter);
            prevLink.classList.add('px-3', 'py-1', 'bg-gray-800', 'hover:bg-gray-700', 'rounded');
            pagination.appendChild(prevLink);
        }

        // 페이지 번호
        for (let i = 1; i <= totalPages; i++) {
            if (i <= 3 || i > totalPages - 3 || (i >= currentPage - 1 && i <= currentPage + 1)) {
                const pageLink = createPageLink(i, i, searchIp, statusFilter);
                if (i == currentPage) {
                    pageLink.classList.add('px-3', 'py-1', 'rounded', 'bg-primary', 'text-white');
                } else {
                    pageLink.classList.add('px-3', 'py-1', 'rounded', 'bg-gray-800', 'hover:bg-gray-700');
                }
                pagination.appendChild(pageLink);
            } else if (i === 4 || i === totalPages - 3) {
                const dots = document.createElement('span');
                dots.textContent = '...';
                dots.classList.add('px-2', 'py-1', 'text-gray-500');
                pagination.appendChild(dots);
            }
        }

        // 다음 버튼
        if (currentPage < totalPages) {
            const nextLink = createPageLink(currentPage + 1, '다음', searchIp, statusFilter);
            nextLink.classList.add('px-3', 'py-1', 'bg-gray-800', 'hover:bg-gray-700', 'rounded');
            pagination.appendChild(nextLink);
        }
    }

    // 페이지 링크 생성
    function createPageLink(page, text, searchIp, statusFilter) {
        const link = document.createElement('a');
        let href = `?page=${page}`;
        if (searchIp) href += `&search_ip=${searchIp}`;
        if (statusFilter) href += `&status_filter=${statusFilter}`;
        link.href = href;
        link.textContent = text;
        return link;
    }

    // HTML 이스케이프
    function escapeHtml(text) {
        if (!text) return '';
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.toString().replace(/[&<>"']/g, m => map[m]);
    }

    // 날짜 포맷
    function formatDateTime(datetime) {
        if (!datetime) return '-';
        const date = new Date(datetime);
        return date.toLocaleString('ko-KR', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit',
            hour: '2-digit',
            minute: '2-digit'
        });
    }

    // 자동 새로고침 (30초마다)
    setInterval(function() {
        fetchLoginAttempts();
        fetchStatistics();
    }, 30000);
</script>

</body>
</html>