<?php
include 'layout/header.php';

// 경고 메시지를 출력하지 않도록 설정
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', 0);
?>

<!-- 모바일 메뉴 열기 버튼 (상단에 배치 필요) -->
<button class="md:hidden fixed top-4 left-4 z-40 bg-primary text-white p-2 rounded-full shadow-lg"
  onclick="toggleSidebar()">
  ☰
</button>

<main class="flex-1 p-8 space-y-8">
  <h2 class="text-3xl font-bold text-primary mb-6">카카오 계정 리스트</h2>

  <!-- 카카오 계정 추가 폼 (인라인) -->
  <div class="bg-white/5 border border-white/10 rounded-xl overflow-hidden shadow mb-6">
    <div class="p-4">
      <h3 class="text-lg font-semibold text-white mb-4">카카오 계정 추가</h3>
      <form id="addKakaoForm" class="space-y-4">
        <div class="grid grid-cols-4 gap-4">
          <div>
            <label class="block text-sm text-gray-400 mb-1">카카오 TID</label>
            <input type="text" name="kakao_tid" required
              class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
              placeholder="Kakao TID">
          </div>
          <div>
            <label class="block text-sm text-gray-400 mb-1">친구 링크</label>
            <input type="text" name="friend_link"
              class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
          </div>
          <div>
            <label class="block text-sm text-gray-400 mb-1">가맹점 계정명</label>
            <input type="text" name="name" required
              class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
          </div>
        </div>
        
        <!-- Default 링크 추가 -->
        <div class="mt-4">
          <label class="block text-sm text-gray-400 mb-2">기본 링크 (임의 금액용)</label>
          <input type="text" name="default_link"
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
            placeholder="임의 금액일 때 사용할 기본 카카오페이 링크">
          <p class="text-xs text-gray-500 mt-1">버튼에 없는 임의 금액일 때 사용되는 기본 링크입니다.</p>
        </div>
        
        <!-- QR Links 섹션 -->
        <div class="mt-6">
          <label class="block text-sm text-gray-400 mb-2">코드 송금링크</label>
          <textarea name="qr_link" rows="10"
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary font-mono text-sm"
            placeholder='{"default": "https://qr.kakaopay.com/...", "10000": "https://qr.kakaopay.com/...", "30000": "https://qr.kakaopay.com/...", ...}'>{
  "default": "",
  "10000": "",
  "30000": "",
  "50000": "",
  "100000": "",
  "500000": "",
  "1000000": "",
  "2000000": ""
}</textarea>
          <p class="text-xs text-gray-500 mt-1">
            • <strong>default</strong>: 임의 금액일 때 사용되는 기본 링크<br>
            • <strong>숫자 키</strong>: 각 금액별 전용 QR 링크
          </p>
        </div>
        
        <button type="submit" class="px-6 py-2 bg-primary text-white rounded hover:bg-teal-600 transition">
          추가
        </button>
      </form>
    </div>
  </div>

  <!-- 검색 필터 -->
  <form method="GET" class="mb-6 flex gap-4 flex-wrap">
    <input type="text" name="search" placeholder="TID, 계정명 검색" value="<?php echo $_GET['search'] ?? ''; ?>"
      class="px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 w-64 focus:outline-none focus:ring-2 focus:ring-primary">

    <button type="submit" class="px-4 py-2 bg-primary text-white rounded hover:bg-teal-600">
      검색
    </button>
  </form>

  <!-- 카카오 계정 리스트 테이블 -->
  <div class="bg-white/5 border border-white/10 rounded-xl overflow-hidden shadow">
    <div class="overflow-x-auto">
      <table class="min-w-full text-sm text-left">
        <thead class="bg-gray-800 text-gray-300">
          <tr>
            <th class="px-4 py-3 border-b border-white/10">ID</th>
            <th class="px-4 py-3 border-b border-white/10">카카오 TID</th>
            <th class="px-4 py-3 border-b border-white/10">가맹점 계정명</th>
            <th class="px-4 py-3 border-b border-white/10">기본 링크</th>
            <th class="px-4 py-3 border-b border-white/10">코드송금 링크</th>
            <th class="px-4 py-3 border-b border-white/10">친구 링크</th>
            <th class="px-4 py-3 border-b border-white/10">연결된 가맹점</th>
            <th class="px-4 py-3 border-b border-white/10">등록일</th>
            <th class="px-4 py-3 border-b border-white/10">관리</th>
          </tr>
        </thead>
        <tbody class="divide-y divide-white/10" id="kakaoTableBody">
          <!-- 동적으로 데이터가 로드됩니다 -->
        </tbody>
      </table>
    </div>
  </div>

  <!-- 페이지네이션 -->
  <div class="flex justify-center space-x-2 text-sm mt-4" id="pagination">
    <!-- 동적으로 생성됩니다 -->
  </div>
</main>

<script>
  // 페이지 로드 시 카카오 계정 목록 가져오기
  document.addEventListener('DOMContentLoaded', function () {
    fetchKakaoAccounts();
  });

  // 카카오 계정 추가 폼 제출
  document.getElementById('addKakaoForm').addEventListener('submit', function (e) {
    e.preventDefault();

    const formData = new FormData(this);
    
    // 기본 링크가 입력되었으면 QR 링크 JSON에 자동으로 추가
    const defaultLink = formData.get('default_link');
    const qrLinkValue = formData.get('qr_link');
    
    if (defaultLink && defaultLink.trim() !== '') {
      try {
        const parsedJson = JSON.parse(qrLinkValue);
        parsedJson.default = defaultLink.trim();
        formData.set('qr_link', JSON.stringify(parsedJson, null, 2));
      } catch (error) {
        // JSON 파싱 실패 시 새로 생성
        const newJson = { default: defaultLink.trim() };
        formData.set('qr_link', JSON.stringify(newJson, null, 2));
      }
    }
    
    // QR 링크 JSON 유효성 검증
    const finalQrLinkValue = formData.get('qr_link');
    
    if (finalQrLinkValue && finalQrLinkValue.trim() !== '') {
      try {
        const parsedJson = JSON.parse(finalQrLinkValue);
        
        // JSON이 객체인지 확인
        if (typeof parsedJson !== 'object' || parsedJson === null || Array.isArray(parsedJson)) {
          alert('QR 링크는 유효한 JSON 객체 형식이어야 합니다.');
          return;
        }
        
        // 키 유효성 검증 (default 키는 허용)
        for (const [key, val] of Object.entries(parsedJson)) {
          if (key !== 'default' && isNaN(Number(key))) {
            alert('QR 링크 JSON의 키는 "default" 또는 숫자여야 합니다.');
            return;
          }
          if (typeof val !== 'string') {
            alert('QR 링크 JSON의 값은 문자열이어야 합니다.');
            return;
          }
        }
      } catch (error) {
        alert(`QR 링크 형식에 어긋납니다 다시 입력해주세요.`);
        return;
      }
    }

    // 유효성 검증 통과 후 서버로 전송
    fetch('ajax/add_kakao_account.php', {
      method: 'POST',
      body: formData
    })
      .then(res => res.json())
      .then(data => {
        if (data.success) {
          alert('카카오 계정이 추가되었습니다.');
          this.reset();
          // 기본 QR Links 템플릿 다시 설정
          const qrTextarea = this.querySelector('textarea[name="qr_link"]');
          qrTextarea.value = `{
  "default": "",
  "10000": "",
  "30000": "",
  "50000": "",
  "100000": "",
  "500000": "",
  "1000000": "",
  "2000000": ""
}`;
          fetchKakaoAccounts();
        } else {
          alert(data.message || '추가 중 오류가 발생했습니다.');
        }
      })
      .catch(err => {
        console.error(err);
        alert('오류가 발생했습니다.');
      });
  });

  // 카카오 계정 목록 가져오기
  function fetchKakaoAccounts() {
    const urlParams = new URLSearchParams(window.location.search);
    const search = urlParams.get('search') || '';
    const page = urlParams.get('page') || 1;

    fetch(`ajax/fetch_kakao_accounts.php?search=${encodeURIComponent(search)}&page=${page}`)
      .then(res => res.json())
      .then(data => {
        if (data.success) {
          displayKakaoAccounts(data.accounts);
          updatePagination(data.totalPages, data.currentPage);
        }
      })
      .catch(console.error);
  }

  // 카카오 계정 목록 표시
  function displayKakaoAccounts(accounts) {
    const tbody = document.getElementById('kakaoTableBody');
    tbody.innerHTML = '';

    if (accounts.length === 0) {
      tbody.innerHTML = '<tr><td colspan="9" class="px-4 py-8 text-center text-gray-400">등록된 카카오 계정이 없습니다.</td></tr>';
      return;
    }

    accounts.forEach(account => {
      const row = document.createElement('tr');
      row.className = 'hover:bg-white/5 transition';
      row.id = `row-${account.id}`;

      // QR 링크에서 default 값 추출
      let defaultLinkValue = '';
      let hasDefaultLink = false;
      
      if (account.qr_link) {
        try {
          const qrData = JSON.parse(account.qr_link);
          if (qrData.default) {
            defaultLinkValue = qrData.default;
            hasDefaultLink = true;
          }
          // default 키가 없으면 추가
          if (!qrData.hasOwnProperty('default')) {
            qrData.default = '';
            account.qr_link = JSON.stringify(qrData, null, 2);
          }
        } catch (e) {
          console.log('QR 링크 파싱 오류:', e);
        }
      }

      // 읽기 모드 표시
      row.innerHTML = `
        <td class="px-4 py-2">${account.id}</td>
        <td class="px-4 py-2">
          <span class="view-mode">${escapeHtml(account.kakao_tid)}</span>
          <input type="text" class="edit-mode hidden w-full px-2 py-1 rounded bg-gray-800 text-white border border-gray-600" 
                value="${escapeHtml(account.kakao_tid)}" data-field="kakao_tid">
        </td>
        <td class="px-4 py-2">
          <span class="view-mode">${escapeHtml(account.name)}</span>
          <input type="text" class="edit-mode hidden w-full px-2 py-1 rounded bg-gray-800 text-white border border-gray-600" 
                value="${escapeHtml(account.name)}" data-field="name">
        </td>
        <td class="px-4 py-2">
          <span class="view-mode text-xs ${hasDefaultLink ? 'text-green-400' : 'text-gray-400'}">
            ${hasDefaultLink ? '설정됨' : '없음'}
          </span>
          <input type="text" class="edit-mode hidden w-full px-2 py-1 rounded bg-gray-800 text-white border border-gray-600" 
                value="${escapeHtml(defaultLinkValue)}" data-field="default_link" placeholder="기본 링크">
        </td>
        <td class="px-4 py-2">
          <span class="view-mode text-xs text-gray-400">${account.qr_link ? '설정됨' : '없음'}</span>
          <textarea class="edit-mode hidden w-full px-2 py-1 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary font-mono text-sm resize-none" 
                    rows="8"
                    data-field="qr_link" 
                    placeholder='{"default": "https://qr.kakaopay.com/...", "10000": "https://qr.kakaopay.com/...", ...}'>${escapeHtml(account.qr_link || '')}</textarea>
        </td>
        <td class="px-4 py-2">
          <span class="view-mode text-xs text-gray-400">${account.friend_link ? '설정됨' : '없음'}</span>
          <input type="text" class="edit-mode hidden w-full px-2 py-1 rounded bg-gray-800 text-white border border-gray-600" 
                value="${escapeHtml(account.friend_link || '')}" data-field="friend_link" placeholder="친구 링크">
        </td>
        <td class="px-4 py-2 text-xs">
          ${account.member_count > 0 ?
            `<span class="text-primary">${account.member_count}개</span>` :
            '<span class="text-gray-500">-</span>'
          }
        </td>
        <td class="px-4 py-2 text-gray-400">${account.created_at || '-'}</td>
        <td class="px-4 py-2">
          <div class="view-mode">
            <button onclick="enterEditMode(${account.id})" class="px-2 py-1 text-xs bg-blue-600 hover:bg-blue-700 text-white rounded mr-1">수정</button>
            <button onclick="deleteKakaoAccount(${account.id})" class="px-2 py-1 text-xs bg-red-600 hover:bg-red-700 text-white rounded">삭제</button>
          </div>
          <div class="edit-mode hidden">
            <button onclick="saveEdit(${account.id})" class="px-2 py-1 text-xs bg-green-600 hover:bg-green-700 text-white rounded mr-1">저장</button>
            <button onclick="cancelEdit(${account.id})" class="px-2 py-1 text-xs bg-gray-600 hover:bg-gray-700 text-white rounded">취소</button>
          </div>
        </td>
      `;
      tbody.appendChild(row);
    });
  }

  // QR Links 보기 모달
  function showQRLinks(id, qrLinksJson) {
    try {
      const qrLinks = JSON.parse(qrLinksJson);
      let content = '<div class="space-y-2">';
      
      // default 링크를 먼저 표시
      if (qrLinks.default) {
        content += `
          <div class="flex justify-between items-center p-2 bg-yellow-800 rounded">
            <span class="text-sm font-medium text-yellow-200">기본 링크</span>
            <a href="${qrLinks.default}" target="_blank" class="text-xs text-blue-400 hover:text-blue-300 truncate max-w-xs">${qrLinks.default}</a>
          </div>
        `;
      }
      
      // 금액별 링크 표시
      for (const [amount, link] of Object.entries(qrLinks)) {
        if (amount !== 'default' && link) {
          const formattedAmount = new Intl.NumberFormat('ko-KR').format(amount);
          content += `
            <div class="flex justify-between items-center p-2 bg-gray-800 rounded">
              <span class="text-sm font-medium">₩${formattedAmount}</span>
              <a href="${link}" target="_blank" class="text-xs text-blue-400 hover:text-blue-300 truncate max-w-xs">${link}</a>
            </div>
          `;
        }
      }
      
      content += '</div>';
      
      // 간단한 모달 표시
      const modal = document.createElement('div');
      modal.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50';
      modal.innerHTML = `
        <div class="bg-gray-900 rounded-lg p-6 max-w-2xl max-h-[80vh] overflow-auto">
          <h3 class="text-lg font-bold mb-4 text-white">QR Links - 계정 ID: ${id}</h3>
          ${content}
          <button onclick="this.closest('.fixed').remove()" class="mt-4 px-4 py-2 bg-gray-700 text-white rounded hover:bg-gray-600">닫기</button>
        </div>
      `;
      document.body.appendChild(modal);
    } catch (e) {
      console.error('QR Links 파싱 오류:', e);
    }
  }

  // 편집 모드 진입
  function enterEditMode(id) {
    const row = document.getElementById(`row-${id}`);
    row.querySelectorAll('.view-mode').forEach(el => el.classList.add('hidden'));
    row.querySelectorAll('.edit-mode').forEach(el => el.classList.remove('hidden'));
  }

  // 편집 취소
  function cancelEdit(id) {
    const row = document.getElementById(`row-${id}`);
    row.querySelectorAll('.view-mode').forEach(el => el.classList.remove('hidden'));
    row.querySelectorAll('.edit-mode').forEach(el => el.classList.add('hidden'));
  }

  // 편집 저장
  function saveEdit(id) {
    const row = document.getElementById(`row-${id}`);
    const data = { id: id };

    const allInputs = row.querySelectorAll('input[data-field], textarea[data-field]');
    
    // 데이터 수집 및 검증
    let isValid = true;
    let errorMessage = '';
    let defaultLinkValue = '';
    let qrLinkValue = '';
    
    allInputs.forEach(input => {
    if (!input.classList.contains('hidden')) {
      const field = input.dataset.field;
      const value = input.value;
      
      if (field === 'default_link') {
        defaultLinkValue = value.trim();
        data[field] = value;
      } else if (field === 'qr_link') {
        qrLinkValue = value;
        
        // qr_link 필드에 대한 JSON 유효성 검증
        if (value.trim() !== '') {
          try {
            const parsedJson = JSON.parse(value);
            
            // JSON이 객체인지 확인
            if (typeof parsedJson !== 'object' || parsedJson === null || Array.isArray(parsedJson)) {
              throw new Error('QR 링크는 유효한 JSON 객체 형식이어야 합니다.');
            }
            
            // 키 유효성 검증 (default 키는 허용)
            for (const [key, val] of Object.entries(parsedJson)) {
              if (key !== 'default' && isNaN(Number(key))) {
                throw new Error('QR 링크 JSON의 키는 "default" 또는 숫자여야 합니다.');
              }
              if (typeof val !== 'string') {
                throw new Error('QR 링크 JSON의 값은 문자열이어야 합니다.');
              }
            }
          } catch (error) {
            isValid = false;
            errorMessage = `QR 링크 형식에 어긋납니다 다시 입력해주세요.`;
            return;
          }
        }
        data[field] = value;
      } else {
        data[field] = value;
      }
    }
  });

    // 기본 링크가 입력되었으면 QR 링크 JSON에 자동으로 추가/업데이트
    if (defaultLinkValue && qrLinkValue) {
      try {
        const parsedJson = JSON.parse(qrLinkValue);
        parsedJson.default = defaultLinkValue;
        data.qr_link = JSON.stringify(parsedJson, null, 2);
      } catch (error) {
        // JSON 파싱 실패 시 새로 생성
        const newJson = { default: defaultLinkValue };
        data.qr_link = JSON.stringify(newJson, null, 2);
      }
    }

    // 유효성 검증 실패 시 저장 중단
    if (!isValid) {
      alert(errorMessage);
      return;
    }

    // 서버로 데이터 전송
    fetch('ajax/update_kakao_account.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(data)
    })
      .then(res => res.json())
      .then(responseData => {
        if (responseData.success) {
          alert('수정되었습니다.');
          fetchKakaoAccounts();
        } else {
          alert(responseData.message || '수정 중 오류가 발생했습니다.');
        }
      })
      .catch(err => {
        console.error('Fetch 에러:', err);
        alert('네트워크 오류가 발생했습니다.');
      });
  }

  // 카카오 계정 삭제
  function deleteKakaoAccount(id) {
    if (!confirm('정말로 이 카카오 계정을 삭제하시겠습니까?\n연결된 가맹점이 있을 경우 연결이 해제됩니다.')) return;

    fetch('ajax/delete_kakao_account.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ id: id })
    })
      .then(res => res.json())
      .then(data => {
        if (data.success) {
          alert('카카오 계정이 삭제되었습니다.');
          fetchKakaoAccounts();
        } else {
          alert(data.message || '삭제 중 오류가 발생했습니다.');
        }
      })
      .catch(err => {
        console.error(err);
        alert('오류가 발생했습니다.');
      });
  }

  // 페이지네이션 업데이트
  function updatePagination(totalPages, currentPage) {
    const pagination = document.getElementById('pagination');
    pagination.innerHTML = '';

    const urlParams = new URLSearchParams(window.location.search);
    const search = urlParams.get('search') || '';

    // 이전 버튼
    if (currentPage > 1) {
      const prevLink = createPageLink(currentPage - 1, '이전', search);
      prevLink.classList.add('px-3', 'py-1', 'bg-gray-800', 'hover:bg-gray-700', 'rounded');
      pagination.appendChild(prevLink);
    }

    // 페이지 번호
    for (let i = 1; i <= totalPages; i++) {
      if (i <= 3 || i > totalPages - 3 || (i >= currentPage - 1 && i <= currentPage + 1)) {
        const pageLink = createPageLink(i, i, search);
        if (i == currentPage) {
          pageLink.classList.add('px-3', 'py-1', 'rounded', 'bg-primary', 'text-white');
        } else {
          pageLink.classList.add('px-3', 'py-1', 'rounded', 'bg-gray-800', 'hover:bg-gray-700');
        }
        pagination.appendChild(pageLink);
      } else if (i === 4 || i === totalPages - 3) {
        const dots = document.createElement('span');
        dots.textContent = '...';
        dots.classList.add('px-2', 'py-1', 'text-gray-500');
        pagination.appendChild(dots);
      }
    }

    // 다음 버튼
    if (currentPage < totalPages) {
      const nextLink = createPageLink(currentPage + 1, '다음', search);
      nextLink.classList.add('px-3', 'py-1', 'bg-gray-800', 'hover:bg-gray-700', 'rounded');
      pagination.appendChild(nextLink);
    }
  }

  // 페이지 링크 생성
  function createPageLink(page, text, search) {
    const link = document.createElement('a');
    link.href = `?page=${page}&search=${search}`;
    link.textContent = text;
    return link;
  }

  // HTML 이스케이프
  function escapeHtml(text) {
    if (!text) return '';
    const map = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#039;'
    };
    return text.toString().replace(/[&<>"']/g, m => map[m]);
  }

  // 문자열 마스킹 (보안 필드용)
  function maskString(str) {
    if (!str || str.length < 8) return '********';
    return str.substring(0, 4) + '****' + str.substring(str.length - 4);
  }
</script>

</body>

</html>